/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.index.bucket;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeSet;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.common.fs.FSUtils;
import org.apache.hudi.common.model.ConsistentHashingNode;
import org.apache.hudi.common.model.HoodieConsistentHashingMetadata;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.util.FileIOUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.HoodieIndexException;
import org.apache.hudi.hadoop.fs.HadoopFSUtils;
import org.apache.hudi.index.bucket.ConsistentBucketIdentifier;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;
import org.apache.hudi.table.HoodieTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsistentBucketIndexUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ConsistentBucketIndexUtils.class);

    public static HoodieConsistentHashingMetadata loadOrCreateMetadata(HoodieTable table, String partition, int numBuckets) {
        Option<HoodieConsistentHashingMetadata> metadataOption = ConsistentBucketIndexUtils.loadMetadata(table, partition);
        if (metadataOption.isPresent()) {
            return metadataOption.get();
        }
        LOG.info("Failed to load metadata, try to create one. Partition: {}.", (Object)partition);
        HoodieConsistentHashingMetadata metadata = new HoodieConsistentHashingMetadata(partition, numBuckets);
        if (ConsistentBucketIndexUtils.saveMetadata(table, metadata, false)) {
            return metadata;
        }
        metadataOption = ConsistentBucketIndexUtils.loadMetadata(table, partition);
        ValidationUtils.checkState(metadataOption.isPresent(), "Failed to load or create metadata, partition: " + partition);
        return metadataOption.get();
    }

    public static Option<HoodieConsistentHashingMetadata> loadMetadata(HoodieTable table, String partition) {
        HoodieTableMetaClient metaClient = table.getMetaClient();
        Path metadataPath = HadoopFSUtils.constructAbsolutePathInHadoopPath(metaClient.getHashingMetadataPath(), partition);
        Path partitionPath = HadoopFSUtils.constructAbsolutePathInHadoopPath(metaClient.getBasePathV2().toString(), partition);
        try {
            FileStatus maxMetadataFile;
            Predicate<FileStatus> hashingMetaCommitFilePredicate = fileStatus -> {
                String filename = fileStatus.getPath().getName();
                return filename.contains(".commit");
            };
            Predicate<FileStatus> hashingMetadataFilePredicate = fileStatus -> {
                String filename = fileStatus.getPath().getName();
                return filename.contains(".hashing_meta");
            };
            FileStatus[] metaFiles = ((FileSystem)metaClient.getStorage().getFileSystem()).listStatus(metadataPath);
            TreeSet commitMetaTss = Arrays.stream(metaFiles).filter(hashingMetaCommitFilePredicate).map(commitFile -> HoodieConsistentHashingMetadata.getTimestampFromFile(commitFile.getPath().getName())).sorted().collect(Collectors.toCollection(TreeSet::new));
            FileStatus[] hashingMetaFiles = (FileStatus[])Arrays.stream(metaFiles).filter(hashingMetadataFilePredicate).sorted(Comparator.comparing(f -> f.getPath().getName())).toArray(FileStatus[]::new);
            String maxCommitMetaFileTs = commitMetaTss.isEmpty() ? null : (String)commitMetaTss.last();
            FileStatus fileStatus2 = maxMetadataFile = hashingMetaFiles.length > 0 ? hashingMetaFiles[hashingMetaFiles.length - 1] : null;
            if (maxMetadataFile != null && HoodieConsistentHashingMetadata.getTimestampFromFile(maxMetadataFile.getPath().getName()).equals("00000000000000")) {
                return ConsistentBucketIndexUtils.loadMetadataFromGivenFile(table, maxMetadataFile);
            }
            if (maxCommitMetaFileTs != null && maxMetadataFile != null && maxCommitMetaFileTs.equals(HoodieConsistentHashingMetadata.getTimestampFromFile(maxMetadataFile.getPath().getName()))) {
                return ConsistentBucketIndexUtils.loadMetadataFromGivenFile(table, maxMetadataFile);
            }
            HoodieTimeline completedCommits = metaClient.getActiveTimeline().getCommitAndReplaceTimeline().filterCompletedInstants();
            ArrayList fixed = new ArrayList();
            Arrays.stream(hashingMetaFiles).forEach(hashingMetaFile -> {
                boolean isRehashingCommitted;
                Path path = hashingMetaFile.getPath();
                String timestamp = HoodieConsistentHashingMetadata.getTimestampFromFile(path.getName());
                if (maxCommitMetaFileTs != null && timestamp.compareTo(maxCommitMetaFileTs) <= 0) {
                    return;
                }
                boolean bl = isRehashingCommitted = completedCommits.containsInstant(timestamp) || timestamp.equals("00000000000000");
                if (isRehashingCommitted) {
                    if (!commitMetaTss.contains(timestamp)) {
                        try {
                            ConsistentBucketIndexUtils.createCommitMarker(table, path, partitionPath);
                        }
                        catch (IOException e) {
                            throw new HoodieIOException("Exception while creating marker file " + path.getName() + " for partition " + partition, e);
                        }
                    }
                    fixed.add(hashingMetaFile);
                } else if (ConsistentBucketIndexUtils.recommitMetadataFile(table, hashingMetaFile, partition)) {
                    fixed.add(hashingMetaFile);
                }
            });
            return fixed.isEmpty() ? Option.empty() : ConsistentBucketIndexUtils.loadMetadataFromGivenFile(table, (FileStatus)fixed.get(fixed.size() - 1));
        }
        catch (FileNotFoundException e) {
            return Option.empty();
        }
        catch (IOException e) {
            LOG.error("Error when loading hashing metadata, partition: " + partition, (Throwable)e);
            throw new HoodieIndexException("Error while loading hashing metadata", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean saveMetadata(HoodieTable table, HoodieConsistentHashingMetadata metadata, boolean overwrite) {
        HoodieStorage storage = table.getStorage();
        StoragePath dir = FSUtils.constructAbsolutePath(table.getMetaClient().getHashingMetadataPath(), metadata.getPartitionPath());
        StoragePath fullPath = new StoragePath(dir, metadata.getFilename());
        try (OutputStream out = storage.create(fullPath, overwrite);){
            byte[] bytes = metadata.toBytes();
            out.write(bytes);
            out.close();
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            LOG.warn("Failed to update bucket metadata: " + metadata, (Throwable)e);
            return false;
        }
    }

    private static void createCommitMarker(HoodieTable table, Path fileStatus, Path partitionPath) throws IOException {
        StoragePath fullPath;
        HoodieStorage storage = table.getStorage();
        if (storage.exists(fullPath = new StoragePath(HadoopFSUtils.convertToStoragePath(partitionPath), HoodieConsistentHashingMetadata.getTimestampFromFile(fileStatus.getName()) + ".commit"))) {
            return;
        }
        try {
            FileIOUtils.createFileInPath(storage, fullPath, Option.of(StringUtils.getUTF8Bytes("")));
        }
        catch (HoodieIOException e) {
            if (!storage.exists(fullPath)) {
                throw e;
            }
            LOG.warn("Failed to create marker but " + fullPath + " exists", (Throwable)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static Option<HoodieConsistentHashingMetadata> loadMetadataFromGivenFile(HoodieTable table, FileStatus metaFile) {
        if (metaFile == null) {
            return Option.empty();
        }
        try (InputStream is = table.getStorage().open(HadoopFSUtils.convertToStoragePath(metaFile.getPath()));){
            byte[] content = FileIOUtils.readAsByteArray(is);
            Option<HoodieConsistentHashingMetadata> option = Option.of(HoodieConsistentHashingMetadata.fromBytes(content));
            return option;
        }
        catch (FileNotFoundException e) {
            return Option.empty();
        }
        catch (IOException e) {
            LOG.error("Error when loading hashing metadata, for path: " + metaFile.getPath().getName(), (Throwable)e);
            throw new HoodieIndexException("Error while loading hashing metadata", e);
        }
    }

    private static boolean recommitMetadataFile(HoodieTable table, FileStatus metaFile, String partition) {
        Path partitionPath = new Path(FSUtils.constructAbsolutePath(table.getMetaClient().getBasePathV2(), partition).toUri());
        String timestamp = HoodieConsistentHashingMetadata.getTimestampFromFile(metaFile.getPath().getName());
        if (table.getPendingCommitTimeline().containsInstant(timestamp)) {
            return false;
        }
        Option<HoodieConsistentHashingMetadata> hoodieConsistentHashingMetadataOption = ConsistentBucketIndexUtils.loadMetadataFromGivenFile(table, metaFile);
        if (!hoodieConsistentHashingMetadataOption.isPresent()) {
            return false;
        }
        HoodieConsistentHashingMetadata hoodieConsistentHashingMetadata = hoodieConsistentHashingMetadataOption.get();
        Predicate<String> hoodieFileGroupIdPredicate = hoodieBaseFile -> hoodieConsistentHashingMetadata.getNodes().stream().anyMatch(node -> node.getFileIdPrefix().equals(hoodieBaseFile));
        if (table.getBaseFileOnlyView().getLatestBaseFiles(partition).map(fileIdPrefix -> FSUtils.getFileIdPfxFromFileId(fileIdPrefix.getFileId())).anyMatch(hoodieFileGroupIdPredicate)) {
            try {
                ConsistentBucketIndexUtils.createCommitMarker(table, metaFile.getPath(), partitionPath);
                return true;
            }
            catch (IOException e) {
                throw new HoodieIOException("Exception while creating marker file " + metaFile.getPath().getName() + " for partition " + partition, e);
            }
        }
        return false;
    }

    public static Map<String, Map<String, Integer>> generatePartitionToFileIdPfxIdxMap(Map<String, ConsistentBucketIdentifier> partitionToIdentifier) {
        HashMap<String, Map<String, Integer>> partitionToFileIdPfxIdxMap = new HashMap<String, Map<String, Integer>>(partitionToIdentifier.size() * 2);
        int count = 0;
        for (ConsistentBucketIdentifier identifier : partitionToIdentifier.values()) {
            HashMap<String, Integer> fileIdPfxToIdx = new HashMap<String, Integer>();
            for (ConsistentHashingNode node : identifier.getNodes()) {
                fileIdPfxToIdx.put(node.getFileIdPrefix(), count++);
            }
            partitionToFileIdPfxIdxMap.put(identifier.getMetadata().getPartitionPath(), fileIdPfxToIdx);
        }
        return partitionToFileIdPfxIdxMap;
    }
}

