/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.gcp.bigquery;

import com.google.cloud.RetryOption;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.BigQueryOptions;
import com.google.cloud.bigquery.CsvOptions;
import com.google.cloud.bigquery.Dataset;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.ExternalTableDefinition;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.FormatOptions;
import com.google.cloud.bigquery.HivePartitioningOptions;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobId;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.StandardSQLTypeName;
import com.google.cloud.bigquery.StandardTableDefinition;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.cloud.bigquery.ViewDefinition;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.VisibleForTesting;
import org.apache.hudi.gcp.bigquery.BigQuerySchemaResolver;
import org.apache.hudi.gcp.bigquery.BigQuerySyncConfig;
import org.apache.hudi.gcp.bigquery.HoodieBigQuerySyncException;
import org.apache.hudi.sync.common.HoodieSyncClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HoodieBigQuerySyncClient
extends HoodieSyncClient {
    private static final Logger LOG = LoggerFactory.getLogger(HoodieBigQuerySyncClient.class);
    protected final BigQuerySyncConfig config;
    private final String projectId;
    private final String billingProjectId;
    private final String bigLakeConnectionId;
    private final String datasetName;
    private final boolean requirePartitionFilter;
    private transient BigQuery bigquery;

    public HoodieBigQuerySyncClient(BigQuerySyncConfig config, HoodieTableMetaClient metaClient) {
        this(config, HoodieBigQuerySyncClient.createBigQueryConnection(config), metaClient);
    }

    @VisibleForTesting
    HoodieBigQuerySyncClient(BigQuerySyncConfig config, BigQuery bigquery, HoodieTableMetaClient metaClient) {
        super(config, metaClient);
        this.config = config;
        this.projectId = config.getString(BigQuerySyncConfig.BIGQUERY_SYNC_PROJECT_ID);
        this.billingProjectId = config.getStringOrDefault(BigQuerySyncConfig.BIGQUERY_SYNC_BILLING_PROJECT_ID, this.projectId);
        this.datasetName = config.getString(BigQuerySyncConfig.BIGQUERY_SYNC_DATASET_NAME);
        this.requirePartitionFilter = config.getBoolean(BigQuerySyncConfig.BIGQUERY_SYNC_REQUIRE_PARTITION_FILTER);
        this.bigquery = bigquery;
        this.bigLakeConnectionId = config.getString(BigQuerySyncConfig.BIGQUERY_SYNC_BIG_LAKE_CONNECTION_ID);
    }

    private static BigQuery createBigQueryConnection(BigQuerySyncConfig config) {
        try {
            return (BigQuery)BigQueryOptions.newBuilder().setLocation(config.getString(BigQuerySyncConfig.BIGQUERY_SYNC_DATASET_LOCATION)).build().getService();
        }
        catch (BigQueryException e) {
            throw new HoodieBigQuerySyncException("Cannot create bigQuery connection ", (Throwable)((Object)e));
        }
    }

    public void createOrUpdateTableUsingBqManifestFile(String tableName, String bqManifestFileUri, String sourceUriPrefix, Schema schema) {
        try {
            String withClauses = String.format("( %s )", BigQuerySchemaResolver.schemaToSqlString(schema));
            String extraOptions = "enable_list_inference=true,";
            if (!StringUtils.isNullOrEmpty(sourceUriPrefix)) {
                withClauses = withClauses + " WITH PARTITION COLUMNS";
                extraOptions = extraOptions + String.format(" hive_partition_uri_prefix=\"%s\", require_hive_partition_filter=%s,", sourceUriPrefix, this.requirePartitionFilter);
            }
            if (!StringUtils.isNullOrEmpty(this.bigLakeConnectionId)) {
                withClauses = withClauses + String.format(" WITH CONNECTION `%s`", this.bigLakeConnectionId);
            }
            String query = String.format("CREATE OR REPLACE EXTERNAL TABLE `%s.%s.%s` %s OPTIONS (%s uris=[\"%s\"], format=\"PARQUET\", file_set_spec_type=\"NEW_LINE_DELIMITED_MANIFEST\")", this.projectId, this.datasetName, tableName, withClauses, extraOptions, bqManifestFileUri);
            QueryJobConfiguration queryConfig = QueryJobConfiguration.newBuilder(query).setUseLegacySql(false).build();
            JobId jobId = JobId.newBuilder().setProject(this.billingProjectId).setRandomJob().build();
            Job queryJob = this.bigquery.create(JobInfo.newBuilder(queryConfig).setJobId(jobId).build(), new BigQuery.JobOption[0]);
            if ((queryJob = queryJob.waitFor(new RetryOption[0])) == null) {
                LOG.error("Job for table creation no longer exists");
            } else if (queryJob.getStatus().getError() != null) {
                LOG.error("Job for table creation failed: {}", (Object)queryJob.getStatus().getError().toString());
            } else {
                LOG.info("External table created using manifest file.");
            }
        }
        catch (BigQueryException | InterruptedException e) {
            throw new HoodieBigQuerySyncException("Failed to create external table using manifest file. ", (Throwable)e);
        }
    }

    public void createManifestTable(String tableName, String sourceUri) {
        try {
            TableId tableId = TableId.of(this.projectId, this.datasetName, tableName);
            CsvOptions csvOptions = CsvOptions.newBuilder().setFieldDelimiter(",").setAllowJaggedRows(false).setAllowQuotedNewLines(false).setSkipLeadingRows(0L).build();
            Schema schema = Schema.of(Field.of("filename", StandardSQLTypeName.STRING, new Field[0]));
            ExternalTableDefinition customTable = ExternalTableDefinition.newBuilder(sourceUri, schema, (FormatOptions)csvOptions).setAutodetect(false).setIgnoreUnknownValues(false).setMaxBadRecords(0).build();
            this.bigquery.create(TableInfo.of(tableId, customTable), new BigQuery.TableOption[0]);
            LOG.info("Manifest External table created.");
        }
        catch (BigQueryException e) {
            throw new HoodieBigQuerySyncException("Manifest External table was not created ", (Throwable)((Object)e));
        }
    }

    public void updateTableSchema(String tableName, Schema schema, List<String> partitionFields) {
        boolean samePartitionFilter;
        boolean sameSchema;
        Table existingTable = this.bigquery.getTable(TableId.of(this.projectId, this.datasetName, tableName), new BigQuery.TableOption[0]);
        ExternalTableDefinition definition = (ExternalTableDefinition)existingTable.getDefinition();
        Schema remoteTableSchema = definition.getSchema();
        ArrayList<Field> finalTableFields = new ArrayList<Field>(schema.getFields());
        List bqPartitionFields = remoteTableSchema.getFields().stream().filter(field -> partitionFields.contains(field.getName())).collect(Collectors.toList());
        finalTableFields.addAll(bqPartitionFields);
        Schema finalSchema = Schema.of(finalTableFields);
        boolean bl = sameSchema = definition.getSchema() != null && definition.getSchema().equals(finalSchema);
        boolean bl2 = partitionFields.isEmpty() || this.requirePartitionFilter == (definition.getHivePartitioningOptions().getRequirePartitionFilter() != null && definition.getHivePartitioningOptions().getRequirePartitionFilter() != false) ? true : (samePartitionFilter = false);
        if (sameSchema && samePartitionFilter) {
            LOG.info("No table update is needed.");
            return;
        }
        if (!StringUtils.isNullOrEmpty(this.bigLakeConnectionId)) {
            Table updatedTable = existingTable.toBuilder().setDefinition(StandardTableDefinition.of(finalSchema)).build();
            updatedTable.update(new BigQuery.TableOption[0]);
        } else {
            ExternalTableDefinition.Builder builder = definition.toBuilder();
            builder.setSchema(finalSchema);
            builder.setAutodetect(false);
            if (definition.getHivePartitioningOptions() != null) {
                builder.setHivePartitioningOptions(definition.getHivePartitioningOptions().toBuilder().setRequirePartitionFilter(this.requirePartitionFilter).build());
            }
            Table updatedTable = existingTable.toBuilder().setDefinition(builder.build()).build();
            this.bigquery.update(updatedTable, new BigQuery.TableOption[0]);
        }
    }

    public void createVersionsTable(String tableName, String sourceUri, String sourceUriPrefix, List<String> partitionFields) {
        try {
            ExternalTableDefinition customTable;
            TableId tableId = TableId.of(this.projectId, this.datasetName, tableName);
            if (partitionFields.isEmpty()) {
                customTable = ExternalTableDefinition.newBuilder(sourceUri, FormatOptions.parquet()).setAutodetect(true).setIgnoreUnknownValues(true).setMaxBadRecords(0).build();
            } else {
                HivePartitioningOptions hivePartitioningOptions = HivePartitioningOptions.newBuilder().setMode("AUTO").setRequirePartitionFilter(false).setSourceUriPrefix(sourceUriPrefix).build();
                customTable = ExternalTableDefinition.newBuilder(sourceUri, FormatOptions.parquet()).setAutodetect(true).setHivePartitioningOptions(hivePartitioningOptions).setIgnoreUnknownValues(true).setMaxBadRecords(0).build();
            }
            this.bigquery.create(TableInfo.of(tableId, customTable), new BigQuery.TableOption[0]);
            LOG.info("External table created using hivepartitioningoptions");
        }
        catch (BigQueryException e) {
            throw new HoodieBigQuerySyncException("External table was not created ", (Throwable)((Object)e));
        }
    }

    public void createSnapshotView(String viewName, String versionsTableName, String manifestTableName) {
        try {
            TableId tableId = TableId.of(this.projectId, this.datasetName, viewName);
            String query = String.format("SELECT * FROM `%s.%s.%s` WHERE _hoodie_file_name IN (SELECT filename FROM `%s.%s.%s`)", this.projectId, this.datasetName, versionsTableName, this.projectId, this.datasetName, manifestTableName);
            ViewDefinition viewDefinition = ViewDefinition.newBuilder(query).setUseLegacySql(false).build();
            this.bigquery.create(TableInfo.of(tableId, viewDefinition), new BigQuery.TableOption[0]);
            LOG.info("View created successfully");
        }
        catch (BigQueryException e) {
            throw new HoodieBigQuerySyncException("View was not created ", (Throwable)((Object)e));
        }
    }

    @Override
    public Map<String, String> getMetastoreSchema(String tableName) {
        return Collections.emptyMap();
    }

    public boolean datasetExists() {
        Dataset dataset = this.bigquery.getDataset(DatasetId.of(this.projectId, this.datasetName), new BigQuery.DatasetOption[0]);
        return dataset != null;
    }

    @Override
    public boolean tableExists(String tableName) {
        TableId tableId = TableId.of(this.projectId, this.datasetName, tableName);
        Table table = this.bigquery.getTable(tableId, BigQuery.TableOption.fields(new BigQuery.TableField[0]));
        return table != null && table.exists();
    }

    public boolean tableNotExistsOrDoesNotMatchSpecification(String tableName) {
        boolean manifestDoesNotExist;
        TableId tableId = TableId.of(this.projectId, this.datasetName, tableName);
        Table table = this.bigquery.getTable(tableId, new BigQuery.TableOption[0]);
        if (table == null || !table.exists()) {
            return true;
        }
        ExternalTableDefinition externalTableDefinition = (ExternalTableDefinition)table.getDefinition();
        boolean bl = manifestDoesNotExist = externalTableDefinition.getSourceUris() == null || externalTableDefinition.getSourceUris().stream().noneMatch(uri -> uri.contains("absolute-path-manifest"));
        if (this.isBasePathUpdated(externalTableDefinition)) {
            return true;
        }
        if (!StringUtils.isNullOrEmpty(this.config.getString(BigQuerySyncConfig.BIGQUERY_SYNC_BIG_LAKE_CONNECTION_ID))) {
            return manifestDoesNotExist || externalTableDefinition.getConnectionId() == null;
        }
        return manifestDoesNotExist;
    }

    private boolean isBasePathUpdated(ExternalTableDefinition externalTableDefinition) {
        boolean isTableBasePathUpdated;
        String basePath = StringUtils.stripEnd(this.getBasePath(), "/");
        if (externalTableDefinition.getHivePartitioningOptions() == null) {
            List sourceUris = Option.ofNullable(externalTableDefinition.getSourceUris()).orElse(Collections.emptyList());
            String basePathWithTrailingSlash = String.format("%s/", basePath);
            boolean isTableBasePathUpdated2 = sourceUris.stream().noneMatch(sourceUri -> sourceUri.startsWith(basePathWithTrailingSlash));
            if (isTableBasePathUpdated2) {
                LOG.warn("Base path in table source uris: {}, new base path: {}", sourceUris, (Object)basePathWithTrailingSlash);
            }
            return isTableBasePathUpdated2;
        }
        String basePathInTableDefinition = externalTableDefinition.getHivePartitioningOptions().getSourceUriPrefix();
        boolean bl = isTableBasePathUpdated = !(basePathInTableDefinition = StringUtils.stripEnd(basePathInTableDefinition, "/")).equals(basePath);
        if (isTableBasePathUpdated) {
            LOG.warn("Base path in table definition: {}, new base path: {}", (Object)basePathInTableDefinition, (Object)basePath);
        }
        return isTableBasePathUpdated;
    }

    @Override
    public void close() {
        this.bigquery = null;
    }
}

