/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.log;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.AlreadyBeingCreatedException;
import org.apache.hadoop.hdfs.protocol.RecoveryInProgressException;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.table.log.AppendResult;
import org.apache.hudi.common.table.log.HoodieLogFileWriteCallback;
import org.apache.hudi.common.table.log.HoodieLogFormat;
import org.apache.hudi.common.table.log.HoodieLogFormatVersion;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.hadoop.fs.HadoopFSUtils;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StorageSchemes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HoodieLogFormatWriter
implements HoodieLogFormat.Writer {
    private static final Logger LOG = LoggerFactory.getLogger(HoodieLogFormatWriter.class);
    private HoodieLogFile logFile;
    private FSDataOutputStream output;
    private final HoodieStorage storage;
    private final long sizeThreshold;
    private final Integer bufferSize;
    private final Short replication;
    private final String rolloverLogWriteToken;
    final HoodieLogFileWriteCallback logFileWriteCallback;
    private boolean closed = false;
    private transient Thread shutdownThread = null;
    private static final String APPEND_UNAVAILABLE_EXCEPTION_MESSAGE = "not sufficiently replicated yet";

    public HoodieLogFormatWriter(HoodieStorage storage, HoodieLogFile logFile, Integer bufferSize, Short replication, Long sizeThreshold, String rolloverLogWriteToken, HoodieLogFileWriteCallback logFileWriteCallback) {
        this.storage = storage;
        this.logFile = logFile;
        this.sizeThreshold = sizeThreshold;
        this.bufferSize = bufferSize != null ? bufferSize.intValue() : storage.getDefaultBufferSize();
        this.replication = replication != null ? replication.shortValue() : storage.getDefaultReplication(logFile.getPath().getParent());
        this.rolloverLogWriteToken = rolloverLogWriteToken;
        this.logFileWriteCallback = logFileWriteCallback;
        this.addShutDownHook();
    }

    public HoodieLogFile getLogFile() {
        return this.logFile;
    }

    public long getSizeThreshold() {
        return this.sizeThreshold;
    }

    private FSDataOutputStream getOutputStream() throws IOException, InterruptedException {
        if (this.output == null) {
            Path path = new Path(this.logFile.getPath().toUri());
            FileSystem fs = (FileSystem)this.storage.getFileSystem();
            if (fs.exists(path)) {
                boolean canAppend;
                boolean isAppendSupported = StorageSchemes.isAppendSupported((String)fs.getScheme());
                boolean bl = canAppend = isAppendSupported ? this.logFileWriteCallback.preLogFileOpen(this.logFile) : false;
                if (canAppend) {
                    LOG.info(this.logFile + " exists. Appending to existing file");
                    try {
                        this.output = fs.append(path, this.bufferSize.intValue());
                    }
                    catch (RemoteException e) {
                        LOG.warn("Remote Exception, attempting to handle or recover lease", (Throwable)e);
                        this.handleAppendExceptionOrRecoverLease(path, e);
                    }
                    catch (IOException ioe) {
                        if (ioe.getMessage().toLowerCase().contains("not supported")) {
                            isAppendSupported = false;
                        }
                        this.close();
                        throw ioe;
                    }
                }
                if (!isAppendSupported || !canAppend) {
                    this.rollOver();
                    this.createNewFile();
                    String rolloverReason = isAppendSupported ? "Append not supported" : "Callback failed";
                    LOG.info(rolloverReason + ". Rolling over to " + this.logFile);
                }
            } else {
                LOG.info(this.logFile + " does not exist. Create a new file");
                this.createNewFile();
            }
        }
        return this.output;
    }

    public AppendResult appendBlock(HoodieLogBlock block) throws IOException, InterruptedException {
        return this.appendBlocks(Collections.singletonList(block));
    }

    public AppendResult appendBlocks(List<HoodieLogBlock> blocks) throws IOException, InterruptedException {
        HoodieLogFormatVersion currentLogFormatVersion = new HoodieLogFormatVersion(1);
        FSDataOutputStream originalOutputStream = this.getOutputStream();
        long startPos = originalOutputStream.getPos();
        long sizeWritten = 0L;
        FSDataOutputStream outputStream = new FSDataOutputStream((OutputStream)originalOutputStream, new FileSystem.Statistics(this.storage.getScheme()), startPos);
        for (HoodieLogBlock block : blocks) {
            long startSize = outputStream.size();
            outputStream.write(HoodieLogFormat.MAGIC);
            byte[] headerBytes = HoodieLogBlock.getLogMetadataBytes((Map)block.getLogBlockHeader());
            byte[] content = block.getContentBytes(this.storage);
            byte[] footerBytes = HoodieLogBlock.getLogMetadataBytes((Map)block.getLogBlockFooter());
            outputStream.writeLong((long)this.getLogBlockLength(content.length, headerBytes.length, footerBytes.length));
            outputStream.writeInt(currentLogFormatVersion.getVersion());
            outputStream.writeInt(block.getBlockType().ordinal());
            outputStream.write(headerBytes);
            outputStream.writeLong((long)content.length);
            outputStream.write(content);
            outputStream.write(footerBytes);
            outputStream.writeLong((long)outputStream.size() - startSize);
            if (outputStream.size() == Integer.MAX_VALUE) {
                throw new HoodieIOException("Blocks appended may overflow. Please decrease log block size or log block amount");
            }
            sizeWritten += (long)outputStream.size() - startSize;
        }
        this.flush();
        AppendResult result = new AppendResult(this.logFile, startPos, sizeWritten);
        this.rolloverIfNeeded();
        return result;
    }

    private int getLogBlockLength(int contentLength, int headerLength, int footerLength) {
        return 8 + headerLength + 8 + contentLength + footerLength + 8;
    }

    private void rolloverIfNeeded() throws IOException {
        if (this.getCurrentSize() > this.sizeThreshold) {
            LOG.info("CurrentSize {} has reached threshold {}. Rolling over to the next version", (Object)this.getCurrentSize(), (Object)this.sizeThreshold);
            this.rollOver();
        }
    }

    private void rollOver() throws IOException {
        this.closeStream();
        this.logFile = this.logFile.rollOver(this.storage, this.rolloverLogWriteToken);
        this.closed = false;
    }

    private void createNewFile() throws IOException {
        this.logFileWriteCallback.preLogFileCreate(this.logFile);
        this.output = new FSDataOutputStream(this.storage.create(this.logFile.getPath(), false, this.bufferSize, this.replication, Long.valueOf(0x20000000L)), new FileSystem.Statistics(this.storage.getScheme()));
    }

    public void close() throws IOException {
        if (null != this.shutdownThread) {
            Runtime.getRuntime().removeShutdownHook(this.shutdownThread);
        }
        this.logFileWriteCallback.preLogFileClose(this.logFile);
        try {
            this.closeStream();
        }
        finally {
            this.logFileWriteCallback.postLogFileClose(this.logFile);
        }
    }

    private void closeStream() throws IOException {
        if (this.output != null) {
            this.flush();
            this.output.close();
            this.output = null;
            this.closed = true;
        }
    }

    private void flush() throws IOException {
        if (this.output == null) {
            return;
        }
        this.output.flush();
        this.output.hsync();
    }

    public long getCurrentSize() throws IOException {
        if (this.closed) {
            throw new IllegalStateException("Cannot get current size as the underlying stream has been closed already");
        }
        if (this.output == null) {
            return 0L;
        }
        return this.output.getPos();
    }

    private void addShutDownHook() {
        this.shutdownThread = new Thread(){

            @Override
            public void run() {
                try {
                    LOG.warn("running logformatwriter hook");
                    if (HoodieLogFormatWriter.this.output != null) {
                        HoodieLogFormatWriter.this.closeStream();
                    }
                }
                catch (Exception e) {
                    LOG.warn(String.format("unable to close output stream for log file %s", HoodieLogFormatWriter.this.logFile), (Throwable)e);
                }
            }
        };
        Runtime.getRuntime().addShutdownHook(this.shutdownThread);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void handleAppendExceptionOrRecoverLease(Path path, RemoteException e) throws IOException, InterruptedException {
        FileSystem fs = (FileSystem)this.storage.getFileSystem();
        if (e.getMessage().contains(APPEND_UNAVAILABLE_EXCEPTION_MESSAGE)) {
            LOG.warn("Failed to open an append stream to the log file. Opening a new log file..", (Throwable)e);
            this.rollOver();
            this.createNewFile();
            return;
        }
        if (e.getClassName().contentEquals(AlreadyBeingCreatedException.class.getName())) {
            LOG.warn("Another task executor writing to the same log file(" + this.logFile + ". Rolling over");
            this.rollOver();
            this.createNewFile();
            return;
        }
        if (e.getClassName().contentEquals(RecoveryInProgressException.class.getName()) && fs instanceof DistributedFileSystem) {
            LOG.warn("Trying to recover log on path " + path);
            if (HadoopFSUtils.recoverDFSFileLease((DistributedFileSystem)fs, path)) {
                LOG.warn("Recovered lease on path " + path);
                this.output = fs.append(path, this.bufferSize.intValue());
                return;
            }
            String msg = "Failed to recover lease on path " + path;
            LOG.warn(msg);
            throw new HoodieException(msg, (Throwable)e);
        }
        try {
            this.closeStream();
            throw new HoodieIOException("Failed to append to the output stream ", (IOException)((Object)e));
        }
        catch (Exception ce) {
            LOG.warn("Failed to close the output stream for " + fs.getClass().getName() + " on path " + path + ". Rolling over to a new log file.");
            this.rollOver();
            this.createNewFile();
        }
    }
}

