/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.io.hfile;

import java.io.IOException;
import java.nio.ByteBuffer;

import org.apache.hadoop.hbase.classification.InterfaceAudience;

/**
 * Interface for a deserializer. Throws an IOException if the serialized data is
 * incomplete or wrong.
 * */
@InterfaceAudience.Private
public interface CacheableDeserializer<T extends Cacheable> {
  /**
   * Returns the deserialized object.
   *
   * @return T the deserialized object.
   */
  T deserialize(ByteBuffer b) throws IOException;

  /**
   * 
   * @param b
   * @param reuse true if Cacheable object can use the given buffer as its
   *          content
   * @return T the deserialized object.
   * @throws IOException
   */
  T deserialize(ByteBuffer b, boolean reuse) throws IOException;

  /**
   * Get the identifier of this deserialiser. Identifier is unique for each
   * deserializer and generated by {@link CacheableDeserializerIdManager}
   * @return identifier number of this cacheable deserializer
   */
  int getDeserialiserIdentifier();
}
