
/*
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hbase.regionserver.wal;

import java.io.IOException;
import java.util.List;

import org.apache.hadoop.hbase.Coprocessor;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.coprocessor.*;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.wal.WAL;
import org.apache.hadoop.hbase.wal.WALKey;

/**
 * Implements the coprocessor environment and runtime support for coprocessors
 * loaded within a {@link WAL}.
 */
@InterfaceAudience.Private
public class WALCoprocessorHost
    extends CoprocessorHost<WALCoprocessorHost.WALEnvironment> {
  
  /**
   * Encapsulation of the environment of each coprocessor
   */
  static class WALEnvironment extends CoprocessorHost.Environment
    implements WALCoprocessorEnvironment {

    private final WAL wal;

    final boolean useLegacyPre;
    final boolean useLegacyPost;

    @Override
    public WAL getWAL() {
      return wal;
    }

    /**
     * Constructor
     * @param implClass - not used
     * @param impl the coprocessor instance
     * @param priority chaining priority
     * @param seq load sequence
     * @param conf configuration
     * @param wal WAL
     */
    public WALEnvironment(Class<?> implClass, final Coprocessor impl,
        final int priority, final int seq, final Configuration conf,
        final WAL wal) {
      super(impl, priority, seq, conf);
      this.wal = wal;
      // Pick which version of the API we'll call.
      // This way we avoid calling the new version on older WALObservers so
      // we can maintain binary compatibility.
      // See notes in javadoc for WALObserver
      useLegacyPre = useLegacyMethod(impl.getClass(), "preWALWrite", ObserverContext.class,
          HRegionInfo.class, WALKey.class, WALEdit.class);
      useLegacyPost = useLegacyMethod(impl.getClass(), "postWALWrite", ObserverContext.class,
          HRegionInfo.class, WALKey.class, WALEdit.class);
    }
  }

  private final WAL wal;

  /**
   * Constructor
   * @param log the write ahead log
   * @param conf the configuration
   */
  public WALCoprocessorHost(final WAL log, final Configuration conf) {
    // We don't want to require an Abortable passed down through (FS)HLog, so
    // this means that a failure to load of a WAL coprocessor won't abort the
    // server. This isn't ideal, and means that security components that
    // utilize a WALObserver will have to check the observer initialization
    // state manually. However, WALObservers will eventually go away so it
    // should be an acceptable state of affairs.
    super(null);
    this.wal = log;
    // load system default cp's from configuration.
    loadSystemCoprocessors(conf, WAL_COPROCESSOR_CONF_KEY);
  }

  @Override
  public WALEnvironment createEnvironment(final Class<?> implClass,
      final Coprocessor instance, final int priority, final int seq,
      final Configuration conf) {
    return new WALEnvironment(implClass, instance, priority, seq, conf,
        this.wal);
  }

  /**
   * @param info
   * @param logKey
   * @param logEdit
   * @return true if default behavior should be bypassed, false otherwise
   * @throws IOException
   */
  public boolean preWALWrite(final HRegionInfo info, final WALKey logKey, final WALEdit logEdit)
      throws IOException {
    boolean bypass = false;
    if (this.coprocessors == null || this.coprocessors.isEmpty()) return bypass;
    ObserverContext<WALCoprocessorEnvironment> ctx = null;
    List<WALEnvironment> envs = coprocessors.get();
    for (int i = 0; i < envs.size(); i++) {
      WALEnvironment env = envs.get(i);
      if (env.getInstance() instanceof WALObserver) {
        final WALObserver observer = (WALObserver)env.getInstance();
        ctx = ObserverContext.createAndPrepare(env, ctx);
        Thread currentThread = Thread.currentThread();
        ClassLoader cl = currentThread.getContextClassLoader();
        try {
          currentThread.setContextClassLoader(env.getClassLoader());
          if (env.useLegacyPre) {
            if (logKey instanceof HLogKey) {
              observer.preWALWrite(ctx, info, (HLogKey)logKey, logEdit);
            } else {
              legacyWarning(observer.getClass(),
                  "There are wal keys present that are not HLogKey.");
            }
          } else {
            observer.preWALWrite(ctx, info, logKey, logEdit);
          }
        } catch (Throwable e) {
          handleCoprocessorThrowable(env, e);
        } finally {
          currentThread.setContextClassLoader(cl);
        }
        bypass |= ctx.shouldBypass();
        if (ctx.shouldComplete()) {
          break;
        }
      }
    }
    return bypass;
  }

  /**
   * @param info
   * @param logKey
   * @param logEdit
   * @throws IOException
   */
  public void postWALWrite(final HRegionInfo info, final WALKey logKey, final WALEdit logEdit)
      throws IOException {
    if (this.coprocessors == null || this.coprocessors.isEmpty()) return;
    ObserverContext<WALCoprocessorEnvironment> ctx = null;
    List<WALEnvironment> envs = coprocessors.get();
    for (int i = 0; i < envs.size(); i++) {
      WALEnvironment env = envs.get(i);
      if (env.getInstance() instanceof WALObserver) {
        final WALObserver observer = (WALObserver)env.getInstance();
        ctx = ObserverContext.createAndPrepare(env, ctx);
        Thread currentThread = Thread.currentThread();
        ClassLoader cl = currentThread.getContextClassLoader();
        try {
          currentThread.setContextClassLoader(env.getClassLoader());
          if (env.useLegacyPost) {
            if (logKey instanceof HLogKey) {
              observer.postWALWrite(ctx, info, (HLogKey)logKey, logEdit);
            } else {
              legacyWarning(observer.getClass(),
                  "There are wal keys present that are not HLogKey.");
            }
          } else {
            observer.postWALWrite(ctx, info, logKey, logEdit);
          }
        } catch (Throwable e) {
          handleCoprocessorThrowable(env, e);
        } finally {
          currentThread.setContextClassLoader(cl);
        }
        if (ctx.shouldComplete()) {
          break;
        }
      }
    }
  }
}
