/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.timeline;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Stream;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.HoodieTimeline;
import org.apache.hudi.common.table.timeline.HoodieDefaultTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.util.FileIOUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class HoodieActiveTimeline
extends HoodieDefaultTimeline {
    public static final SimpleDateFormat COMMIT_FORMATTER = new SimpleDateFormat("yyyyMMddHHmmss");
    public static final Set<String> VALID_EXTENSIONS_IN_ACTIVE_TIMELINE = new HashSet<String>(Arrays.asList(".commit", ".inflight", ".deltacommit", ".deltacommit.inflight", ".savepoint", ".savepoint.inflight", ".clean", ".clean.inflight", INFLIGHT_COMPACTION_EXTENSION, REQUESTED_COMPACTION_EXTENSION, ".restore.inflight", ".restore"));
    private static final transient Logger log = LogManager.getLogger(HoodieActiveTimeline.class);
    protected HoodieTableMetaClient metaClient;

    public static String createNewCommitTime() {
        return COMMIT_FORMATTER.format(new Date());
    }

    protected HoodieActiveTimeline(HoodieTableMetaClient metaClient, Set<String> includedExtensions) {
        try {
            this.setInstants(HoodieTableMetaClient.scanHoodieInstantsFromFileSystem(metaClient.getFs(), new Path(metaClient.getMetaPath()), includedExtensions));
            log.info((Object)("Loaded instants " + this.getInstants()));
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to scan metadata", e);
        }
        this.metaClient = metaClient;
        this.details = this::getInstantDetails;
    }

    public HoodieActiveTimeline(HoodieTableMetaClient metaClient) {
        this(metaClient, (Set<String>)new ImmutableSet.Builder().addAll(VALID_EXTENSIONS_IN_ACTIVE_TIMELINE).build());
    }

    public HoodieActiveTimeline() {
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
    }

    public HoodieTimeline getCommitsTimeline() {
        return this.getTimelineOfActions(Sets.newHashSet((Object[])new String[]{"commit", "deltacommit"}));
    }

    @Override
    public HoodieTimeline getCommitsAndCompactionTimeline() {
        return this.getTimelineOfActions(Sets.newHashSet((Object[])new String[]{"commit", "deltacommit", "compaction"}));
    }

    public HoodieTimeline getAllCommitsTimeline() {
        return this.getTimelineOfActions(Sets.newHashSet((Object[])new String[]{"commit", "deltacommit", "clean", "compaction", "savepoint", "rollback"}));
    }

    public HoodieTimeline getCommitTimeline() {
        return this.getTimelineOfActions(Sets.newHashSet((Object[])new String[]{"commit"}));
    }

    public HoodieTimeline getDeltaCommitTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("deltacommit"), this::getInstantDetails);
    }

    public HoodieTimeline getTimelineOfActions(Set<String> actions) {
        return new HoodieDefaultTimeline(this.getInstants().filter((? super T s) -> actions.contains(s.getAction())), this::getInstantDetails);
    }

    public HoodieTimeline getCleanerTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("clean"), this::getInstantDetails);
    }

    public HoodieTimeline getRollbackTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("rollback"), this::getInstantDetails);
    }

    public HoodieTimeline getSavePointTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("savepoint"), this::getInstantDetails);
    }

    public HoodieTimeline getRestoreTimeline() {
        return new HoodieDefaultTimeline(this.filterInstantsByAction("restore"), this::getInstantDetails);
    }

    protected Stream<HoodieInstant> filterInstantsByAction(String action) {
        return this.getInstants().filter((? super T s) -> s.getAction().equals(action));
    }

    public void createInflight(HoodieInstant instant) {
        log.info((Object)("Creating a new in-flight instant " + instant));
        this.createFileInMetaPath(instant.getFileName(), Option.empty());
    }

    public void saveAsComplete(HoodieInstant instant, Option<byte[]> data) {
        log.info((Object)("Marking instant complete " + instant));
        Preconditions.checkArgument((boolean)instant.isInflight(), (Object)("Could not mark an already completed instant as complete again " + instant));
        this.transitionState(instant, HoodieTimeline.getCompletedInstant(instant), data);
        log.info((Object)("Completed " + instant));
    }

    public void revertToInflight(HoodieInstant instant) {
        log.info((Object)("Reverting instant to inflight " + instant));
        this.revertCompleteToInflight(instant, HoodieTimeline.getInflightInstant(instant));
        log.info((Object)("Reverted " + instant + " to inflight"));
    }

    public void deleteInflight(HoodieInstant instant) {
        Preconditions.checkArgument((boolean)instant.isInflight());
        this.deleteInstantFile(instant);
    }

    public void deleteCompactionRequested(HoodieInstant instant) {
        Preconditions.checkArgument((boolean)instant.isRequested());
        Preconditions.checkArgument((instant.getAction() == "compaction" ? 1 : 0) != 0);
        this.deleteInstantFile(instant);
    }

    private void deleteInstantFile(HoodieInstant instant) {
        log.info((Object)("Deleting instant " + instant));
        Path inFlightCommitFilePath = new Path(this.metaClient.getMetaPath(), instant.getFileName());
        try {
            boolean result = this.metaClient.getFs().delete(inFlightCommitFilePath, false);
            if (!result) {
                throw new HoodieIOException("Could not delete in-flight instant " + instant);
            }
            log.info((Object)("Removed in-flight " + instant));
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not remove inflight commit " + inFlightCommitFilePath, e);
        }
    }

    @Override
    public Option<byte[]> getInstantDetails(HoodieInstant instant) {
        Path detailPath = new Path(this.metaClient.getMetaPath(), instant.getFileName());
        return this.readDataFromPath(detailPath);
    }

    public Option<byte[]> getInstantAuxiliaryDetails(HoodieInstant instant) {
        Path detailPath = new Path(this.metaClient.getMetaAuxiliaryPath(), instant.getFileName());
        return this.readDataFromPath(detailPath);
    }

    public HoodieInstant revertCompactionInflightToRequested(HoodieInstant inflightInstant) {
        Preconditions.checkArgument((boolean)inflightInstant.getAction().equals("compaction"));
        Preconditions.checkArgument((boolean)inflightInstant.isInflight());
        HoodieInstant requestedInstant = new HoodieInstant(HoodieInstant.State.REQUESTED, "compaction", inflightInstant.getTimestamp());
        this.transitionState(inflightInstant, requestedInstant, Option.empty());
        return requestedInstant;
    }

    public HoodieInstant transitionCompactionRequestedToInflight(HoodieInstant requestedInstant) {
        Preconditions.checkArgument((boolean)requestedInstant.getAction().equals("compaction"));
        Preconditions.checkArgument((boolean)requestedInstant.isRequested());
        HoodieInstant inflightInstant = new HoodieInstant(HoodieInstant.State.INFLIGHT, "compaction", requestedInstant.getTimestamp());
        this.transitionState(requestedInstant, inflightInstant, Option.empty());
        return inflightInstant;
    }

    public HoodieInstant transitionCompactionInflightToComplete(HoodieInstant inflightInstant, Option<byte[]> data) {
        Preconditions.checkArgument((boolean)inflightInstant.getAction().equals("compaction"));
        Preconditions.checkArgument((boolean)inflightInstant.isInflight());
        HoodieInstant commitInstant = new HoodieInstant(HoodieInstant.State.COMPLETED, "commit", inflightInstant.getTimestamp());
        this.transitionState(inflightInstant, commitInstant, data);
        return commitInstant;
    }

    private void createFileInAuxiliaryFolder(HoodieInstant instant, Option<byte[]> data) {
        Path fullPath = new Path(this.metaClient.getMetaAuxiliaryPath(), instant.getFileName());
        this.createFileInPath(fullPath, data);
    }

    private void transitionState(HoodieInstant fromInstant, HoodieInstant toInstant, Option<byte[]> data) {
        Preconditions.checkArgument((boolean)fromInstant.getTimestamp().equals(toInstant.getTimestamp()));
        Path commitFilePath = new Path(this.metaClient.getMetaPath(), toInstant.getFileName());
        try {
            Path inflightCommitFile = new Path(this.metaClient.getMetaPath(), fromInstant.getFileName());
            this.createFileInMetaPath(fromInstant.getFileName(), data);
            boolean success = this.metaClient.getFs().rename(inflightCommitFile, commitFilePath);
            if (!success) {
                throw new HoodieIOException("Could not rename " + inflightCommitFile + " to " + commitFilePath);
            }
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not complete " + fromInstant, e);
        }
    }

    private void revertCompleteToInflight(HoodieInstant completed, HoodieInstant inflight) {
        Preconditions.checkArgument((boolean)completed.getTimestamp().equals(inflight.getTimestamp()));
        Path inFlightCommitFilePath = new Path(this.metaClient.getMetaPath(), inflight.getFileName());
        try {
            if (!this.metaClient.getFs().exists(inFlightCommitFilePath)) {
                Path commitFilePath = new Path(this.metaClient.getMetaPath(), completed.getFileName());
                boolean success = this.metaClient.getFs().rename(commitFilePath, inFlightCommitFilePath);
                if (!success) {
                    throw new HoodieIOException("Could not rename " + commitFilePath + " to " + inFlightCommitFilePath);
                }
            }
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not complete revert " + completed, e);
        }
    }

    public void saveToInflight(HoodieInstant instant, Option<byte[]> content) {
        Preconditions.checkArgument((boolean)instant.isInflight());
        this.createFileInMetaPath(instant.getFileName(), content);
    }

    public void saveToCompactionRequested(HoodieInstant instant, Option<byte[]> content) {
        Preconditions.checkArgument((boolean)instant.getAction().equals("compaction"));
        this.createFileInAuxiliaryFolder(instant, content);
        this.createFileInMetaPath(instant.getFileName(), content);
    }

    private void createFileInMetaPath(String filename, Option<byte[]> content) {
        Path fullPath = new Path(this.metaClient.getMetaPath(), filename);
        this.createFileInPath(fullPath, content);
    }

    private void createFileInPath(Path fullPath, Option<byte[]> content) {
        try {
            if (!this.metaClient.getFs().exists(fullPath)) {
                if (this.metaClient.getFs().createNewFile(fullPath)) {
                    log.info((Object)("Created a new file in meta path: " + fullPath));
                } else {
                    throw new HoodieIOException("Failed to create file " + fullPath);
                }
            }
            if (content.isPresent()) {
                FSDataOutputStream fsout = this.metaClient.getFs().create(fullPath, true);
                fsout.write(content.get());
                fsout.close();
            }
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to create file " + fullPath, e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Option<byte[]> readDataFromPath(Path detailPath) {
        try (FSDataInputStream is = this.metaClient.getFs().open(detailPath);){
            Option<byte[]> option = Option.of(FileIOUtils.readAsByteArray((InputStream)is));
            return option;
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not read commit details from " + detailPath, e);
        }
    }

    public HoodieActiveTimeline reload() {
        return new HoodieActiveTimeline(this.metaClient);
    }
}

