/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.log;

import java.io.IOException;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.AlreadyBeingCreatedException;
import org.apache.hadoop.hdfs.protocol.RecoveryInProgressException;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hudi.common.fs.FSUtils;
import org.apache.hudi.common.fs.StorageSchemes;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.table.log.HoodieLogFormat;
import org.apache.hudi.common.table.log.HoodieLogFormatVersion;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class HoodieLogFormatWriter
implements HoodieLogFormat.Writer {
    private static final Logger LOG = LogManager.getLogger(HoodieLogFormatWriter.class);
    private HoodieLogFile logFile;
    private final FileSystem fs;
    private final long sizeThreshold;
    private final Integer bufferSize;
    private final Short replication;
    private final String logWriteToken;
    private final String rolloverLogWriteToken;
    private FSDataOutputStream output;
    private static final String APPEND_UNAVAILABLE_EXCEPTION_MESSAGE = "not sufficiently replicated yet";

    HoodieLogFormatWriter(FileSystem fs, HoodieLogFile logFile, Integer bufferSize, Short replication, Long sizeThreshold, String logWriteToken, String rolloverLogWriteToken) throws IOException, InterruptedException {
        this.fs = fs;
        this.logFile = logFile;
        this.sizeThreshold = sizeThreshold;
        this.bufferSize = bufferSize;
        this.replication = replication;
        this.logWriteToken = logWriteToken;
        this.rolloverLogWriteToken = rolloverLogWriteToken;
        this.addShutDownHook();
        Path path = logFile.getPath();
        if (fs.exists(path)) {
            boolean isAppendSupported = StorageSchemes.isAppendSupported(fs.getScheme());
            if (isAppendSupported) {
                LOG.info((Object)(logFile + " exists. Appending to existing file"));
                try {
                    this.output = fs.append(path, bufferSize.intValue());
                }
                catch (RemoteException e) {
                    LOG.warn((Object)"Remote Exception, attempting to handle or recover lease", (Throwable)e);
                    this.handleAppendExceptionOrRecoverLease(path, e);
                }
                catch (IOException ioe) {
                    if (ioe.getMessage().toLowerCase().contains("not supported")) {
                        isAppendSupported = false;
                    }
                    this.close();
                    throw ioe;
                }
            }
            if (!isAppendSupported) {
                this.logFile = logFile.rollOver(fs, rolloverLogWriteToken);
                LOG.info((Object)("Append not supported.. Rolling over to " + logFile));
                this.createNewFile();
            }
        } else {
            LOG.info((Object)(logFile + " does not exist. Create a new file"));
            this.createNewFile();
        }
    }

    public FileSystem getFs() {
        return this.fs;
    }

    @Override
    public HoodieLogFile getLogFile() {
        return this.logFile;
    }

    public long getSizeThreshold() {
        return this.sizeThreshold;
    }

    @Override
    public HoodieLogFormat.Writer appendBlock(HoodieLogBlock block) throws IOException, InterruptedException {
        HoodieLogFormatVersion currentLogFormatVersion = new HoodieLogFormatVersion(1);
        long currentSize = this.output.size();
        this.output.write(HoodieLogFormat.MAGIC);
        byte[] headerBytes = HoodieLogBlock.getLogMetadataBytes(block.getLogBlockHeader());
        byte[] content = block.getContentBytes();
        byte[] footerBytes = HoodieLogBlock.getLogMetadataBytes(block.getLogBlockFooter());
        this.output.writeLong((long)this.getLogBlockLength(content.length, headerBytes.length, footerBytes.length));
        this.output.writeInt(currentLogFormatVersion.getVersion());
        this.output.writeInt(block.getBlockType().ordinal());
        this.output.write(headerBytes);
        this.output.writeLong((long)content.length);
        this.output.write(content);
        this.output.write(footerBytes);
        this.output.writeLong((long)this.output.size() - currentSize);
        this.flush();
        return this.rolloverIfNeeded();
    }

    private int getLogBlockLength(int contentLength, int headerLength, int footerLength) {
        return 8 + headerLength + 8 + contentLength + footerLength + 8;
    }

    private HoodieLogFormat.Writer rolloverIfNeeded() throws IOException, InterruptedException {
        if (this.getCurrentSize() > this.sizeThreshold) {
            LOG.info((Object)("CurrentSize " + this.getCurrentSize() + " has reached threshold " + this.sizeThreshold + ". Rolling over to the next version"));
            HoodieLogFile newLogFile = this.logFile.rollOver(this.fs, this.rolloverLogWriteToken);
            this.close();
            return new HoodieLogFormatWriter(this.fs, newLogFile, this.bufferSize, this.replication, this.sizeThreshold, this.logWriteToken, this.rolloverLogWriteToken);
        }
        return this;
    }

    private void createNewFile() throws IOException {
        this.output = this.fs.create(this.logFile.getPath(), false, this.bufferSize.intValue(), this.replication.shortValue(), 0x20000000L, null);
    }

    @Override
    public void close() throws IOException {
        this.flush();
        this.output.close();
        this.output = null;
    }

    private void flush() throws IOException {
        if (this.output == null) {
            return;
        }
        this.output.flush();
        this.output.hsync();
    }

    @Override
    public long getCurrentSize() throws IOException {
        if (this.output == null) {
            throw new IllegalStateException("Cannot get current size as the underlying stream has been closed already");
        }
        return this.output.getPos();
    }

    private void addShutDownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                try {
                    if (HoodieLogFormatWriter.this.output != null) {
                        HoodieLogFormatWriter.this.close();
                    }
                }
                catch (Exception e) {
                    LOG.warn((Object)("unable to close output stream for log file " + HoodieLogFormatWriter.this.logFile), (Throwable)e);
                }
            }
        });
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void handleAppendExceptionOrRecoverLease(Path path, RemoteException e) throws IOException, InterruptedException {
        if (e.getMessage().contains(APPEND_UNAVAILABLE_EXCEPTION_MESSAGE)) {
            LOG.warn((Object)"Failed to open an append stream to the log file. Opening a new log file..", (Throwable)e);
            this.logFile = this.logFile.rollOver(this.fs, this.rolloverLogWriteToken);
            this.createNewFile();
            return;
        }
        if (e.getClassName().contentEquals(AlreadyBeingCreatedException.class.getName())) {
            LOG.warn((Object)("Another task executor writing to the same log file(" + this.logFile + ". Rolling over"));
            this.logFile = this.logFile.rollOver(this.fs, this.rolloverLogWriteToken);
            this.createNewFile();
            return;
        }
        if (e.getClassName().contentEquals(RecoveryInProgressException.class.getName()) && this.fs instanceof DistributedFileSystem) {
            LOG.warn((Object)("Trying to recover log on path " + path));
            if (FSUtils.recoverDFSFileLease((DistributedFileSystem)this.fs, path)) {
                LOG.warn((Object)("Recovered lease on path " + path));
                this.output = this.fs.append(path, this.bufferSize.intValue());
                return;
            }
            LOG.warn((Object)("Failed to recover lease on path " + path));
            throw new HoodieException(e);
        }
        try {
            this.close();
            throw new HoodieIOException("Failed to append to the output stream ", (IOException)((Object)e));
        }
        catch (Exception ce) {
            LOG.warn((Object)("Failed to close the output stream for " + this.fs.getClass().getName() + " on path " + path + ". Rolling over to a new log file."));
            this.logFile = this.logFile.rollOver(this.fs, this.rolloverLogWriteToken);
            this.createNewFile();
        }
    }
}

