/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.util.collection;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.Serializable;
import java.net.InetAddress;
import java.util.AbstractMap;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import org.apache.hudi.common.fs.SizeAwareDataOutputStream;
import org.apache.hudi.common.util.BufferedRandomAccessFile;
import org.apache.hudi.common.util.SerializationUtils;
import org.apache.hudi.common.util.SpillableMapUtils;
import org.apache.hudi.common.util.collection.LazyFileIterable;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.HoodieNotSupportedException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public final class DiskBasedMap<T extends Serializable, R extends Serializable>
implements Map<T, R>,
Iterable<R> {
    public static final int BUFFER_SIZE = 131072;
    private static final Logger LOG = LogManager.getLogger(DiskBasedMap.class);
    private final Map<T, ValueMetadata> valueMetadataMap;
    private File writeOnlyFile;
    private SizeAwareDataOutputStream writeOnlyFileHandle;
    private FileOutputStream fileOutputStream;
    private AtomicLong filePosition;
    private String filePath;
    private ThreadLocal<BufferedRandomAccessFile> randomAccessFile = new ThreadLocal();
    private Queue<BufferedRandomAccessFile> openedAccessFiles = new ConcurrentLinkedQueue<BufferedRandomAccessFile>();

    public DiskBasedMap(String baseFilePath) throws IOException {
        this.valueMetadataMap = new ConcurrentHashMap<T, ValueMetadata>();
        this.writeOnlyFile = new File(baseFilePath, UUID.randomUUID().toString());
        this.filePath = this.writeOnlyFile.getPath();
        this.initFile(this.writeOnlyFile);
        this.fileOutputStream = new FileOutputStream(this.writeOnlyFile, true);
        this.writeOnlyFileHandle = new SizeAwareDataOutputStream(this.fileOutputStream, 131072);
        this.filePosition = new AtomicLong(0L);
    }

    private BufferedRandomAccessFile getRandomAccessFile() {
        try {
            BufferedRandomAccessFile readHandle = this.randomAccessFile.get();
            if (readHandle == null) {
                readHandle = new BufferedRandomAccessFile(this.filePath, "r");
                readHandle.seek(0L);
                this.randomAccessFile.set(readHandle);
                this.openedAccessFiles.offer(readHandle);
            }
            return readHandle;
        }
        catch (IOException ioe) {
            throw new HoodieException(ioe);
        }
    }

    private void initFile(File writeOnlyFile) throws IOException {
        if (writeOnlyFile.exists()) {
            writeOnlyFile.delete();
        }
        if (!writeOnlyFile.getParentFile().exists()) {
            writeOnlyFile.getParentFile().mkdir();
        }
        writeOnlyFile.createNewFile();
        LOG.info((Object)("Spilling to file location " + writeOnlyFile.getAbsolutePath() + " in host (" + InetAddress.getLocalHost().getHostAddress() + ") with hostname (" + InetAddress.getLocalHost().getHostName() + ")"));
        writeOnlyFile.deleteOnExit();
        this.addShutDownHook();
    }

    private void addShutDownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                try {
                    if (DiskBasedMap.this.writeOnlyFileHandle != null) {
                        DiskBasedMap.this.writeOnlyFileHandle.flush();
                        DiskBasedMap.this.fileOutputStream.getChannel().force(false);
                        DiskBasedMap.this.writeOnlyFileHandle.close();
                    }
                    while (!DiskBasedMap.this.openedAccessFiles.isEmpty()) {
                        BufferedRandomAccessFile file = (BufferedRandomAccessFile)DiskBasedMap.this.openedAccessFiles.poll();
                        if (null == file) continue;
                        try {
                            file.close();
                        }
                        catch (IOException iOException) {}
                    }
                    DiskBasedMap.this.writeOnlyFile.delete();
                }
                catch (Exception e) {
                    DiskBasedMap.this.writeOnlyFile.delete();
                }
            }
        });
    }

    private void flushToDisk() {
        try {
            this.writeOnlyFileHandle.flush();
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to flush to DiskBasedMap file", e);
        }
    }

    @Override
    public Iterator<R> iterator() {
        return new LazyFileIterable(this.filePath, this.valueMetadataMap).iterator();
    }

    public long sizeOfFileOnDiskInBytes() {
        return this.filePosition.get();
    }

    @Override
    public int size() {
        return this.valueMetadataMap.size();
    }

    @Override
    public boolean isEmpty() {
        return this.valueMetadataMap.isEmpty();
    }

    @Override
    public boolean containsKey(Object key) {
        return this.valueMetadataMap.containsKey(key);
    }

    @Override
    public boolean containsValue(Object value) {
        throw new HoodieNotSupportedException("unable to compare values in map");
    }

    @Override
    public R get(Object key) {
        ValueMetadata entry = this.valueMetadataMap.get(key);
        if (entry == null) {
            return null;
        }
        return this.get(entry);
    }

    private R get(ValueMetadata entry) {
        return (R)((Serializable)DiskBasedMap.get(entry, this.getRandomAccessFile()));
    }

    public static <R> R get(ValueMetadata entry, RandomAccessFile file) {
        try {
            return (R)SerializationUtils.deserialize(SpillableMapUtils.readBytesFromDisk(file, entry.getOffsetOfValue(), entry.getSizeOfValue()));
        }
        catch (IOException e) {
            throw new HoodieIOException("Unable to readFromDisk Hoodie Record from disk", e);
        }
    }

    private synchronized R put(T key, R value, boolean flush) {
        try {
            byte[] val = SerializationUtils.serialize(value);
            Integer valueSize = val.length;
            Long timestamp = System.currentTimeMillis();
            this.valueMetadataMap.put(key, new ValueMetadata(this.filePath, valueSize, this.filePosition.get(), timestamp));
            byte[] serializedKey = SerializationUtils.serialize(key);
            this.filePosition.set(SpillableMapUtils.spillToDisk(this.writeOnlyFileHandle, new FileEntry(SpillableMapUtils.generateChecksum(val), serializedKey.length, valueSize, serializedKey, val, timestamp)));
            if (flush) {
                this.flushToDisk();
            }
        }
        catch (IOException io) {
            throw new HoodieIOException("Unable to store data in Disk Based map", io);
        }
        return value;
    }

    @Override
    public R put(T key, R value) {
        return this.put(key, value, true);
    }

    @Override
    public R remove(Object key) {
        Object value = this.get(key);
        this.valueMetadataMap.remove(key);
        return (R)value;
    }

    @Override
    public void putAll(Map<? extends T, ? extends R> m) {
        for (Map.Entry<T, R> entry : m.entrySet()) {
            this.put((Serializable)entry.getKey(), (Serializable)entry.getValue(), false);
        }
        this.flushToDisk();
    }

    @Override
    public void clear() {
        this.valueMetadataMap.clear();
    }

    @Override
    public Set<T> keySet() {
        return this.valueMetadataMap.keySet();
    }

    @Override
    public Collection<R> values() {
        throw new HoodieException("Unsupported Operation Exception");
    }

    public Stream<R> valueStream() {
        BufferedRandomAccessFile file = this.getRandomAccessFile();
        return ((Stream)this.valueMetadataMap.values().stream().sorted().sequential()).map(valueMetaData -> (Serializable)DiskBasedMap.get(valueMetaData, file));
    }

    @Override
    public Set<Map.Entry<T, R>> entrySet() {
        HashSet<Map.Entry<T, R>> entrySet = new HashSet<Map.Entry<T, R>>();
        for (Serializable key : this.valueMetadataMap.keySet()) {
            entrySet.add(new AbstractMap.SimpleEntry<Serializable, Object>(key, this.get(key)));
        }
        return entrySet;
    }

    public static final class ValueMetadata
    implements Comparable<ValueMetadata> {
        private String filePath;
        private Integer sizeOfValue;
        private Long offsetOfValue;
        private Long timestamp;

        protected ValueMetadata(String filePath, int sizeOfValue, long offsetOfValue, long timestamp) {
            this.filePath = filePath;
            this.sizeOfValue = sizeOfValue;
            this.offsetOfValue = offsetOfValue;
            this.timestamp = timestamp;
        }

        public String getFilePath() {
            return this.filePath;
        }

        public int getSizeOfValue() {
            return this.sizeOfValue;
        }

        public Long getOffsetOfValue() {
            return this.offsetOfValue;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        @Override
        public int compareTo(ValueMetadata o) {
            return Long.compare(this.offsetOfValue, o.offsetOfValue);
        }
    }

    public static final class FileEntry {
        private Long crc;
        private Integer sizeOfKey;
        private Integer sizeOfValue;
        private byte[] key;
        private byte[] value;
        private Long timestamp;

        public FileEntry(long crc, int sizeOfKey, int sizeOfValue, byte[] key, byte[] value, long timestamp) {
            this.crc = crc;
            this.sizeOfKey = sizeOfKey;
            this.sizeOfValue = sizeOfValue;
            this.key = key;
            this.value = value;
            this.timestamp = timestamp;
        }

        public long getCrc() {
            return this.crc;
        }

        public int getSizeOfKey() {
            return this.sizeOfKey;
        }

        public int getSizeOfValue() {
            return this.sizeOfValue;
        }

        public byte[] getKey() {
            return this.key;
        }

        public byte[] getValue() {
            return this.value;
        }

        public long getTimestamp() {
            return this.timestamp;
        }
    }
}

