/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.config;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.Instant;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.function.BiConsumer;
import javax.annotation.concurrent.Immutable;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.common.config.ConfigClassProperty;
import org.apache.hudi.common.config.ConfigGroups;
import org.apache.hudi.common.config.ConfigProperty;
import org.apache.hudi.common.config.HoodieConfig;
import org.apache.hudi.common.config.OrderedProperties;
import org.apache.hudi.common.config.TypedProperties;
import org.apache.hudi.common.model.HoodieFunctionalIndexDefinition;
import org.apache.hudi.common.util.BinaryUtil;
import org.apache.hudi.common.util.ConfigUtils;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.index.secondary.SecondaryIndexType;
import org.apache.hudi.metadata.MetadataPartitionType;

@ConfigClassProperty(name="Common Index Configs", groupName=ConfigGroups.Names.INDEXING, subGroupName=ConfigGroups.SubGroupNames.FUNCTIONAL_INDEX, areCommonConfigs=true, description="")
@Immutable
public class HoodieFunctionalIndexConfig
extends HoodieConfig {
    public static final String INDEX_DEFINITION_FILE = "index.properties";
    public static final String INDEX_DEFINITION_FILE_BACKUP = "index.properties.backup";
    public static final ConfigProperty<String> INDEX_NAME = ConfigProperty.key("hoodie.functional.index.name").noDefaultValue().sinceVersion("1.0.0").withDocumentation("Name of the functional index. This is also used for the partition name in the metadata table.");
    public static final ConfigProperty<String> INDEX_TYPE = ConfigProperty.key("hoodie.functional.index.type").defaultValue(MetadataPartitionType.COLUMN_STATS.name()).withValidValues(MetadataPartitionType.COLUMN_STATS.name(), MetadataPartitionType.BLOOM_FILTERS.name()).sinceVersion("1.0.0").withDocumentation("Type of the functional index. Default is `column_stats` if there are no functions and expressions in the command. Valid options could be BITMAP, COLUMN_STATS, LUCENE, etc. If index_type is not provided, and there are functions or expressions in the command then a functional index using column stats will be created.");
    public static final ConfigProperty<String> INDEX_FUNCTION = ConfigProperty.key("hoodie.functional.index.function").noDefaultValue().sinceVersion("1.0.0").withDocumentation("Function to be used for building the functional index.");
    public static final ConfigProperty<String> INDEX_DEFINITION_CHECKSUM = ConfigProperty.key("hoodie.table.checksum").noDefaultValue().sinceVersion("1.0.0").withDocumentation("Index definition checksum is used to guard against partial writes in HDFS. It is added as the last entry in index.properties and then used to validate while reading table config.");
    private static final String INDEX_DEFINITION_CHECKSUM_FORMAT = "%s.%s";

    public static void create(FileSystem fs, Path metadataFolder, Properties properties) throws IOException {
        if (!fs.exists(metadataFolder)) {
            fs.mkdirs(metadataFolder);
        }
        HoodieConfig hoodieConfig = new HoodieConfig(properties);
        Path propertyPath = new Path(metadataFolder, INDEX_DEFINITION_FILE);
        try (FSDataOutputStream outputStream = fs.create(propertyPath);){
            if (!hoodieConfig.contains(INDEX_NAME)) {
                throw new IllegalArgumentException(INDEX_NAME.key() + " property needs to be specified");
            }
            hoodieConfig.setDefaultValue(INDEX_TYPE);
            HoodieFunctionalIndexConfig.storeProperties(hoodieConfig.getProps(), outputStream);
        }
    }

    public static void update(FileSystem fs, Path metadataFolder, Properties updatedProps) {
        HoodieFunctionalIndexConfig.modify(fs, metadataFolder, updatedProps, ConfigUtils::upsertProperties);
    }

    public static void delete(FileSystem fs, Path metadataFolder, Set<String> deletedProps) {
        Properties props = new Properties();
        deletedProps.forEach(p -> props.setProperty((String)p, ""));
        HoodieFunctionalIndexConfig.modify(fs, metadataFolder, props, ConfigUtils::deleteProperties);
    }

    public static void recover(FileSystem fs, Path metadataFolder) throws IOException {
        Path cfgPath = new Path(metadataFolder, INDEX_DEFINITION_FILE);
        Path backupCfgPath = new Path(metadataFolder, INDEX_DEFINITION_FILE_BACKUP);
        ConfigUtils.recoverIfNeeded(fs, cfgPath, backupCfgPath);
    }

    private static void modify(FileSystem fs, Path metadataFolder, Properties modifyProps, BiConsumer<Properties, Properties> modifyFn) {
        Path cfgPath = new Path(metadataFolder, INDEX_DEFINITION_FILE);
        Path backupCfgPath = new Path(metadataFolder, INDEX_DEFINITION_FILE_BACKUP);
        try {
            String checksum;
            ConfigUtils.recoverIfNeeded(fs, cfgPath, backupCfgPath);
            TypedProperties props = ConfigUtils.fetchConfigs(fs, metadataFolder.toString(), INDEX_DEFINITION_FILE, INDEX_DEFINITION_FILE_BACKUP, 5, 1000);
            try (FSDataOutputStream out = fs.create(backupCfgPath, false);){
                HoodieFunctionalIndexConfig.storeProperties(props, out);
            }
            fs.delete(cfgPath, false);
            try (FSDataOutputStream out = fs.create(cfgPath, true);){
                modifyFn.accept(props, modifyProps);
                checksum = HoodieFunctionalIndexConfig.storeProperties(props, out);
            }
            var9_12 = null;
            try (FSDataInputStream in = fs.open(cfgPath);){
                props.clear();
                props.load((InputStream)in);
                if (!props.containsKey(INDEX_DEFINITION_CHECKSUM.key()) || !props.getProperty(INDEX_DEFINITION_CHECKSUM.key()).equals(checksum)) {
                    fs.delete(cfgPath, false);
                    throw new HoodieIOException("Checksum property missing or does not match.");
                }
            }
            catch (Throwable throwable) {
                var9_12 = throwable;
                throw throwable;
            }
            fs.delete(backupCfgPath, false);
        }
        catch (IOException e) {
            throw new HoodieIOException("Error updating table configs.", e);
        }
    }

    private static Properties getOrderedPropertiesWithTableChecksum(Properties props) {
        OrderedProperties orderedProps = new OrderedProperties(props);
        ((Properties)orderedProps).put(INDEX_DEFINITION_CHECKSUM.key(), String.valueOf(HoodieFunctionalIndexConfig.generateChecksum(props)));
        return orderedProps;
    }

    private static String storeProperties(Properties props, FSDataOutputStream outputStream) throws IOException {
        String checksum;
        if (HoodieFunctionalIndexConfig.isValidChecksum(props)) {
            checksum = props.getProperty(INDEX_DEFINITION_CHECKSUM.key());
            props.store((OutputStream)outputStream, "Updated at " + Instant.now());
        } else {
            Properties propsWithChecksum = HoodieFunctionalIndexConfig.getOrderedPropertiesWithTableChecksum(props);
            propsWithChecksum.store((OutputStream)outputStream, "Properties saved on " + Instant.now());
            checksum = propsWithChecksum.getProperty(INDEX_DEFINITION_CHECKSUM.key());
            props.setProperty(INDEX_DEFINITION_CHECKSUM.key(), checksum);
        }
        return checksum;
    }

    private static boolean isValidChecksum(Properties props) {
        return props.containsKey(INDEX_DEFINITION_CHECKSUM.key()) && HoodieFunctionalIndexConfig.validateChecksum(props);
    }

    public static long generateChecksum(Properties props) {
        if (!props.containsKey(INDEX_NAME.key())) {
            throw new IllegalArgumentException(INDEX_NAME.key() + " property needs to be specified");
        }
        String table = props.getProperty(INDEX_NAME.key());
        String database = props.getProperty(INDEX_TYPE.key(), "");
        return BinaryUtil.generateChecksum(StringUtils.getUTF8Bytes(String.format(INDEX_DEFINITION_CHECKSUM_FORMAT, database, table)));
    }

    public static boolean validateChecksum(Properties props) {
        return Long.parseLong(props.getProperty(INDEX_DEFINITION_CHECKSUM.key())) == HoodieFunctionalIndexConfig.generateChecksum(props);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static HoodieFunctionalIndexConfig copy(HoodieFunctionalIndexConfig functionalIndexConfig) {
        return HoodieFunctionalIndexConfig.newBuilder().fromIndexConfig(functionalIndexConfig).build();
    }

    public static HoodieFunctionalIndexConfig merge(HoodieFunctionalIndexConfig functionalIndexConfig1, HoodieFunctionalIndexConfig functionalIndexConfig2) {
        return HoodieFunctionalIndexConfig.newBuilder().fromIndexConfig(functionalIndexConfig1).fromIndexConfig(functionalIndexConfig2).build();
    }

    public static HoodieFunctionalIndexConfig fromIndexDefinition(HoodieFunctionalIndexDefinition indexDefinition) {
        return HoodieFunctionalIndexConfig.newBuilder().withIndexName(indexDefinition.getIndexName()).withIndexType(indexDefinition.getIndexType()).withIndexFunction(indexDefinition.getIndexFunction()).build();
    }

    public String getIndexName() {
        return this.getString(INDEX_NAME);
    }

    public String getIndexType() {
        return this.getString(INDEX_TYPE);
    }

    public String getIndexFunction() {
        return this.getString(INDEX_FUNCTION);
    }

    public boolean isIndexUsingBloomFilter() {
        return this.getIndexType().equalsIgnoreCase(MetadataPartitionType.BLOOM_FILTERS.name());
    }

    public boolean isIndexUsingLucene() {
        return this.getIndexType().equalsIgnoreCase(SecondaryIndexType.LUCENE.name());
    }

    public boolean isIndexUsingColumnStats() {
        return this.getIndexType().equalsIgnoreCase(MetadataPartitionType.COLUMN_STATS.name());
    }

    public boolean isIndexUsingRecordIndex() {
        return this.getIndexType().equalsIgnoreCase(MetadataPartitionType.RECORD_INDEX.name());
    }

    public static class Builder {
        private final HoodieFunctionalIndexConfig functionalIndexConfig = new HoodieFunctionalIndexConfig();

        public Builder fromFile(File propertiesFile) throws IOException {
            try (FileReader reader = new FileReader(propertiesFile);){
                this.functionalIndexConfig.getProps().load(reader);
                Builder builder = this;
                return builder;
            }
        }

        public Builder fromProperties(Properties props) {
            this.functionalIndexConfig.getProps().putAll((Map<?, ?>)props);
            return this;
        }

        public Builder fromIndexConfig(HoodieFunctionalIndexConfig functionalIndexConfig) {
            this.functionalIndexConfig.getProps().putAll((Map<?, ?>)functionalIndexConfig.getProps());
            return this;
        }

        public Builder withIndexName(String indexName) {
            this.functionalIndexConfig.setValue(INDEX_NAME, indexName);
            return this;
        }

        public Builder withIndexType(String indexType) {
            this.functionalIndexConfig.setValue(INDEX_TYPE, indexType);
            return this;
        }

        public Builder withIndexFunction(String indexFunction) {
            this.functionalIndexConfig.setValue(INDEX_FUNCTION, indexFunction);
            return this;
        }

        public HoodieFunctionalIndexConfig build() {
            this.functionalIndexConfig.setDefaults(HoodieFunctionalIndexConfig.class.getName());
            return this.functionalIndexConfig;
        }
    }
}

