/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.timeline;

import java.io.Serializable;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.ValidationUtils;

public class HoodieInstant
implements Serializable,
Comparable<HoodieInstant> {
    private static final Pattern NAME_FORMAT = Pattern.compile("^(\\d+(_\\d+)?)(\\.\\w+)(\\.\\D+)?$");
    private static final String DELIMITER = ".";
    private static final String UNDERSCORE = "_";
    private static final String FILE_NAME_FORMAT_ERROR = "The provided file name %s does not conform to the required format";
    private static final Map<String, String> COMPARABLE_ACTIONS = HoodieInstant.createComparableActionsMap();
    public static final Comparator<HoodieInstant> ACTION_COMPARATOR = Comparator.comparing(instant -> HoodieInstant.getComparableAction(instant.getAction()));
    public static final Comparator<HoodieInstant> COMPARATOR = Comparator.comparing(HoodieInstant::getTimestamp).thenComparing(ACTION_COMPARATOR).thenComparing(HoodieInstant::getState);
    public static final Comparator<HoodieInstant> STATE_TRANSITION_COMPARATOR = Comparator.comparing(HoodieInstant::getCompletionTime).thenComparing(COMPARATOR);
    private static final String EMPTY_FILE_EXTENSION = "";
    private final State state;
    private final String action;
    private final String timestamp;
    private final String completionTime;

    public static String getComparableAction(String action) {
        return COMPARABLE_ACTIONS.getOrDefault(action, action);
    }

    public static String extractTimestamp(String fileName) throws IllegalArgumentException {
        Matcher matcher = NAME_FORMAT.matcher(fileName);
        if (matcher.find()) {
            String timestamp = matcher.group(1);
            return timestamp.contains(UNDERSCORE) ? timestamp.split(UNDERSCORE)[0] : timestamp;
        }
        throw new IllegalArgumentException("Failed to retrieve timestamp from name: " + String.format(FILE_NAME_FORMAT_ERROR, fileName));
    }

    public static String getTimelineFileExtension(String fileName) {
        Objects.requireNonNull(fileName);
        Matcher matcher = NAME_FORMAT.matcher(fileName);
        if (matcher.find()) {
            return fileName.substring(matcher.group(1).length());
        }
        return EMPTY_FILE_EXTENSION;
    }

    public HoodieInstant(FileStatus fileStatus) {
        String[] timestamps;
        String fileName = fileStatus.getPath().getName();
        Matcher matcher = NAME_FORMAT.matcher(fileName);
        if (matcher.find()) {
            timestamps = matcher.group(1).split(UNDERSCORE);
            this.timestamp = timestamps[0];
            if (matcher.group(3).equals(".inflight")) {
                this.action = "commit";
                this.state = State.INFLIGHT;
            } else {
                this.action = matcher.group(3).replaceFirst(DELIMITER, EMPTY_FILE_EXTENSION);
                this.state = matcher.groupCount() == 4 && matcher.group(4) != null ? State.valueOf(matcher.group(4).replaceFirst(DELIMITER, EMPTY_FILE_EXTENSION).toUpperCase()) : State.COMPLETED;
            }
        } else {
            throw new IllegalArgumentException("Failed to construct HoodieInstant: " + String.format(FILE_NAME_FORMAT_ERROR, fileName));
        }
        this.completionTime = timestamps.length > 1 ? timestamps[1] : null;
    }

    public HoodieInstant(boolean isInflight, String action, String timestamp) {
        this.state = isInflight ? State.INFLIGHT : State.COMPLETED;
        this.action = action;
        this.timestamp = timestamp;
        this.completionTime = null;
    }

    public HoodieInstant(State state, String action, String timestamp) {
        this.state = state;
        this.action = action;
        this.timestamp = timestamp;
        this.completionTime = null;
    }

    public HoodieInstant(State state, String action, String timestamp, String completionTime) {
        this.state = state;
        this.action = action;
        this.timestamp = timestamp;
        this.completionTime = completionTime;
    }

    public boolean isCompleted() {
        return this.state == State.COMPLETED;
    }

    public boolean isInflight() {
        return this.state == State.INFLIGHT;
    }

    public boolean isRequested() {
        return this.state == State.REQUESTED;
    }

    public String getAction() {
        return this.action;
    }

    public String getTimestamp() {
        return this.timestamp;
    }

    private static Map<String, String> createComparableActionsMap() {
        HashMap<String, String> comparableMap = new HashMap<String, String>();
        comparableMap.put("compaction", "commit");
        comparableMap.put("logcompaction", "deltacommit");
        return comparableMap;
    }

    private String getPendingFileName() {
        if ("commit".equals(this.action)) {
            if (this.isInflight()) {
                return HoodieTimeline.makeInflightCommitFileName(this.timestamp);
            }
            if (this.isRequested()) {
                return HoodieTimeline.makeRequestedCommitFileName(this.timestamp);
            }
        } else if ("clean".equals(this.action)) {
            if (this.isInflight()) {
                return HoodieTimeline.makeInflightCleanerFileName(this.timestamp);
            }
            if (this.isRequested()) {
                return HoodieTimeline.makeRequestedCleanerFileName(this.timestamp);
            }
        } else if ("rollback".equals(this.action)) {
            if (this.isInflight()) {
                return HoodieTimeline.makeInflightRollbackFileName(this.timestamp);
            }
            if (this.isRequested()) {
                return HoodieTimeline.makeRequestedRollbackFileName(this.timestamp);
            }
        } else {
            if ("savepoint".equals(this.action)) {
                return HoodieTimeline.makeInflightSavePointFileName(this.timestamp);
            }
            if ("deltacommit".equals(this.action)) {
                if (this.isInflight()) {
                    return HoodieTimeline.makeInflightDeltaFileName(this.timestamp);
                }
                if (this.isRequested()) {
                    return HoodieTimeline.makeRequestedDeltaFileName(this.timestamp);
                }
            } else if ("compaction".equals(this.action)) {
                if (this.isInflight()) {
                    return HoodieTimeline.makeInflightCompactionFileName(this.timestamp);
                }
                if (this.isRequested()) {
                    return HoodieTimeline.makeRequestedCompactionFileName(this.timestamp);
                }
            } else if ("logcompaction".equals(this.action)) {
                if (this.isInflight()) {
                    return HoodieTimeline.makeInflightLogCompactionFileName(this.timestamp);
                }
                if (this.isRequested()) {
                    return HoodieTimeline.makeRequestedLogCompactionFileName(this.timestamp);
                }
            } else if ("restore".equals(this.action)) {
                if (this.isInflight()) {
                    return HoodieTimeline.makeInflightRestoreFileName(this.timestamp);
                }
                if (this.isRequested()) {
                    return HoodieTimeline.makeRequestedRestoreFileName(this.timestamp);
                }
            } else if ("replacecommit".equals(this.action)) {
                if (this.isInflight()) {
                    return HoodieTimeline.makeInflightReplaceFileName(this.timestamp);
                }
                if (this.isRequested()) {
                    return HoodieTimeline.makeRequestedReplaceFileName(this.timestamp);
                }
            } else if ("indexing".equals(this.action)) {
                if (this.isInflight()) {
                    return HoodieTimeline.makeInflightIndexFileName(this.timestamp);
                }
                if (this.isRequested()) {
                    return HoodieTimeline.makeRequestedIndexFileName(this.timestamp);
                }
            } else if ("schemacommit".equals(this.action)) {
                if (this.isInflight()) {
                    return HoodieTimeline.makeInflightSchemaFileName(this.timestamp);
                }
                if (this.isRequested()) {
                    return HoodieTimeline.makeRequestSchemaFileName(this.timestamp);
                }
            }
        }
        throw new IllegalArgumentException("Cannot get file name for unknown action " + this.action);
    }

    private String getCompleteFileName(String completionTime) {
        ValidationUtils.checkArgument(!StringUtils.isNullOrEmpty(completionTime), "Completion time should not be empty");
        String timestampWithCompletionTime = this.timestamp + UNDERSCORE + completionTime;
        switch (this.action) {
            case "commit": 
            case "compaction": {
                return HoodieTimeline.makeCommitFileName(timestampWithCompletionTime);
            }
            case "clean": {
                return HoodieTimeline.makeCleanerFileName(timestampWithCompletionTime);
            }
            case "rollback": {
                return HoodieTimeline.makeRollbackFileName(timestampWithCompletionTime);
            }
            case "savepoint": {
                return HoodieTimeline.makeSavePointFileName(timestampWithCompletionTime);
            }
            case "deltacommit": 
            case "logcompaction": {
                return HoodieTimeline.makeDeltaFileName(timestampWithCompletionTime);
            }
            case "restore": {
                return HoodieTimeline.makeRestoreFileName(timestampWithCompletionTime);
            }
            case "replacecommit": {
                return HoodieTimeline.makeReplaceFileName(timestampWithCompletionTime);
            }
            case "indexing": {
                return HoodieTimeline.makeIndexCommitFileName(timestampWithCompletionTime);
            }
            case "schemacommit": {
                return HoodieTimeline.makeSchemaFileName(timestampWithCompletionTime);
            }
        }
        throw new IllegalArgumentException("Cannot get complete instant's file name for unknown action " + this.action);
    }

    public String getFileName() {
        if (this.isCompleted()) {
            return this.getCompleteFileName(this.completionTime);
        }
        return this.getPendingFileName();
    }

    public String getFileName(String completionTime) {
        ValidationUtils.checkState(this.isCompleted());
        return this.getCompleteFileName(completionTime);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        HoodieInstant that = (HoodieInstant)o;
        return this.state == that.state && Objects.equals(this.action, that.action) && Objects.equals(this.timestamp, that.timestamp);
    }

    public State getState() {
        return this.state;
    }

    public String getCompletionTime() {
        return this.completionTime;
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.state, this.action, this.timestamp});
    }

    @Override
    public int compareTo(HoodieInstant o) {
        return COMPARATOR.compare(this, o);
    }

    public String toString() {
        return "[" + (this.isInflight() || this.isRequested() ? "==>" : EMPTY_FILE_EXTENSION) + this.timestamp + (StringUtils.isNullOrEmpty(this.completionTime) ? EMPTY_FILE_EXTENSION : "__" + this.completionTime) + "__" + this.action + "__" + (Object)((Object)this.state) + "]";
    }

    public static enum State {
        REQUESTED,
        INFLIGHT,
        COMPLETED,
        NIL;

    }
}

