/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.io.storage;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.io.SeekableDataInputStream;
import org.apache.hudi.io.util.IOUtils;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;
import org.apache.hudi.storage.StoragePathFilter;
import org.apache.hudi.storage.StoragePathInfo;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

public abstract class TestHoodieStorageBase {
    @TempDir
    protected Path tempDir;
    protected static final String[] RELATIVE_FILE_PATHS = new String[]{"w/1.file", "w/2.file", "x/1.file", "x/2.file", "x/y/1.file", "x/y/2.file", "x/z/1.file", "x/z/2.file"};
    private static final byte[] EMPTY_BYTES = new byte[0];

    protected abstract HoodieStorage getStorage(Object var1, Object var2);

    protected abstract Object getFileSystem(Object var1);

    protected abstract Object getConf();

    @AfterEach
    public void cleanUpTempDir() {
        HoodieStorage storage = this.getStorage();
        try {
            for (StoragePathInfo pathInfo : storage.listDirectEntries(new StoragePath(this.getTempDir()))) {
                StoragePath path = pathInfo.getPath();
                if (pathInfo.isDirectory()) {
                    storage.deleteDirectory(path);
                    continue;
                }
                storage.deleteFile(path);
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testGetScheme() {
        Assertions.assertEquals((Object)"file", (Object)this.getStorage().getScheme());
    }

    @Test
    public void testGetUri() throws URISyntaxException {
        Assertions.assertEquals((Object)new URI("file:///"), (Object)this.getStorage().getUri());
    }

    @Test
    public void testCreateWriteAndRead() throws IOException {
        HoodieStorage storage = this.getStorage();
        StoragePath path = new StoragePath(this.getTempDir(), "testCreateAppendAndRead/1.file");
        Assertions.assertFalse((boolean)storage.exists(path));
        storage.create(path).close();
        this.validatePathInfo(storage, path, EMPTY_BYTES, false);
        byte[] data = new byte[]{2, 42, 49, -98, -23, 66, 9};
        try (OutputStream stream = storage.create(path);){
            stream.write(data);
            stream.flush();
        }
        this.validatePathInfo(storage, path, data, false);
        Assertions.assertThrows(IOException.class, () -> storage.create(path, false));
        this.validatePathInfo(storage, path, data, false);
        Assertions.assertThrows(IOException.class, () -> storage.create(path, false));
        this.validatePathInfo(storage, path, data, false);
        StoragePath path2 = new StoragePath(this.getTempDir(), "testCreateAppendAndRead/2.file");
        Assertions.assertFalse((boolean)storage.exists(path2));
        Assertions.assertTrue((boolean)storage.createNewFile(path2));
        this.validatePathInfo(storage, path2, EMPTY_BYTES, false);
        Assertions.assertFalse((boolean)storage.createNewFile(path2));
        StoragePath path3 = new StoragePath(this.getTempDir(), "testCreateAppendAndRead/3.file");
        Assertions.assertFalse((boolean)storage.exists(path3));
        storage.createImmutableFileInPath(path3, Option.of((Object)data));
        this.validatePathInfo(storage, path3, data, false);
        StoragePath path4 = new StoragePath(this.getTempDir(), "testCreateAppendAndRead/4");
        Assertions.assertFalse((boolean)storage.exists(path4));
        Assertions.assertTrue((boolean)storage.createDirectory(path4));
        this.validatePathInfo(storage, path4, EMPTY_BYTES, true);
        Assertions.assertTrue((boolean)storage.createDirectory(path4));
    }

    @Test
    public void testSeekable() throws IOException {
        HoodieStorage storage = this.getStorage();
        StoragePath path = new StoragePath(this.getTempDir(), "testSeekable/1.file");
        Assertions.assertFalse((boolean)storage.exists(path));
        byte[] data = new byte[]{2, 42, 49, -98, -23, 66, 9, 34, 79};
        try (OutputStream stream = storage.create(path);){
            stream.write(data);
            stream.flush();
        }
        var5_5 = null;
        try (SeekableDataInputStream seekableStream = storage.openSeekable(path, true);){
            this.validateSeekableDataInputStream(seekableStream, data);
        }
        catch (Throwable throwable) {
            var5_5 = throwable;
            throw throwable;
        }
        seekableStream = storage.openSeekable(path, 2, true);
        var5_5 = null;
        try {
            this.validateSeekableDataInputStream(seekableStream, data);
        }
        catch (Throwable throwable) {
            var5_5 = throwable;
            throw throwable;
        }
        finally {
            if (seekableStream != null) {
                if (var5_5 != null) {
                    try {
                        seekableStream.close();
                    }
                    catch (Throwable throwable) {
                        var5_5.addSuppressed(throwable);
                    }
                } else {
                    seekableStream.close();
                }
            }
        }
    }

    private void validateSeekableDataInputStream(SeekableDataInputStream seekableStream, byte[] expectedData) throws IOException {
        ArrayList<Integer> positionList = new ArrayList<Integer>();
        positionList.add(1);
        positionList.add(expectedData.length / 2);
        positionList.add(expectedData.length - 1);
        for (int i = 0; i < expectedData.length; ++i) {
            positionList.add(i);
        }
        Assertions.assertEquals((long)0L, (long)seekableStream.getPos());
        for (Integer pos : positionList) {
            seekableStream.seek((long)pos.intValue());
            Assertions.assertEquals((Integer)pos, (int)((int)seekableStream.getPos()));
            Assertions.assertEquals((byte)expectedData[pos], (byte)seekableStream.readByte());
        }
    }

    @Test
    public void testListing() throws IOException {
        HoodieStorage storage = this.getStorage();
        this.prepareFilesOnStorage(storage);
        this.validatePathInfoList(Arrays.stream(new StoragePathInfo[]{this.getStoragePathInfo("x/1.file", false), this.getStoragePathInfo("x/2.file", false), this.getStoragePathInfo("x/y", true), this.getStoragePathInfo("x/z", true)}).collect(Collectors.toList()), storage.listDirectEntries(new StoragePath(this.getTempDir(), "x")));
        this.validatePathInfoList(Arrays.stream(new StoragePathInfo[]{this.getStoragePathInfo("x/1.file", false), this.getStoragePathInfo("x/2.file", false), this.getStoragePathInfo("x/y/1.file", false), this.getStoragePathInfo("x/y/2.file", false), this.getStoragePathInfo("x/z/1.file", false), this.getStoragePathInfo("x/z/2.file", false)}).collect(Collectors.toList()), storage.listFiles(new StoragePath(this.getTempDir(), "x")));
        this.validatePathInfoList(Arrays.stream(new StoragePathInfo[]{this.getStoragePathInfo("x/2.file", false)}).collect(Collectors.toList()), storage.listDirectEntries(new StoragePath(this.getTempDir(), "x"), (StoragePathFilter & Serializable)e -> e.getName().contains("2")));
        this.validatePathInfoList(Arrays.stream(new StoragePathInfo[]{this.getStoragePathInfo("w/1.file", false), this.getStoragePathInfo("w/2.file", false), this.getStoragePathInfo("x/z/1.file", false), this.getStoragePathInfo("x/z/2.file", false)}).collect(Collectors.toList()), storage.listDirectEntries(Arrays.stream(new StoragePath[]{new StoragePath(this.getTempDir(), "w"), new StoragePath(this.getTempDir(), "x/z")}).collect(Collectors.toList())));
        Assertions.assertThrows(FileNotFoundException.class, () -> storage.listDirectEntries(new StoragePath(this.getTempDir(), "*")));
        this.validatePathInfoList(Arrays.stream(new StoragePathInfo[]{this.getStoragePathInfo("x/y/1.file", false), this.getStoragePathInfo("x/z/1.file", false)}).collect(Collectors.toList()), storage.globEntries(new StoragePath(this.getTempDir(), "x/*/1.file")));
        this.validatePathInfoList(Arrays.stream(new StoragePathInfo[]{this.getStoragePathInfo("x/1.file", false), this.getStoragePathInfo("x/2.file", false)}).collect(Collectors.toList()), storage.globEntries(new StoragePath(this.getTempDir(), "x/*.file")));
        this.validatePathInfoList(Arrays.stream(new StoragePathInfo[]{this.getStoragePathInfo("x/y/1.file", false)}).collect(Collectors.toList()), storage.globEntries(new StoragePath(this.getTempDir(), "x/*/*.file"), (StoragePathFilter & Serializable)e -> e.getParent().getName().equals("y") && e.getName().contains("1")));
    }

    @Test
    public void testFileNotFound() throws IOException {
        HoodieStorage storage = this.getStorage();
        StoragePath filePath = new StoragePath(this.getTempDir(), "testFileNotFound/1.file");
        StoragePath dirPath = new StoragePath(this.getTempDir(), "testFileNotFound/2");
        Assertions.assertFalse((boolean)storage.exists(filePath));
        Assertions.assertThrows(FileNotFoundException.class, () -> storage.open(filePath));
        Assertions.assertThrows(FileNotFoundException.class, () -> storage.getPathInfo(filePath));
        Assertions.assertThrows(FileNotFoundException.class, () -> storage.listDirectEntries(filePath));
        Assertions.assertThrows(FileNotFoundException.class, () -> storage.listDirectEntries(dirPath));
        Assertions.assertThrows(FileNotFoundException.class, () -> storage.listDirectEntries(dirPath, (StoragePathFilter & Serializable)e -> true));
        Assertions.assertThrows(FileNotFoundException.class, () -> storage.listDirectEntries(Arrays.stream(new StoragePath[]{dirPath}).collect(Collectors.toList())));
    }

    @Test
    public void testRename() throws IOException {
        HoodieStorage storage = this.getStorage();
        StoragePath path = new StoragePath(this.getTempDir(), "testRename/1.file");
        Assertions.assertFalse((boolean)storage.exists(path));
        storage.create(path).close();
        this.validatePathInfo(storage, path, EMPTY_BYTES, false);
        StoragePath newPath = new StoragePath(this.getTempDir(), "testRename/1_renamed.file");
        Assertions.assertTrue((boolean)storage.rename(path, newPath));
        Assertions.assertFalse((boolean)storage.exists(path));
        this.validatePathInfo(storage, newPath, EMPTY_BYTES, false);
    }

    @Test
    public void testDelete() throws IOException {
        HoodieStorage storage = this.getStorage();
        StoragePath path = new StoragePath(this.getTempDir(), "testDelete/1.file");
        Assertions.assertFalse((boolean)storage.exists(path));
        storage.create(path).close();
        Assertions.assertTrue((boolean)storage.exists(path));
        Assertions.assertTrue((boolean)storage.deleteFile(path));
        Assertions.assertFalse((boolean)storage.exists(path));
        Assertions.assertFalse((boolean)storage.deleteFile(path));
        StoragePath path2 = new StoragePath(this.getTempDir(), "testDelete/2");
        Assertions.assertFalse((boolean)storage.exists(path2));
        Assertions.assertTrue((boolean)storage.createDirectory(path2));
        Assertions.assertTrue((boolean)storage.exists(path2));
        Assertions.assertTrue((boolean)storage.deleteDirectory(path2));
        Assertions.assertFalse((boolean)storage.exists(path2));
        Assertions.assertFalse((boolean)storage.deleteDirectory(path2));
    }

    @Test
    public void testGetFileSystem() {
        Object conf = this.getConf();
        Object fs = this.getFileSystem(conf);
        HoodieStorage storage = this.getStorage(fs, conf);
        Assertions.assertSame((Object)fs, (Object)storage.getFileSystem());
    }

    protected String getTempDir() {
        return "file:" + this.tempDir.toUri().getPath();
    }

    private void prepareFilesOnStorage(HoodieStorage storage) throws IOException {
        String dir = this.getTempDir();
        for (String relativePath : RELATIVE_FILE_PATHS) {
            storage.create(new StoragePath(dir, relativePath)).close();
        }
    }

    private HoodieStorage getStorage() {
        Object conf = this.getConf();
        return this.getStorage(this.getFileSystem(conf), conf);
    }

    private StoragePathInfo getStoragePathInfo(String subPath, boolean isDirectory) {
        return new StoragePathInfo(new StoragePath(this.getTempDir(), subPath), 0L, isDirectory, 1, 1000000L, 10L);
    }

    private void validatePathInfo(HoodieStorage storage, StoragePath path, byte[] data, boolean isDirectory) throws IOException {
        Assertions.assertTrue((boolean)storage.exists(path));
        StoragePathInfo pathInfo = storage.getPathInfo(path);
        Assertions.assertEquals((Object)path, (Object)pathInfo.getPath());
        Assertions.assertEquals((Object)isDirectory, (Object)pathInfo.isDirectory());
        Assertions.assertEquals((Object)(!isDirectory ? 1 : 0), (Object)pathInfo.isFile());
        if (!isDirectory) {
            Assertions.assertEquals((long)data.length, (long)pathInfo.getLength());
            try (InputStream stream = storage.open(path);){
                Assertions.assertArrayEquals((byte[])data, (byte[])IOUtils.readAsByteArray((InputStream)stream, (int)data.length));
            }
        }
        Assertions.assertTrue((pathInfo.getModificationTime() > 0L ? 1 : 0) != 0);
    }

    private void validatePathInfoList(List<StoragePathInfo> expected, List<StoragePathInfo> actual) {
        Assertions.assertEquals((int)expected.size(), (int)actual.size());
        List sortedExpected = expected.stream().sorted(Comparator.comparing(StoragePathInfo::getPath)).collect(Collectors.toList());
        List sortedActual = actual.stream().sorted(Comparator.comparing(StoragePathInfo::getPath)).collect(Collectors.toList());
        for (int i = 0; i < expected.size(); ++i) {
            Assertions.assertEquals((Object)((StoragePathInfo)sortedExpected.get(i)).getPath(), (Object)((StoragePathInfo)sortedActual.get(i)).getPath());
            Assertions.assertEquals((Object)((StoragePathInfo)sortedExpected.get(i)).isDirectory(), (Object)((StoragePathInfo)sortedActual.get(i)).isDirectory());
            Assertions.assertEquals((Object)((StoragePathInfo)sortedExpected.get(i)).isFile(), (Object)((StoragePathInfo)sortedActual.get(i)).isFile());
            if (((StoragePathInfo)sortedExpected.get(i)).isFile()) {
                Assertions.assertEquals((long)((StoragePathInfo)sortedExpected.get(i)).getLength(), (long)((StoragePathInfo)sortedActual.get(i)).getLength());
            }
            Assertions.assertTrue((((StoragePathInfo)sortedActual.get(i)).getModificationTime() > 0L ? 1 : 0) != 0);
        }
    }
}

