/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.util;

import java.lang.constant.Constable;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Random;
import org.apache.hudi.common.util.StringUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class TestStringUtils {
    private static final String[] STRINGS = new String[]{"This", "is", "a", "test"};
    private static final String CHARACTERS_FOR_RANDOM_GEN = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789_/";
    private static final Random RANDOM = new SecureRandom();

    private static String toHexString(byte[] bytes) {
        StringBuilder sb = new StringBuilder(bytes.length * 2);
        for (byte b : bytes) {
            sb.append(String.format("%02x", b));
        }
        return sb.toString();
    }

    public static String generateRandomString(int length) {
        if (length < 1) {
            throw new IllegalArgumentException("Length must be greater than 0");
        }
        StringBuilder builder = new StringBuilder(length);
        for (int i = 0; i < length; ++i) {
            int randomIndex = RANDOM.nextInt(CHARACTERS_FOR_RANDOM_GEN.length());
            builder.append(CHARACTERS_FOR_RANDOM_GEN.charAt(randomIndex));
        }
        return new String(StringUtils.getUTF8Bytes((String)builder.toString()), StandardCharsets.UTF_8);
    }

    @Test
    public void testStringJoinWithDelim() {
        String joinedString = StringUtils.joinUsingDelim((String)"-", (String[])STRINGS);
        Assertions.assertEquals((int)STRINGS.length, (int)joinedString.split("-").length);
    }

    @Test
    public void testStringJoin() {
        Assertions.assertNotEquals(null, (Object)StringUtils.join((String[])new String[]{""}));
        Assertions.assertNotEquals(null, (Object)StringUtils.join((String[])STRINGS));
    }

    @Test
    public void testStringJoinWithMap() {
        HashMap<String, Constable> map = new HashMap<String, Constable>();
        map.put("a", Integer.valueOf(1));
        map.put("b", Boolean.valueOf(true));
        Assertions.assertNotNull((Object)StringUtils.join(map));
        Assertions.assertEquals((Object)"{a=1, b=true}", (Object)StringUtils.join(map));
    }

    @Test
    public void testStringJoinWithJavaImpl() {
        Assertions.assertNull((Object)StringUtils.join((CharSequence)",", null));
        Assertions.assertEquals((Object)"", (Object)String.join((CharSequence)",", Collections.singletonList("")));
        Assertions.assertEquals((Object)",", (Object)String.join((CharSequence)",", Arrays.asList("", "")));
        Assertions.assertEquals((Object)"a,", (Object)String.join((CharSequence)",", Arrays.asList("a", "")));
    }

    @Test
    public void testStringNullToEmpty() {
        String str = "This is a test";
        Assertions.assertEquals((Object)str, (Object)StringUtils.nullToEmpty((String)str));
        Assertions.assertEquals((Object)"", (Object)StringUtils.nullToEmpty(null));
    }

    @Test
    public void testStringObjToString() {
        Assertions.assertNull((Object)StringUtils.objToString(null));
        Assertions.assertEquals((Object)"Test String", (Object)StringUtils.objToString((Object)"Test String"));
        ByteBuffer byteBuffer1 = ByteBuffer.wrap(StringUtils.getUTF8Bytes((String)"1234"));
        ByteBuffer byteBuffer2 = ByteBuffer.wrap(StringUtils.getUTF8Bytes((String)"5678"));
        Assertions.assertEquals((Object)byteBuffer1.toString(), (Object)byteBuffer2.toString());
        Assertions.assertNotEquals((Object)StringUtils.objToString((Object)byteBuffer1), (Object)StringUtils.objToString((Object)byteBuffer2));
    }

    @Test
    public void testStringEmptyToNull() {
        Assertions.assertNull((Object)StringUtils.emptyToNull((String)""));
        Assertions.assertEquals((Object)"Test String", (Object)StringUtils.emptyToNull((String)"Test String"));
    }

    @Test
    public void testStringNullOrEmpty() {
        Assertions.assertTrue((boolean)StringUtils.isNullOrEmpty(null));
        Assertions.assertTrue((boolean)StringUtils.isNullOrEmpty((String)""));
        Assertions.assertNotEquals(null, (Object)StringUtils.isNullOrEmpty((String)"this is not empty"));
        Assertions.assertTrue((boolean)StringUtils.isNullOrEmpty((String)""));
    }

    @Test
    public void testSplit() {
        Assertions.assertEquals(new ArrayList(), (Object)StringUtils.split(null, (String)","));
        Assertions.assertEquals(new ArrayList(), (Object)StringUtils.split((String)"", (String)","));
        Assertions.assertEquals(Arrays.asList("a", "b", "c"), (Object)StringUtils.split((String)"a,b, c", (String)","));
        Assertions.assertEquals(Arrays.asList("a", "b", "c"), (Object)StringUtils.split((String)"a,b,, c ", (String)","));
    }

    @Test
    public void testHexString() {
        String str = "abcd";
        Assertions.assertEquals((Object)StringUtils.toHexString((byte[])StringUtils.getUTF8Bytes((String)str)), (Object)TestStringUtils.toHexString(StringUtils.getUTF8Bytes((String)str)));
    }

    @Test
    public void testTruncate() {
        Assertions.assertNull((Object)StringUtils.truncate(null, (int)10, (int)10));
        Assertions.assertEquals((Object)"http://use...ons/latest", (Object)StringUtils.truncate((String)"http://username:password@myregistry.com:5000/versions/latest", (int)10, (int)10));
        Assertions.assertEquals((Object)"http://abc.com", (Object)StringUtils.truncate((String)"http://abc.com", (int)10, (int)10));
    }

    @Test
    public void testCompareVersions() {
        Assertions.assertTrue((StringUtils.compareVersions((String)"1.10", (String)"1.9") > 0 ? 1 : 0) != 0);
        Assertions.assertTrue((StringUtils.compareVersions((String)"1.9", (String)"1.10") < 0 ? 1 : 0) != 0);
        Assertions.assertTrue((StringUtils.compareVersions((String)"1.100.1", (String)"1.10") > 0 ? 1 : 0) != 0);
        Assertions.assertTrue((StringUtils.compareVersions((String)"1.10.1", (String)"1.10") > 0 ? 1 : 0) != 0);
        Assertions.assertEquals((int)0, (int)StringUtils.compareVersions((String)"1.10", (String)"1.10"));
    }

    @Test
    void testConcatenateWithinThreshold() {
        String a = TestStringUtils.generateRandomString(1000);
        String b = TestStringUtils.generateRandomString(1048);
        int threshold = 2048;
        String result = StringUtils.concatenateWithThreshold((String)a, (String)b, (int)threshold);
        Assertions.assertEquals((int)threshold, (int)StringUtils.getUTF8Bytes((String)result).length);
        Assertions.assertEquals((Object)(a + b), (Object)result);
        String a2 = TestStringUtils.generateRandomString(900);
        String b2 = TestStringUtils.generateRandomString(1000);
        String result2 = StringUtils.concatenateWithThreshold((String)a2, (String)b2, (int)threshold);
        Assertions.assertEquals((Object)(a2 + b2), (Object)result2);
    }

    @Test
    void testConcatenateInvalidInput() {
        String a = TestStringUtils.generateRandomString(900);
        String b = TestStringUtils.generateRandomString(3000);
        Exception exception = (Exception)Assertions.assertThrows(IllegalArgumentException.class, () -> StringUtils.concatenateWithThreshold((String)a, (String)b, (int)2048));
        String expectedMessage = "Length of the Second string to concatenate exceeds the threshold (3000 > 2048)";
        String actualMessage = exception.getMessage();
        Assertions.assertTrue((boolean)actualMessage.contains(expectedMessage));
    }

    @Test
    void testConcatenateTruncateCase() {
        Assertions.assertEquals((Object)"ad", (Object)StringUtils.concatenateWithThreshold((String)"a\u00e9", (String)"d", (int)3));
        Assertions.assertEquals((Object)"\u4e16d", (Object)StringUtils.concatenateWithThreshold((String)"\u4e16\u754c", (String)"d", (int)4));
        Assertions.assertEquals((Object)"ad", (Object)StringUtils.concatenateWithThreshold((String)"ab", (String)"d", (int)2));
    }

    @Test
    void testGenerateInvalidRandomString() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> TestStringUtils.generateRandomString(-1));
    }

    @Test
    void testToStringWithThreshold() {
        String str1 = "string_value1";
        String str2 = "string_value2";
        String str3 = "string_value3";
        Assertions.assertEquals((Object)"", (Object)StringUtils.toStringWithThreshold(null, (int)10));
        Assertions.assertEquals((Object)"", (Object)StringUtils.toStringWithThreshold(Collections.emptyList(), (int)10));
        Assertions.assertEquals((Object)"..", (Object)StringUtils.toStringWithThreshold(Collections.singletonList(str1), (int)2));
        Assertions.assertEquals((Object)"string_...", (Object)StringUtils.toStringWithThreshold(Collections.singletonList(str1), (int)(str1.length() - 3)));
        Assertions.assertEquals((Object)"[string_value1]", (Object)StringUtils.toStringWithThreshold(Collections.singletonList(str1), (int)0));
        Assertions.assertEquals((Object)str1, (Object)StringUtils.toStringWithThreshold(Collections.singletonList(str1), (int)str1.length()));
        Assertions.assertEquals((Object)str1, (Object)StringUtils.toStringWithThreshold(Collections.singletonList(str1), (int)(str1.length() + 10)));
        ArrayList<String> stringList = new ArrayList<String>();
        stringList.add(str1);
        stringList.add(str2);
        stringList.add(str3);
        Assertions.assertEquals((Object)"string_val...", (Object)StringUtils.toStringWithThreshold(stringList, (int)str1.length()));
        Assertions.assertEquals((Object)"string_valu...", (Object)StringUtils.toStringWithThreshold(stringList, (int)(str1.length() + 1)));
        Assertions.assertEquals((Object)"string_value1,string...", (Object)StringUtils.toStringWithThreshold(stringList, (int)(str1.length() + str2.length() - 3)));
        Assertions.assertEquals((Object)"string_value1,string_v...", (Object)StringUtils.toStringWithThreshold(stringList, (int)(str1.length() + str2.length() - 1)));
        Assertions.assertEquals((Object)"string_value1,string_value2,strin...", (Object)StringUtils.toStringWithThreshold(stringList, (int)(str1.length() + str2.length() + str3.length() - 3)));
        Assertions.assertEquals((Object)"string_value1,string_value2,string_value3", (Object)StringUtils.toStringWithThreshold(stringList, (int)(str1.length() + str2.length() + str3.length() + 2)));
        Assertions.assertEquals((Object)"[string_value1, string_value2, string_value3]", (Object)StringUtils.toStringWithThreshold(stringList, (int)-1));
    }

    @Test
    public void testStripEnd() {
        Assertions.assertNull((Object)StringUtils.stripEnd(null, (String)"ab"));
        Assertions.assertEquals((Object)"", (Object)StringUtils.stripEnd((String)"", (String)"ab"));
        Assertions.assertEquals((Object)"abc", (Object)StringUtils.stripEnd((String)"abc", null));
        Assertions.assertEquals((Object)"abc", (Object)StringUtils.stripEnd((String)"abc  ", null));
        Assertions.assertEquals((Object)"abc", (Object)StringUtils.stripEnd((String)"abc", (String)""));
        Assertions.assertEquals((Object)"abc", (Object)StringUtils.stripEnd((String)"abcabab", (String)"ab"));
    }
}

