/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.util;

import java.io.IOException;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;
import org.apache.hudi.common.util.ReflectionUtils;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.exception.HoodieException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RetryHelper<T, R extends Exception>
implements Serializable {
    private static final Logger LOG = LoggerFactory.getLogger(RetryHelper.class);
    private static final List<? extends Class<? extends Exception>> DEFAULT_RETRY_EXCEPTIONS = Arrays.asList(IOException.class, RuntimeException.class);
    private transient CheckedFunction<T, R> func;
    private final int num;
    private final long maxIntervalTime;
    private final long initialIntervalTime;
    private String taskInfo = "N/A";
    private List<? extends Class<? extends Exception>> retryExceptionsClasses;

    public RetryHelper(long maxRetryIntervalMs, int maxRetryNumbers, long initialRetryIntervalMs, List<Class<? extends Exception>> retryExceptions, String taskInfo) {
        this.num = maxRetryNumbers;
        this.initialIntervalTime = initialRetryIntervalMs;
        this.maxIntervalTime = maxRetryIntervalMs;
        this.retryExceptionsClasses = retryExceptions;
        this.taskInfo = taskInfo;
    }

    public RetryHelper(long maxRetryIntervalMs, int maxRetryNumbers, long initialRetryIntervalMs, String retryExceptions) {
        this.num = maxRetryNumbers;
        this.initialIntervalTime = initialRetryIntervalMs;
        this.maxIntervalTime = maxRetryIntervalMs;
        if (StringUtils.isNullOrEmpty(retryExceptions)) {
            this.retryExceptionsClasses = DEFAULT_RETRY_EXCEPTIONS;
        } else {
            try {
                this.retryExceptionsClasses = Arrays.stream(retryExceptions.split(",")).map(exception -> (Exception)ReflectionUtils.loadClass(exception, "")).map(Object::getClass).collect(Collectors.toList());
            }
            catch (HoodieException e) {
                LOG.error("Exception while loading retry exceptions classes '" + retryExceptions + "'.", (Throwable)e);
                this.retryExceptionsClasses = DEFAULT_RETRY_EXCEPTIONS;
            }
        }
    }

    public RetryHelper(long maxRetryIntervalMs, int maxRetryNumbers, long initialRetryIntervalMs, String retryExceptions, String taskInfo) {
        this(maxRetryIntervalMs, maxRetryNumbers, initialRetryIntervalMs, retryExceptions);
        this.taskInfo = taskInfo == null ? "None" : taskInfo;
    }

    public RetryHelper<T, R> tryWith(CheckedFunction<T, R> func) {
        this.func = func;
        return this;
    }

    public T start(CheckedFunction<T, R> func) throws R {
        int retries = 0;
        T functionResult = null;
        while (true) {
            long waitTime = Math.min(this.getWaitTimeExp(retries), this.maxIntervalTime);
            try {
                functionResult = func.get();
            }
            catch (Exception e) {
                if (!this.checkIfExceptionInRetryList(e)) {
                    throw e;
                }
                if (retries++ >= this.num) {
                    String message = "Still failed to " + this.taskInfo + " after retried " + this.num + " times.";
                    LOG.error(message, (Throwable)e);
                    throw e;
                }
                LOG.warn("Task [{}] failed. current retry number {}, will retry after {} ms.", new Object[]{this.taskInfo, retries, waitTime, e});
                try {
                    Thread.sleep(waitTime);
                }
                catch (InterruptedException interruptedException) {}
                continue;
            }
            break;
        }
        if (retries > 0) {
            LOG.info("Success to {} after retried {} times.", (Object)this.taskInfo, (Object)retries);
        }
        return functionResult;
    }

    public T start() throws R {
        return this.start(this.func);
    }

    private boolean checkIfExceptionInRetryList(Exception e) {
        boolean inRetryList = false;
        for (Class<? extends Exception> clazz : this.retryExceptionsClasses) {
            if (!clazz.isInstance(e)) continue;
            inRetryList = true;
            break;
        }
        return inRetryList;
    }

    private long getWaitTimeExp(int retryCount) {
        Random random = new Random();
        if (0 == retryCount) {
            return this.initialIntervalTime;
        }
        long waitTime = (long)Math.pow(2.0, retryCount) * this.initialIntervalTime;
        if (waitTime < 0L) {
            return Long.MAX_VALUE;
        }
        return (waitTime += (long)random.nextInt(100)) < 0L ? Long.MAX_VALUE : waitTime;
    }

    @FunctionalInterface
    public static interface CheckedFunction<T, R extends Exception>
    extends Serializable {
        public T get() throws R;
    }
}

