/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.log;

import java.io.EOFException;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import javax.annotation.Nullable;
import org.apache.avro.Schema;
import org.apache.hudi.common.config.HoodieReaderConfig;
import org.apache.hudi.common.fs.FSUtils;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.model.HoodieRecord;
import org.apache.hudi.common.table.log.HoodieLogFormat;
import org.apache.hudi.common.table.log.HoodieLogFormatVersion;
import org.apache.hudi.common.table.log.block.HoodieAvroDataBlock;
import org.apache.hudi.common.table.log.block.HoodieCDCDataBlock;
import org.apache.hudi.common.table.log.block.HoodieCommandBlock;
import org.apache.hudi.common.table.log.block.HoodieCorruptBlock;
import org.apache.hudi.common.table.log.block.HoodieDeleteBlock;
import org.apache.hudi.common.table.log.block.HoodieHFileDataBlock;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.common.table.log.block.HoodieParquetDataBlock;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.exception.CorruptedLogFileException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.HoodieNotSupportedException;
import org.apache.hudi.internal.schema.InternalSchema;
import org.apache.hudi.io.SeekableDataInputStream;
import org.apache.hudi.io.util.IOUtils;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;
import org.apache.hudi.storage.StorageSchemes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HoodieLogFileReader
implements HoodieLogFormat.Reader {
    public static final int DEFAULT_BUFFER_SIZE = 0x1000000;
    private static final int BLOCK_SCAN_READ_BUFFER_SIZE = 0x100000;
    private static final Logger LOG = LoggerFactory.getLogger(HoodieLogFileReader.class);
    private static final String REVERSE_LOG_READER_HAS_NOT_BEEN_ENABLED = "Reverse log reader has not been enabled";
    private final HoodieStorage storage;
    private final HoodieLogFile logFile;
    private final int bufferSize;
    private final byte[] magicBuffer = new byte[6];
    private final Schema readerSchema;
    private final InternalSchema internalSchema;
    private final String keyField;
    private long reverseLogFilePosition;
    private long lastReverseLogFilePosition;
    private final boolean reverseReader;
    private final boolean enableRecordLookups;
    private boolean closed = false;
    private SeekableDataInputStream inputStream;

    public HoodieLogFileReader(HoodieStorage storage, HoodieLogFile logFile, Schema readerSchema, int bufferSize) throws IOException {
        this(storage, logFile, readerSchema, bufferSize, false);
    }

    public HoodieLogFileReader(HoodieStorage storage, HoodieLogFile logFile, Schema readerSchema, int bufferSize, boolean reverseReader) throws IOException {
        this(storage, logFile, readerSchema, bufferSize, reverseReader, false, HoodieRecord.RECORD_KEY_METADATA_FIELD);
    }

    public HoodieLogFileReader(HoodieStorage storage, HoodieLogFile logFile, Schema readerSchema, int bufferSize, boolean reverseReader, boolean enableRecordLookups, String keyField) throws IOException {
        this(storage, logFile, readerSchema, bufferSize, reverseReader, enableRecordLookups, keyField, InternalSchema.getEmptyInternalSchema());
    }

    public HoodieLogFileReader(HoodieStorage storage, HoodieLogFile logFile, Schema readerSchema, int bufferSize, boolean reverseReader, boolean enableRecordLookups, String keyField, InternalSchema internalSchema) throws IOException {
        this.storage = storage;
        StoragePath updatedPath = FSUtils.makeQualified(storage, logFile.getPath());
        this.logFile = updatedPath.equals(logFile.getPath()) ? logFile : new HoodieLogFile(updatedPath, logFile.getFileSize());
        this.bufferSize = bufferSize;
        this.inputStream = HoodieLogFileReader.getDataInputStream(this.storage, this.logFile, bufferSize);
        this.readerSchema = readerSchema;
        this.reverseReader = reverseReader;
        this.enableRecordLookups = enableRecordLookups;
        this.keyField = keyField;
        InternalSchema internalSchema2 = this.internalSchema = internalSchema == null ? InternalSchema.getEmptyInternalSchema() : internalSchema;
        if (this.reverseReader) {
            this.reverseLogFilePosition = this.lastReverseLogFilePosition = this.logFile.getFileSize();
        }
    }

    @Override
    public HoodieLogFile getLogFile() {
        return this.logFile;
    }

    private HoodieLogBlock readBlock() throws IOException {
        Map<HoodieLogBlock.FooterMetadataType, String> footer;
        int blockSize;
        long blockStartPos = this.inputStream.getPos();
        try {
            blockSize = (int)this.inputStream.readLong();
        }
        catch (EOFException | CorruptedLogFileException e) {
            return this.createCorruptBlock(blockStartPos);
        }
        boolean isCorrupted = this.isBlockCorrupted(blockSize);
        if (isCorrupted) {
            return this.createCorruptBlock(blockStartPos);
        }
        HoodieLogFormat.LogFormatVersion nextBlockVersion = this.readVersion();
        HoodieLogBlock.HoodieLogBlockType blockType = this.tryReadBlockType(nextBlockVersion);
        Map<HoodieLogBlock.HeaderMetadataType, String> header = nextBlockVersion.hasHeader() ? HoodieLogBlock.getHeaderMetadata(this.inputStream) : null;
        int contentLength = nextBlockVersion.getVersion() != 0 ? (int)this.inputStream.readLong() : blockSize;
        long contentPosition = this.inputStream.getPos();
        boolean shouldReadLazily = nextBlockVersion.getVersion() != 0;
        Option<byte[]> content = HoodieLogBlock.tryReadContent(this.inputStream, contentLength, shouldReadLazily);
        Map<HoodieLogBlock.FooterMetadataType, String> map = footer = nextBlockVersion.hasFooter() ? HoodieLogBlock.getFooterMetadata(this.inputStream) : null;
        if (nextBlockVersion.hasLogBlockLength()) {
            this.inputStream.readLong();
        }
        long blockEndPos = this.inputStream.getPos();
        HoodieLogBlock.HoodieLogBlockContentLocation logBlockContentLoc = new HoodieLogBlock.HoodieLogBlockContentLocation(this.storage, this.logFile, contentPosition, contentLength, blockEndPos);
        switch (Objects.requireNonNull(blockType)) {
            case AVRO_DATA_BLOCK: {
                if (nextBlockVersion.getVersion() == 0) {
                    return HoodieAvroDataBlock.getBlock(content.get(), this.readerSchema, this.internalSchema);
                }
                return new HoodieAvroDataBlock(() -> HoodieLogFileReader.getDataInputStream(this.storage, this.logFile, this.bufferSize), content, true, logBlockContentLoc, this.getTargetReaderSchemaForBlock(), header, footer, this.keyField);
            }
            case HFILE_DATA_BLOCK: {
                ValidationUtils.checkState(nextBlockVersion.getVersion() != 0, String.format("HFile block could not be of version (%d)", 0));
                return new HoodieHFileDataBlock(() -> HoodieLogFileReader.getDataInputStream(this.storage, this.logFile, this.bufferSize), content, true, logBlockContentLoc, Option.ofNullable(this.readerSchema), header, footer, this.enableRecordLookups, this.logFile.getPath(), this.storage.getConf().getBoolean(HoodieReaderConfig.USE_NATIVE_HFILE_READER.key(), HoodieReaderConfig.USE_NATIVE_HFILE_READER.defaultValue()));
            }
            case PARQUET_DATA_BLOCK: {
                ValidationUtils.checkState(nextBlockVersion.getVersion() != 0, String.format("Parquet block could not be of version (%d)", 0));
                return new HoodieParquetDataBlock(() -> HoodieLogFileReader.getDataInputStream(this.storage, this.logFile, this.bufferSize), content, true, logBlockContentLoc, this.getTargetReaderSchemaForBlock(), header, footer, this.keyField);
            }
            case DELETE_BLOCK: {
                return new HoodieDeleteBlock(content, () -> HoodieLogFileReader.getDataInputStream(this.storage, this.logFile, this.bufferSize), true, Option.of(logBlockContentLoc), header, footer);
            }
            case COMMAND_BLOCK: {
                return new HoodieCommandBlock(content, () -> HoodieLogFileReader.getDataInputStream(this.storage, this.logFile, this.bufferSize), true, Option.of(logBlockContentLoc), header, footer);
            }
            case CDC_DATA_BLOCK: {
                return new HoodieCDCDataBlock(() -> HoodieLogFileReader.getDataInputStream(this.storage, this.logFile, this.bufferSize), content, true, logBlockContentLoc, this.readerSchema, header, this.keyField);
            }
        }
        throw new HoodieNotSupportedException("Unsupported Block " + (Object)((Object)blockType));
    }

    private Option<Schema> getTargetReaderSchemaForBlock() {
        if (this.internalSchema.isEmptySchema()) {
            return Option.ofNullable(this.readerSchema);
        }
        return Option.empty();
    }

    @Nullable
    private HoodieLogBlock.HoodieLogBlockType tryReadBlockType(HoodieLogFormat.LogFormatVersion blockVersion) throws IOException {
        if (blockVersion.getVersion() == 0) {
            return null;
        }
        int type = this.inputStream.readInt();
        ValidationUtils.checkArgument(type < HoodieLogBlock.HoodieLogBlockType.values().length, "Invalid block byte type found " + type);
        return HoodieLogBlock.HoodieLogBlockType.values()[type];
    }

    private HoodieLogBlock createCorruptBlock(long blockStartPos) throws IOException {
        LOG.info("Log {} has a corrupted block at {}", (Object)this.logFile, (Object)blockStartPos);
        this.inputStream.seek(blockStartPos);
        long nextBlockOffset = this.scanForNextAvailableBlockOffset();
        this.inputStream.seek(blockStartPos);
        LOG.info("Next available block in {} starts at {}", (Object)this.logFile, (Object)nextBlockOffset);
        int corruptedBlockSize = (int)(nextBlockOffset - blockStartPos);
        long contentPosition = this.inputStream.getPos();
        Option<byte[]> corruptedBytes = HoodieLogBlock.tryReadContent(this.inputStream, corruptedBlockSize, true);
        HoodieLogBlock.HoodieLogBlockContentLocation logBlockContentLoc = new HoodieLogBlock.HoodieLogBlockContentLocation(this.storage, this.logFile, contentPosition, corruptedBlockSize, nextBlockOffset);
        return new HoodieCorruptBlock(corruptedBytes, () -> HoodieLogFileReader.getDataInputStream(this.storage, this.logFile, this.bufferSize), true, Option.of(logBlockContentLoc), new HashMap<HoodieLogBlock.HeaderMetadataType, String>(), new HashMap<HoodieLogBlock.FooterMetadataType, String>());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean isBlockCorrupted(int blocksize) throws IOException {
        long blockSizeFromFooter;
        if (StorageSchemes.isWriteTransactional(this.storage.getScheme())) {
            return false;
        }
        long currentPos = this.inputStream.getPos();
        try {
            this.inputStream.seek(currentPos + (long)blocksize - 8L);
            blockSizeFromFooter = this.inputStream.readLong() - (long)this.magicBuffer.length;
        }
        catch (EOFException e) {
            LOG.info("Found corrupted block in file {} with block size({}) running past EOF", (Object)this.logFile, (Object)blocksize);
            this.inputStream.seek(currentPos);
            return true;
        }
        if ((long)blocksize != blockSizeFromFooter) {
            LOG.info("Found corrupted block in file {}. Header block size({}) did not match the footer block size({})", this.logFile, blocksize, blockSizeFromFooter);
            this.inputStream.seek(currentPos);
            return true;
        }
        try {
            this.readMagic();
            boolean e = false;
            return e;
        }
        catch (CorruptedLogFileException e) {
            LOG.info("Found corrupted block in file {}. No magic hash found right after footer block size entry", (Object)this.logFile);
            boolean bl = true;
            return bl;
        }
        finally {
            this.inputStream.seek(currentPos);
        }
    }

    private long scanForNextAvailableBlockOffset() throws IOException {
        byte[] dataBuf = new byte[0x100000];
        boolean eof = false;
        while (true) {
            long currentPos = this.inputStream.getPos();
            try {
                Arrays.fill(dataBuf, (byte)0);
                this.inputStream.readFully(dataBuf, 0, dataBuf.length);
            }
            catch (EOFException e) {
                eof = true;
            }
            long pos = IOUtils.indexOf(dataBuf, HoodieLogFormat.MAGIC);
            if (pos >= 0L) {
                return currentPos + pos;
            }
            if (eof) {
                return this.inputStream.getPos();
            }
            this.inputStream.seek(currentPos + (long)dataBuf.length - (long)HoodieLogFormat.MAGIC.length);
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.closed) {
            LOG.info("Closing Log file reader {}", (Object)this.logFile.getFileName());
            if (null != this.inputStream) {
                this.inputStream.close();
            }
            this.closed = true;
        }
    }

    @Override
    public boolean hasNext() {
        try {
            return this.readMagic();
        }
        catch (IOException e) {
            throw new HoodieIOException("IOException when reading logfile " + this.logFile, e);
        }
    }

    private HoodieLogFormat.LogFormatVersion readVersion() throws IOException {
        return new HoodieLogFormatVersion(this.inputStream.readInt());
    }

    private boolean readMagic() throws IOException {
        try {
            if (!this.hasNextMagic()) {
                throw new CorruptedLogFileException(this.logFile + " could not be read. Did not find the magic bytes at the start of the block");
            }
            return true;
        }
        catch (EOFException e) {
            return false;
        }
    }

    private boolean hasNextMagic() throws IOException {
        this.inputStream.readFully(this.magicBuffer, 0, 6);
        return Arrays.equals(this.magicBuffer, HoodieLogFormat.MAGIC);
    }

    @Override
    public HoodieLogBlock next() {
        try {
            return this.readBlock();
        }
        catch (IOException io) {
            throw new HoodieIOException("IOException when reading logblock from log file " + this.logFile, io);
        }
    }

    @Override
    public boolean hasPrev() {
        try {
            if (!this.reverseReader) {
                throw new HoodieNotSupportedException(REVERSE_LOG_READER_HAS_NOT_BEEN_ENABLED);
            }
            this.reverseLogFilePosition = this.lastReverseLogFilePosition;
            this.reverseLogFilePosition -= 8L;
            this.lastReverseLogFilePosition = this.reverseLogFilePosition;
            this.inputStream.seek(this.reverseLogFilePosition);
        }
        catch (Exception e) {
            return false;
        }
        return true;
    }

    @Override
    public HoodieLogBlock prev() throws IOException {
        if (!this.reverseReader) {
            throw new HoodieNotSupportedException(REVERSE_LOG_READER_HAS_NOT_BEEN_ENABLED);
        }
        long blockSize = this.inputStream.readLong();
        long blockEndPos = this.inputStream.getPos();
        try {
            this.inputStream.seek(this.reverseLogFilePosition - blockSize);
        }
        catch (Exception e) {
            this.inputStream.seek(blockEndPos);
            throw new CorruptedLogFileException("Found possible corrupted block, cannot read log file in reverse, fallback to forward reading of logfile");
        }
        boolean hasNext = this.hasNext();
        this.reverseLogFilePosition -= blockSize;
        this.lastReverseLogFilePosition = this.reverseLogFilePosition;
        return this.next();
    }

    public long moveToPrev() throws IOException {
        if (!this.reverseReader) {
            throw new HoodieNotSupportedException(REVERSE_LOG_READER_HAS_NOT_BEEN_ENABLED);
        }
        this.inputStream.seek(this.lastReverseLogFilePosition);
        long blockSize = this.inputStream.readLong();
        this.inputStream.seek(this.reverseLogFilePosition - blockSize);
        this.reverseLogFilePosition -= blockSize;
        this.lastReverseLogFilePosition = this.reverseLogFilePosition;
        return this.reverseLogFilePosition;
    }

    @Override
    public void remove() {
        throw new UnsupportedOperationException("Remove not supported for HoodieLogFileReader");
    }

    public static SeekableDataInputStream getDataInputStream(HoodieStorage storage, HoodieLogFile logFile, int bufferSize) {
        try {
            return storage.openSeekable(logFile.getPath(), bufferSize, true);
        }
        catch (IOException e) {
            throw new HoodieIOException("Unable to get seekable input stream for " + logFile, e);
        }
    }
}

