/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.read;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.apache.hudi.common.model.HoodieTableType;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.log.InstantRange;
import org.apache.hudi.common.table.timeline.CompletionTimeQueryView;
import org.apache.hudi.common.table.timeline.HoodieArchivedTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.util.ClusteringUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.common.util.VisibleForTesting;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IncrementalQueryAnalyzer {
    public static final String START_COMMIT_EARLIEST = "earliest";
    private static final Logger LOG = LoggerFactory.getLogger(IncrementalQueryAnalyzer.class);
    private final HoodieTableMetaClient metaClient;
    private final Option<String> startCompletionTime;
    private final Option<String> endCompletionTime;
    private final InstantRange.RangeType rangeType;
    private final boolean skipCompaction;
    private final boolean skipClustering;
    private final boolean skipInsertOverwrite;
    private final boolean readCdcFromChangelog;
    private final int limit;

    private IncrementalQueryAnalyzer(HoodieTableMetaClient metaClient, String startCompletionTime, String endCompletionTime, InstantRange.RangeType rangeType, boolean skipCompaction, boolean skipClustering, boolean skipInsertOverwrite, boolean readCdcFromChangelog, int limit) {
        this.metaClient = metaClient;
        this.startCompletionTime = Option.ofNullable(startCompletionTime);
        this.endCompletionTime = Option.ofNullable(endCompletionTime);
        this.rangeType = rangeType;
        this.skipCompaction = skipCompaction;
        this.skipClustering = skipClustering;
        this.skipInsertOverwrite = skipInsertOverwrite;
        this.readCdcFromChangelog = readCdcFromChangelog;
        this.limit = limit;
    }

    public Option<String> getStartCompletionTime() {
        return this.startCompletionTime;
    }

    public static Builder builder() {
        return new Builder();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public QueryContext analyze() {
        try (CompletionTimeQueryView completionTimeQueryView = this.metaClient.getTimelineLayout().getTimelineFactory().createCompletionTimeQueryView(this.metaClient);){
            List<String> instants;
            if (completionTimeQueryView.isEmptyTable()) {
                QueryContext queryContext = QueryContext.EMPTY;
                return queryContext;
            }
            HoodieTimeline filteredTimeline = this.getFilteredTimeline(this.metaClient);
            List<String> instantTimeList = completionTimeQueryView.getInstantTimes(filteredTimeline, this.startCompletionTime, this.endCompletionTime, this.rangeType);
            if (instantTimeList.isEmpty()) {
                QueryContext queryContext = QueryContext.EMPTY;
                return queryContext;
            }
            Pair<List<String>, List<String>> splitInstantTime = IncrementalQueryAnalyzer.splitInstantByActiveness(instantTimeList, completionTimeQueryView);
            HashSet<String> instantTimeSet = new HashSet<String>(instantTimeList);
            List<String> archivedInstantTime = splitInstantTime.getLeft();
            List<String> activeInstantTime = splitInstantTime.getRight();
            ArrayList<HoodieInstant> archivedInstants = new ArrayList();
            List<Object> activeInstants = new ArrayList();
            HoodieTimeline archivedReadTimeline = null;
            if (!activeInstantTime.isEmpty()) {
                activeInstants = filteredTimeline.getInstantsAsStream().filter(instant -> instantTimeSet.contains(instant.requestedTime())).collect(Collectors.toList());
                if (this.limit > 0 && this.limit < activeInstants.size()) {
                    activeInstants = activeInstants.subList(0, this.limit);
                }
            }
            if (!archivedInstantTime.isEmpty()) {
                archivedReadTimeline = this.getArchivedReadTimeline(this.metaClient, archivedInstantTime.get(0));
                archivedInstants = archivedReadTimeline.getInstantsAsStream().filter(instant -> instantTimeSet.contains(instant.requestedTime())).collect(Collectors.toList());
            }
            if ((instants = Stream.concat(archivedInstants.stream(), activeInstants.stream()).map(HoodieInstant::requestedTime).collect(Collectors.toList())).isEmpty()) {
                QueryContext queryContext = QueryContext.EMPTY;
                return queryContext;
            }
            if (this.startCompletionTime.isEmpty() && this.endCompletionTime.isPresent()) {
                instants = Collections.singletonList(instants.get(instants.size() - 1));
            }
            String lastInstant = (String)instants.get(instants.size() - 1);
            String startInstant = START_COMMIT_EARLIEST.equalsIgnoreCase(this.startCompletionTime.orElse(null)) ? null : (this.startCompletionTime.isEmpty() ? lastInstant : (String)instants.get(0));
            String endInstant = this.endCompletionTime.isEmpty() ? null : lastInstant;
            QueryContext queryContext = QueryContext.create(startInstant, endInstant, instants, archivedInstants, activeInstants, filteredTimeline, archivedReadTimeline);
            return queryContext;
        }
        catch (Exception ex) {
            LOG.error("Got exception when generating incremental query info", ex);
            throw new HoodieException(ex);
        }
    }

    private static Pair<List<String>, List<String>> splitInstantByActiveness(List<String> instantTimeList, CompletionTimeQueryView completionTimeQueryView) {
        int firstActiveIdx = IntStream.range(0, instantTimeList.size()).filter(i -> !completionTimeQueryView.isArchived((String)instantTimeList.get(i))).findFirst().orElse(-1);
        if (firstActiveIdx == -1) {
            return Pair.of(instantTimeList, Collections.emptyList());
        }
        if (firstActiveIdx == 0) {
            return Pair.of(Collections.emptyList(), instantTimeList);
        }
        return Pair.of(instantTimeList.subList(0, firstActiveIdx), instantTimeList.subList(firstActiveIdx, instantTimeList.size()));
    }

    private HoodieTimeline getFilteredTimeline(HoodieTableMetaClient metaClient) {
        HoodieTimeline timeline = metaClient.getCommitsAndCompactionTimeline().filterCompletedAndCompactionInstants();
        return IncrementalQueryAnalyzer.filterInstantsAsPerUserConfigs(metaClient, timeline, this.skipCompaction, this.skipClustering, this.skipInsertOverwrite, this.readCdcFromChangelog);
    }

    private HoodieTimeline getArchivedReadTimeline(HoodieTableMetaClient metaClient, String startInstant) {
        HoodieArchivedTimeline archivedTimeline = metaClient.getArchivedTimeline(startInstant, false);
        HoodieTimeline archivedCompleteTimeline = archivedTimeline.getCommitsTimeline().filterCompletedInstants();
        return IncrementalQueryAnalyzer.filterInstantsAsPerUserConfigs(metaClient, archivedCompleteTimeline, this.skipCompaction, this.skipClustering, this.skipInsertOverwrite, this.readCdcFromChangelog);
    }

    @VisibleForTesting
    public static HoodieTimeline filterInstantsAsPerUserConfigs(HoodieTableMetaClient metaClient, HoodieTimeline timeline, boolean skipCompaction, boolean skipClustering, boolean skipInsertOverwrite) {
        return IncrementalQueryAnalyzer.filterInstantsAsPerUserConfigs(metaClient, timeline, skipCompaction, skipClustering, skipInsertOverwrite, false);
    }

    private static HoodieTimeline filterInstantsAsPerUserConfigs(HoodieTableMetaClient metaClient, HoodieTimeline timeline, boolean skipCompaction, boolean skipClustering, boolean skipInsertOverwrite, boolean readCdcFromChangelog) {
        HoodieTimeline oriTimeline = timeline;
        if (metaClient.getTableType() == HoodieTableType.MERGE_ON_READ && skipCompaction) {
            timeline = timeline.filter(instant -> !instant.getAction().equals("commit"));
        }
        if (metaClient.getTableType() == HoodieTableType.MERGE_ON_READ && metaClient.getTableConfig().isCDCEnabled() && readCdcFromChangelog) {
            timeline = timeline.filter(instant -> !instant.getAction().equals("deltacommit"));
        }
        if (skipClustering) {
            timeline = timeline.filter(instant -> !ClusteringUtils.isCompletedClusteringInstant(instant, oriTimeline));
        }
        if (skipInsertOverwrite) {
            timeline = timeline.filter(instant -> !ClusteringUtils.isInsertOverwriteInstant(instant, oriTimeline));
        }
        return timeline;
    }

    public static class QueryContext {
        public static final QueryContext EMPTY = new QueryContext(null, null, Collections.emptyList(), Collections.emptyList(), Collections.emptyList(), null, null);
        private final Option<String> startInstant;
        private final Option<String> endInstant;
        private final List<HoodieInstant> archivedInstants;
        private final List<HoodieInstant> activeInstants;
        private final HoodieTimeline activeTimeline;
        private final HoodieTimeline archivedTimeline;
        private final List<String> instants;

        private QueryContext(@Nullable String startInstant, @Nullable String endInstant, List<String> instants, List<HoodieInstant> archivedInstants, List<HoodieInstant> activeInstants, HoodieTimeline activeTimeline, @Nullable HoodieTimeline archivedTimeline) {
            this.startInstant = Option.ofNullable(startInstant);
            this.endInstant = Option.ofNullable(endInstant);
            this.archivedInstants = archivedInstants;
            this.activeInstants = activeInstants;
            this.activeTimeline = activeTimeline;
            this.archivedTimeline = archivedTimeline;
            this.instants = instants;
        }

        public static QueryContext create(@Nullable String startInstant, @Nullable String endInstant, List<String> instants, List<HoodieInstant> archivedInstants, List<HoodieInstant> activeInstants, HoodieTimeline activeTimeline, @Nullable HoodieTimeline archivedTimeline) {
            return new QueryContext(startInstant, endInstant, instants, archivedInstants, activeInstants, activeTimeline, archivedTimeline);
        }

        public boolean isEmpty() {
            return this.instants.isEmpty();
        }

        public List<String> getInstantTimeList() {
            return this.instants;
        }

        public Option<String> getStartInstant() {
            return this.startInstant;
        }

        public Option<String> getEndInstant() {
            return this.endInstant;
        }

        public String getLastInstant() {
            ValidationUtils.checkState(!this.instants.isEmpty());
            return this.instants.get(this.instants.size() - 1);
        }

        public List<HoodieInstant> getInstants() {
            return Stream.concat(this.archivedInstants.stream(), this.activeInstants.stream()).collect(Collectors.toList());
        }

        public List<HoodieInstant> getArchivedInstants() {
            return this.archivedInstants;
        }

        public List<HoodieInstant> getActiveInstants() {
            return this.activeInstants;
        }

        public boolean isConsumingFromEarliest() {
            return this.startInstant.isEmpty();
        }

        public boolean isConsumingToLatest() {
            return this.endInstant.isEmpty();
        }

        public String getMaxCompletionTime() {
            if (this.activeInstants.size() > 0) {
                return this.activeInstants.stream().map(HoodieInstant::getCompletionTime).filter(Objects::nonNull).max(String::compareTo).get();
            }
            return this.activeTimeline.getInstantsAsStream().map(HoodieInstant::getCompletionTime).filter(Objects::nonNull).max(String::compareTo).get();
        }

        public Option<InstantRange> getInstantRange() {
            if (this.isConsumingFromEarliest()) {
                if (this.isConsumingToLatest()) {
                    return Option.empty();
                }
                return Option.of(InstantRange.builder().startInstant(this.startInstant.orElse(null)).endInstant(this.endInstant.orElse(null)).rangeType(InstantRange.RangeType.CLOSED_CLOSED).nullableBoundary(true).build());
            }
            return Option.of(InstantRange.builder().rangeType(InstantRange.RangeType.EXACT_MATCH).explicitInstants(new HashSet<String>(this.instants)).build());
        }

        public HoodieTimeline getActiveTimeline() {
            return this.activeTimeline;
        }

        @Nullable
        public HoodieTimeline getArchivedTimeline() {
            return this.archivedTimeline;
        }
    }

    public static class Builder {
        private String startCompletionTime;
        private String endCompletionTime;
        private InstantRange.RangeType rangeType;
        private HoodieTableMetaClient metaClient;
        private boolean skipCompaction = false;
        private boolean skipClustering = false;
        private boolean skipInsertOverwrite = false;
        private boolean readCdcFromChangelog = false;
        private int limit = -1;

        public Builder startCompletionTime(String startCompletionTime) {
            this.startCompletionTime = startCompletionTime;
            return this;
        }

        public Builder endCompletionTime(String endCompletionTime) {
            this.endCompletionTime = endCompletionTime;
            return this;
        }

        public Builder rangeType(InstantRange.RangeType rangeType) {
            this.rangeType = rangeType;
            return this;
        }

        public Builder metaClient(HoodieTableMetaClient metaClient) {
            this.metaClient = metaClient;
            return this;
        }

        public Builder skipCompaction(boolean skipCompaction) {
            this.skipCompaction = skipCompaction;
            return this;
        }

        public Builder skipClustering(boolean skipClustering) {
            this.skipClustering = skipClustering;
            return this;
        }

        public Builder skipInsertOverwrite(boolean skipInsertOverwrite) {
            this.skipInsertOverwrite = skipInsertOverwrite;
            return this;
        }

        public Builder readCdcFromChangelog(boolean readCdcFromChangelog) {
            this.readCdcFromChangelog = readCdcFromChangelog;
            return this;
        }

        public Builder limit(int limit) {
            this.limit = limit;
            return this;
        }

        public IncrementalQueryAnalyzer build() {
            return new IncrementalQueryAnalyzer(Objects.requireNonNull(this.metaClient), this.startCompletionTime, this.endCompletionTime, Objects.requireNonNull(this.rangeType), this.skipCompaction, this.skipClustering, this.skipInsertOverwrite, this.readCdcFromChangelog, this.limit);
        }
    }
}

