/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.util;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;
import org.apache.hudi.storage.StoragePathInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileIOUtils {
    public static final Logger LOG = LoggerFactory.getLogger(FileIOUtils.class);
    public static final long KB = 1024L;

    public static void deleteDirectory(File directory) throws IOException {
        if (directory.exists()) {
            Files.walk(directory.toPath(), new FileVisitOption[0]).sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
            directory.delete();
            if (directory.exists()) {
                throw new IOException("Unable to delete directory " + directory);
            }
        }
    }

    public static void mkdir(File directory) throws IOException {
        if (!directory.exists()) {
            directory.mkdirs();
        }
        if (!directory.isDirectory()) {
            throw new IOException("Unable to create :" + directory);
        }
    }

    public static String readAsUTFString(InputStream input) throws IOException {
        return FileIOUtils.readAsUTFString(input, 128);
    }

    public static String readAsUTFString(InputStream input, int length) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream(length);
        FileIOUtils.copy(input, bos);
        return new String(bos.toByteArray(), StandardCharsets.UTF_8);
    }

    public static List<String> readAsUTFStringLines(InputStream input) {
        ArrayList<String> lines = new ArrayList();
        BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(input, StandardCharsets.UTF_8));
        lines = bufferedReader.lines().collect(Collectors.toList());
        FileIOUtils.closeQuietly(bufferedReader);
        return lines;
    }

    public static void copy(InputStream inputStream, OutputStream outputStream) throws IOException {
        int len;
        byte[] buffer = new byte[1024];
        while ((len = inputStream.read(buffer)) != -1) {
            outputStream.write(buffer, 0, len);
        }
    }

    public static void copy(HoodieStorage storage, StoragePath sourceFilePath, StoragePath destFilePath) {
        InputStream inputStream = null;
        OutputStream outputStream = null;
        try {
            inputStream = storage.open(sourceFilePath);
            outputStream = storage.create(destFilePath, false);
            FileIOUtils.copy(inputStream, outputStream);
        }
        catch (IOException e) {
            try {
                throw new HoodieIOException(String.format("Cannot copy from %s to %s", sourceFilePath.toString(), destFilePath.toString()), e);
            }
            catch (Throwable throwable) {
                FileIOUtils.closeQuietly(inputStream);
                FileIOUtils.closeQuietly(outputStream);
                throw throwable;
            }
        }
        FileIOUtils.closeQuietly(inputStream);
        FileIOUtils.closeQuietly(outputStream);
    }

    public static byte[] readAsByteArray(InputStream input) throws IOException {
        return FileIOUtils.readAsByteArray(input, 128);
    }

    public static byte[] readAsByteArray(InputStream input, int outputSize) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream(outputSize);
        FileIOUtils.copy(input, bos);
        return bos.toByteArray();
    }

    public static void writeStringToFile(String str, String filePath) throws IOException {
        try (PrintStream out = new PrintStream(new FileOutputStream(filePath));){
            out.println(str);
            out.flush();
        }
    }

    public static void closeQuietly(Closeable closeable) {
        if (closeable == null) {
            return;
        }
        try {
            closeable.close();
        }
        catch (IOException e) {
            LOG.warn("IOException during close", e);
        }
    }

    public static void createFileInPath(HoodieStorage storage, StoragePath fullPath, Option<byte[]> content, boolean ignoreIOE) {
        block17: {
            try {
                if (!storage.exists(fullPath)) {
                    if (storage.createNewFile(fullPath)) {
                        LOG.info("Created a new file in meta path: " + fullPath);
                    } else {
                        throw new HoodieIOException("Failed to create file " + fullPath);
                    }
                }
                if (!content.isPresent()) break block17;
                try (OutputStream out = storage.create(fullPath, true);){
                    out.write(content.get());
                }
            }
            catch (IOException e) {
                LOG.warn("Failed to create file " + fullPath, e);
                if (ignoreIOE) break block17;
                throw new HoodieIOException("Failed to create file " + fullPath, e);
            }
        }
    }

    public static void createFileInPath(HoodieStorage storage, StoragePath fullPath, Option<byte[]> content) {
        FileIOUtils.createFileInPath(storage, fullPath, content, false);
    }

    public static boolean copy(HoodieStorage srcStorage, StoragePath src, HoodieStorage dstStorage, StoragePath dst, boolean deleteSource, boolean overwrite) throws IOException {
        StoragePathInfo pathInfo = srcStorage.getPathInfo(src);
        return FileIOUtils.copy(srcStorage, pathInfo, dstStorage, dst, deleteSource, overwrite);
    }

    public static boolean copy(HoodieStorage srcStorage, StoragePathInfo srcPathInfo, HoodieStorage dstStorage, StoragePath dst, boolean deleteSource, boolean overwrite) throws IOException {
        StoragePath src = srcPathInfo.getPath();
        if (srcPathInfo.isDirectory()) {
            if (!dstStorage.createDirectory(dst)) {
                return false;
            }
            List<StoragePathInfo> contents = srcStorage.listDirectEntries(src);
            for (StoragePathInfo subPathInfo : contents) {
                FileIOUtils.copy(srcStorage, subPathInfo, dstStorage, new StoragePath(dst, subPathInfo.getPath().getName()), deleteSource, overwrite);
            }
        } else {
            try (InputStream in = srcStorage.open(src);
                 OutputStream out = dstStorage.create(dst, overwrite);){
                FileIOUtils.copy(in, out);
            }
            catch (IOException e) {
                throw new IOException("Error copying source file " + src + " to the destination file " + dst, e);
            }
        }
        if (deleteSource) {
            if (srcPathInfo.isDirectory()) {
                return srcStorage.deleteDirectory(src);
            }
            return srcStorage.deleteFile(src);
        }
        return true;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Option<byte[]> readDataFromPath(HoodieStorage storage, StoragePath detailPath, boolean ignoreIOE) {
        try (InputStream is = storage.open(detailPath);){
            Option<byte[]> option = Option.of(FileIOUtils.readAsByteArray(is));
            return option;
        }
        catch (IOException e) {
            LOG.warn("Could not read commit details from " + detailPath, e);
            if (ignoreIOE) return Option.empty();
            throw new HoodieIOException("Could not read commit details from " + detailPath, e);
        }
    }

    public static Option<byte[]> readDataFromPath(HoodieStorage storage, StoragePath detailPath) {
        return FileIOUtils.readDataFromPath(storage, detailPath, false);
    }

    public static String[] getConfiguredLocalDirs() {
        if (FileIOUtils.isRunningInYarnContainer()) {
            return FileIOUtils.getYarnLocalDirs().split(",");
        }
        if (System.getProperty("java.io.tmpdir") != null) {
            return System.getProperty("java.io.tmpdir").split(",");
        }
        return null;
    }

    private static boolean isRunningInYarnContainer() {
        return System.getenv("CONTAINER_ID") != null && System.getenv("LOCAL_DIRS") != null;
    }

    private static String getYarnLocalDirs() {
        String localDirs = System.getenv("LOCAL_DIRS");
        if (localDirs == null) {
            throw new HoodieIOException("Yarn Local dirs can't be empty");
        }
        return localDirs;
    }

    public static String getDefaultSpillableMapBasePath() {
        String[] localDirs = FileIOUtils.getConfiguredLocalDirs();
        if (localDirs == null) {
            return "/tmp/";
        }
        List<String> localDirLists = Arrays.asList(localDirs);
        Collections.shuffle(localDirLists);
        return !localDirLists.isEmpty() ? localDirLists.get(0) : "/tmp/";
    }
}

