/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.org.apache.hadoop.hbase.io.hfile.bucket;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.util.StringUtils;
import org.apache.hudi.org.apache.hadoop.hbase.exceptions.IllegalArgumentIOException;
import org.apache.hudi.org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hudi.org.apache.hadoop.hbase.io.hfile.bucket.BucketEntry;
import org.apache.hudi.org.apache.hadoop.hbase.io.hfile.bucket.PersistentIOEngine;
import org.apache.hudi.org.apache.hadoop.hbase.nio.ByteBuff;
import org.apache.hudi.org.apache.hbase.thirdparty.com.google.common.base.Preconditions;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class FileIOEngine
extends PersistentIOEngine {
    private static final Logger LOG = LoggerFactory.getLogger(FileIOEngine.class);
    public static final String FILE_DELIMITER = ",";
    private final FileChannel[] fileChannels;
    private final RandomAccessFile[] rafs;
    private final ReentrantLock[] channelLocks;
    private final long sizePerFile;
    private final long capacity;
    private FileReadAccessor readAccessor = new FileReadAccessor();
    private FileWriteAccessor writeAccessor = new FileWriteAccessor();

    public FileIOEngine(long capacity, boolean maintainPersistence, String ... filePaths) throws IOException {
        super(filePaths);
        File file;
        this.sizePerFile = capacity / (long)filePaths.length;
        this.capacity = this.sizePerFile * (long)filePaths.length;
        this.fileChannels = new FileChannel[filePaths.length];
        if (!maintainPersistence) {
            for (String filePath : filePaths) {
                file = new File(filePath);
                if (!file.exists()) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("File " + filePath + " already exists. Deleting!!");
                }
                file.delete();
            }
        }
        this.rafs = new RandomAccessFile[filePaths.length];
        this.channelLocks = new ReentrantLock[filePaths.length];
        for (int i = 0; i < filePaths.length; ++i) {
            String filePath = filePaths[i];
            try {
                this.rafs[i] = new RandomAccessFile(filePath, "rw");
                long totalSpace = new File(filePath).getTotalSpace();
                if (totalSpace < this.sizePerFile) {
                    String msg = "Only " + StringUtils.byteDesc((long)totalSpace) + " total space under " + filePath + ", not enough for requested " + StringUtils.byteDesc((long)this.sizePerFile);
                    LOG.warn(msg);
                }
                if ((file = new File(filePath)).length() != this.sizePerFile) {
                    this.rafs[i].setLength(this.sizePerFile);
                }
                this.fileChannels[i] = this.rafs[i].getChannel();
                this.channelLocks[i] = new ReentrantLock();
                LOG.info("Allocating cache " + StringUtils.byteDesc((long)this.sizePerFile) + ", on the path:" + filePath);
                continue;
            }
            catch (IOException fex) {
                LOG.error("Failed allocating cache on " + filePath, fex);
                this.shutdown();
                throw fex;
            }
        }
    }

    public String toString() {
        return "ioengine=" + this.getClass().getSimpleName() + ", paths=" + Arrays.asList(this.filePaths) + ", capacity=" + String.format("%,d", this.capacity);
    }

    @Override
    public boolean isPersistent() {
        return true;
    }

    @Override
    public Cacheable read(BucketEntry be) throws IOException {
        long offset = be.offset();
        int length = be.getLength();
        Preconditions.checkArgument(length >= 0, "Length of read can not be less than 0.");
        ByteBuff dstBuff = be.allocator.allocate(length);
        if (length != 0) {
            try {
                this.accessFile(this.readAccessor, dstBuff, offset);
                if (dstBuff.limit() != length) {
                    throw new IllegalArgumentIOException("Only " + dstBuff.limit() + " bytes read, " + length + " expected");
                }
            }
            catch (IOException ioe) {
                dstBuff.release();
                throw ioe;
            }
        }
        dstBuff.rewind();
        return be.wrapAsCacheable(dstBuff);
    }

    void closeFileChannels() {
        for (FileChannel fileChannel : this.fileChannels) {
            try {
                fileChannel.close();
            }
            catch (IOException e) {
                LOG.warn("Failed to close FileChannel", e);
            }
        }
    }

    @Override
    public void write(ByteBuffer srcBuffer, long offset) throws IOException {
        this.write(ByteBuff.wrap(srcBuffer), offset);
    }

    @Override
    public void sync() throws IOException {
        for (int i = 0; i < this.fileChannels.length; ++i) {
            try {
                if (this.fileChannels[i] == null) continue;
                this.fileChannels[i].force(true);
                continue;
            }
            catch (IOException ie) {
                LOG.warn("Failed syncing data to " + this.filePaths[i]);
                throw ie;
            }
        }
    }

    @Override
    public void shutdown() {
        for (int i = 0; i < this.filePaths.length; ++i) {
            try {
                if (this.fileChannels[i] != null) {
                    this.fileChannels[i].close();
                }
                if (this.rafs[i] == null) continue;
                this.rafs[i].close();
                continue;
            }
            catch (IOException ex) {
                LOG.error("Failed closing " + this.filePaths[i] + " when shudown the IOEngine", ex);
            }
        }
    }

    @Override
    public void write(ByteBuff srcBuff, long offset) throws IOException {
        if (!srcBuff.hasRemaining()) {
            return;
        }
        this.accessFile(this.writeAccessor, srcBuff, offset);
    }

    private void accessFile(FileAccessor accessor, ByteBuff buff, long globalOffset) throws IOException {
        block5: {
            int startFileNum = this.getFileNum(globalOffset);
            int remainingAccessDataLen = buff.remaining();
            int endFileNum = this.getFileNum(globalOffset + (long)remainingAccessDataLen - 1L);
            int accessFileNum = startFileNum;
            long accessOffset = this.getAbsoluteOffsetInFile(accessFileNum, globalOffset);
            int bufLimit = buff.limit();
            while (true) {
                FileChannel fileChannel = this.fileChannels[accessFileNum];
                int accessLen = 0;
                if (endFileNum > accessFileNum) {
                    buff.limit((int)((long)(buff.limit() - remainingAccessDataLen) + this.sizePerFile - accessOffset));
                }
                try {
                    accessLen = accessor.access(fileChannel, buff, accessOffset);
                }
                catch (ClosedByInterruptException e) {
                    throw e;
                }
                catch (ClosedChannelException e) {
                    this.refreshFileConnection(accessFileNum, e);
                    continue;
                }
                buff.limit(bufLimit);
                if (accessLen >= remainingAccessDataLen) break block5;
                remainingAccessDataLen -= accessLen;
                accessOffset = 0L;
                if (++accessFileNum >= this.fileChannels.length) break;
            }
            throw new IOException("Required data len " + StringUtils.byteDesc((long)buff.remaining()) + " exceed the engine's capacity " + StringUtils.byteDesc((long)this.capacity) + " where offset=" + globalOffset);
        }
    }

    private long getAbsoluteOffsetInFile(int fileNum, long globalOffset) {
        return globalOffset - (long)fileNum * this.sizePerFile;
    }

    private int getFileNum(long offset) {
        if (offset < 0L) {
            throw new IllegalArgumentException("Unexpected offset " + offset);
        }
        int fileNum = (int)(offset / this.sizePerFile);
        if (fileNum >= this.fileChannels.length) {
            throw new RuntimeException("Not expected offset " + offset + " where capacity=" + this.capacity);
        }
        return fileNum;
    }

    FileChannel[] getFileChannels() {
        return this.fileChannels;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void refreshFileConnection(int accessFileNum, IOException ioe) throws IOException {
        ReentrantLock channelLock = this.channelLocks[accessFileNum];
        channelLock.lock();
        try {
            FileChannel fileChannel = this.fileChannels[accessFileNum];
            if (fileChannel != null) {
                if (fileChannel.isOpen()) {
                    return;
                }
                fileChannel.close();
            }
            LOG.warn("Caught ClosedChannelException accessing BucketCache, reopening file: " + this.filePaths[accessFileNum], ioe);
            this.rafs[accessFileNum] = new RandomAccessFile(this.filePaths[accessFileNum], "rw");
            this.fileChannels[accessFileNum] = this.rafs[accessFileNum].getChannel();
        }
        finally {
            channelLock.unlock();
        }
    }

    private static class FileWriteAccessor
    implements FileAccessor {
        private FileWriteAccessor() {
        }

        @Override
        public int access(FileChannel fileChannel, ByteBuff buff, long accessOffset) throws IOException {
            return buff.write(fileChannel, accessOffset);
        }
    }

    private static class FileReadAccessor
    implements FileAccessor {
        private FileReadAccessor() {
        }

        @Override
        public int access(FileChannel fileChannel, ByteBuff buff, long accessOffset) throws IOException {
            return buff.read(fileChannel, accessOffset);
        }
    }

    private static interface FileAccessor {
        public int access(FileChannel var1, ByteBuff var2, long var3) throws IOException;
    }
}

