/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hudi.common.SerializableConfiguration;
import org.apache.hudi.common.io.storage.HoodieWrapperFileSystem;
import org.apache.hudi.common.model.HoodieTableType;
import org.apache.hudi.common.model.TimelineLayoutVersion;
import org.apache.hudi.common.table.HoodieTableConfig;
import org.apache.hudi.common.table.HoodieTimeline;
import org.apache.hudi.common.table.TimelineLayout;
import org.apache.hudi.common.table.timeline.HoodieActiveTimeline;
import org.apache.hudi.common.table.timeline.HoodieArchivedTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.util.ConsistencyGuardConfig;
import org.apache.hudi.common.util.FSUtils;
import org.apache.hudi.common.util.FailSafeConsistencyGuard;
import org.apache.hudi.common.util.NoOpConsistencyGuard;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.TableNotFoundException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class HoodieTableMetaClient
implements Serializable {
    private static final Logger LOG = LogManager.getLogger(HoodieTableMetaClient.class);
    public static String METAFOLDER_NAME = ".hoodie";
    public static String TEMPFOLDER_NAME = METAFOLDER_NAME + File.separator + ".temp";
    public static String AUXILIARYFOLDER_NAME = METAFOLDER_NAME + File.separator + ".aux";
    public static final String MARKER_EXTN = ".marker";
    private String basePath;
    private transient HoodieWrapperFileSystem fs;
    private String metaPath;
    private boolean loadActiveTimelineOnLoad;
    private SerializableConfiguration hadoopConf;
    private HoodieTableType tableType;
    private TimelineLayoutVersion timelineLayoutVersion;
    private HoodieTableConfig tableConfig;
    private HoodieActiveTimeline activeTimeline;
    private HoodieArchivedTimeline archivedTimeline;
    private ConsistencyGuardConfig consistencyGuardConfig = ConsistencyGuardConfig.newBuilder().build();

    public HoodieTableMetaClient(Configuration conf, String basePath) throws TableNotFoundException {
        this(conf, basePath, false);
    }

    public HoodieTableMetaClient(Configuration conf, String basePath, String payloadClassName) {
        this(conf, basePath, false, ConsistencyGuardConfig.newBuilder().build(), Option.of(TimelineLayoutVersion.CURR_LAYOUT_VERSION), payloadClassName);
    }

    public HoodieTableMetaClient(Configuration conf, String basePath, boolean loadActiveTimelineOnLoad, ConsistencyGuardConfig consistencyGuardConfig, Option<TimelineLayoutVersion> layoutVersion) {
        this(conf, basePath, loadActiveTimelineOnLoad, consistencyGuardConfig, layoutVersion, null);
    }

    public HoodieTableMetaClient(Configuration conf, String basePath, boolean loadActiveTimelineOnLoad) {
        this(conf, basePath, loadActiveTimelineOnLoad, ConsistencyGuardConfig.newBuilder().build(), Option.of(TimelineLayoutVersion.CURR_LAYOUT_VERSION), null);
    }

    public HoodieTableMetaClient(Configuration conf, String basePath, boolean loadActiveTimelineOnLoad, ConsistencyGuardConfig consistencyGuardConfig, Option<TimelineLayoutVersion> layoutVersion, String payloadClassName) throws TableNotFoundException {
        LOG.info((Object)("Loading HoodieTableMetaClient from " + basePath));
        this.basePath = basePath;
        this.consistencyGuardConfig = consistencyGuardConfig;
        this.hadoopConf = new SerializableConfiguration(conf);
        Path basePathDir = new Path(this.basePath);
        this.metaPath = new Path(basePath, METAFOLDER_NAME).toString();
        Path metaPathDir = new Path(this.metaPath);
        this.fs = this.getFs();
        TableNotFoundException.checkTableValidity(this.fs, basePathDir, metaPathDir);
        this.tableConfig = new HoodieTableConfig(this.fs, this.metaPath, payloadClassName);
        this.tableType = this.tableConfig.getTableType();
        Option<TimelineLayoutVersion> tableConfigVersion = this.tableConfig.getTimelineLayoutVersion();
        if (layoutVersion.isPresent() && tableConfigVersion.isPresent()) {
            Preconditions.checkArgument((layoutVersion.get().compareTo(tableConfigVersion.get()) >= 0 ? 1 : 0) != 0, (Object)("Layout Version defined in hoodie properties has higher version (" + tableConfigVersion.get() + ") than the one passed in config (" + layoutVersion.get() + ")"));
        }
        this.timelineLayoutVersion = layoutVersion.orElseGet(() -> this.tableConfig.getTimelineLayoutVersion().get());
        this.loadActiveTimelineOnLoad = loadActiveTimelineOnLoad;
        LOG.info((Object)("Finished Loading Table of type " + (Object)((Object)this.tableType) + "(version=" + this.timelineLayoutVersion + ") from " + basePath));
        if (loadActiveTimelineOnLoad) {
            LOG.info((Object)("Loading Active commit timeline for " + basePath));
            this.getActiveTimeline();
        }
    }

    public HoodieTableMetaClient() {
    }

    public static HoodieTableMetaClient reload(HoodieTableMetaClient oldMetaClient) {
        return new HoodieTableMetaClient(oldMetaClient.hadoopConf.get(), oldMetaClient.basePath, oldMetaClient.loadActiveTimelineOnLoad, oldMetaClient.consistencyGuardConfig, Option.of(oldMetaClient.timelineLayoutVersion), null);
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        this.fs = null;
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();
    }

    public String getBasePath() {
        return this.basePath;
    }

    public HoodieTableType getTableType() {
        return this.tableType;
    }

    public String getMetaPath() {
        return this.metaPath;
    }

    public String getTempFolderPath() {
        return this.basePath + File.separator + TEMPFOLDER_NAME;
    }

    public String getMarkerFolderPath(String instantTs) {
        return String.format("%s%s%s", this.getTempFolderPath(), File.separator, instantTs);
    }

    public String getMetaAuxiliaryPath() {
        return this.basePath + File.separator + AUXILIARYFOLDER_NAME;
    }

    public String getArchivePath() {
        String archiveFolder = this.tableConfig.getArchivelogFolder();
        if (archiveFolder.equals("")) {
            return this.getMetaPath();
        }
        return this.getMetaPath() + "/" + archiveFolder;
    }

    public HoodieTableConfig getTableConfig() {
        return this.tableConfig;
    }

    public TimelineLayoutVersion getTimelineLayoutVersion() {
        return this.timelineLayoutVersion;
    }

    public HoodieWrapperFileSystem getFs() {
        if (this.fs == null) {
            FileSystem fileSystem = FSUtils.getFs(this.metaPath, this.hadoopConf.newCopy());
            Preconditions.checkArgument((!(fileSystem instanceof HoodieWrapperFileSystem) ? 1 : 0) != 0, (Object)"File System not expected to be that of HoodieWrapperFileSystem");
            this.fs = new HoodieWrapperFileSystem(fileSystem, this.consistencyGuardConfig.isConsistencyCheckEnabled() ? new FailSafeConsistencyGuard(fileSystem, this.consistencyGuardConfig) : new NoOpConsistencyGuard());
        }
        return this.fs;
    }

    public FileSystem getRawFs() {
        return this.getFs().getFileSystem();
    }

    public Configuration getHadoopConf() {
        return this.hadoopConf.get();
    }

    public synchronized HoodieActiveTimeline getActiveTimeline() {
        if (this.activeTimeline == null) {
            this.activeTimeline = new HoodieActiveTimeline(this);
        }
        return this.activeTimeline;
    }

    public synchronized HoodieActiveTimeline reloadActiveTimeline() {
        this.activeTimeline = new HoodieActiveTimeline(this);
        return this.activeTimeline;
    }

    public ConsistencyGuardConfig getConsistencyGuardConfig() {
        return this.consistencyGuardConfig;
    }

    public synchronized HoodieArchivedTimeline getArchivedTimeline() {
        if (this.archivedTimeline == null) {
            this.archivedTimeline = new HoodieArchivedTimeline(this);
        }
        return this.archivedTimeline;
    }

    public static HoodieTableMetaClient initTableType(Configuration hadoopConf, String basePath, String tableType, String tableName, String archiveLogFolder, String payloadClassName) throws IOException {
        return HoodieTableMetaClient.initTableType(hadoopConf, basePath, HoodieTableType.valueOf(tableType), tableName, archiveLogFolder, payloadClassName, null);
    }

    public static HoodieTableMetaClient initTableType(Configuration hadoopConf, String basePath, HoodieTableType tableType, String tableName, String payloadClassName) throws IOException {
        return HoodieTableMetaClient.initTableType(hadoopConf, basePath, tableType, tableName, null, payloadClassName, null);
    }

    public static HoodieTableMetaClient initTableType(Configuration hadoopConf, String basePath, HoodieTableType tableType, String tableName, String archiveLogFolder, String payloadClassName, Integer timelineLayoutVersion) throws IOException {
        Properties properties = new Properties();
        properties.setProperty("hoodie.table.name", tableName);
        properties.setProperty("hoodie.table.type", tableType.name());
        if (tableType == HoodieTableType.MERGE_ON_READ && payloadClassName != null) {
            properties.setProperty("hoodie.compaction.payload.class", payloadClassName);
        }
        if (null != archiveLogFolder) {
            properties.put("hoodie.archivelog.folder", archiveLogFolder);
        }
        if (null != timelineLayoutVersion) {
            properties.put("hoodie.timeline.layout.version", String.valueOf(timelineLayoutVersion));
        }
        return HoodieTableMetaClient.initTableAndGetMetaClient(hadoopConf, basePath, properties);
    }

    public static HoodieTableMetaClient initTableAndGetMetaClient(Configuration hadoopConf, String basePath, Properties props) throws IOException {
        Path auxiliaryFolder;
        Path temporaryFolder;
        Path archiveLogDir;
        String archiveLogPropVal;
        Path metaPathDir;
        LOG.info((Object)("Initializing " + basePath + " as hoodie table " + basePath));
        Path basePathDir = new Path(basePath);
        FileSystem fs = FSUtils.getFs(basePath, hadoopConf);
        if (!fs.exists(basePathDir)) {
            fs.mkdirs(basePathDir);
        }
        if (!fs.exists(metaPathDir = new Path(basePath, METAFOLDER_NAME))) {
            fs.mkdirs(metaPathDir);
        }
        if (!(archiveLogPropVal = props.getProperty("hoodie.archivelog.folder", "")).equals("") && !fs.exists(archiveLogDir = new Path(metaPathDir, archiveLogPropVal))) {
            fs.mkdirs(archiveLogDir);
        }
        if (!fs.exists(temporaryFolder = new Path(basePath, TEMPFOLDER_NAME))) {
            fs.mkdirs(temporaryFolder);
        }
        if (!fs.exists(auxiliaryFolder = new Path(basePath, AUXILIARYFOLDER_NAME))) {
            fs.mkdirs(auxiliaryFolder);
        }
        HoodieTableConfig.createHoodieProperties(fs, metaPathDir, props);
        HoodieTableMetaClient metaClient = new HoodieTableMetaClient(hadoopConf, basePath);
        LOG.info((Object)("Finished initializing Table of type " + (Object)((Object)metaClient.getTableConfig().getTableType()) + " from " + basePath));
        return metaClient;
    }

    public static FileStatus[] scanFiles(FileSystem fs, Path metaPath, PathFilter nameFilter) throws IOException {
        return fs.listStatus(metaPath, nameFilter);
    }

    public HoodieTimeline getCommitsTimeline() {
        switch (this.getTableType()) {
            case COPY_ON_WRITE: {
                return this.getActiveTimeline().getCommitTimeline();
            }
            case MERGE_ON_READ: {
                return this.getActiveTimeline().getCommitsTimeline();
            }
        }
        throw new HoodieException("Unsupported table type :" + (Object)((Object)this.getTableType()));
    }

    public HoodieTimeline getCommitsAndCompactionTimeline() {
        switch (this.getTableType()) {
            case COPY_ON_WRITE: {
                return this.getActiveTimeline().getCommitTimeline();
            }
            case MERGE_ON_READ: {
                return this.getActiveTimeline().getCommitsAndCompactionTimeline();
            }
        }
        throw new HoodieException("Unsupported table type :" + (Object)((Object)this.getTableType()));
    }

    public HoodieTimeline getCommitTimeline() {
        switch (this.getTableType()) {
            case COPY_ON_WRITE: 
            case MERGE_ON_READ: {
                return this.getActiveTimeline().getCommitTimeline();
            }
        }
        throw new HoodieException("Unsupported table type :" + (Object)((Object)this.getTableType()));
    }

    public String getCommitActionType() {
        switch (this.getTableType()) {
            case COPY_ON_WRITE: {
                return "commit";
            }
            case MERGE_ON_READ: {
                return "deltacommit";
            }
        }
        throw new HoodieException("Could not commit on unknown table type " + (Object)((Object)this.getTableType()));
    }

    public List<HoodieInstant> scanHoodieInstantsFromFileSystem(Set<String> includedExtensions, boolean applyLayoutVersionFilters) throws IOException {
        return this.scanHoodieInstantsFromFileSystem(new Path(this.metaPath), includedExtensions, applyLayoutVersionFilters);
    }

    public List<HoodieInstant> scanHoodieInstantsFromFileSystem(Path timelinePath, Set<String> includedExtensions, boolean applyLayoutVersionFilters) throws IOException {
        Stream<HoodieInstant> instantStream = Arrays.stream(HoodieTableMetaClient.scanFiles(this.getFs(), timelinePath, path -> {
            String extension = FSUtils.getFileExtension(path.getName());
            return includedExtensions.contains(extension);
        })).map(HoodieInstant::new);
        if (applyLayoutVersionFilters) {
            instantStream = TimelineLayout.getLayout(this.getTimelineLayoutVersion()).filterHoodieInstants(instantStream);
        }
        return instantStream.sorted().collect(Collectors.toList());
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        HoodieTableMetaClient that = (HoodieTableMetaClient)o;
        return Objects.equals(this.basePath, that.basePath) && this.tableType == that.tableType;
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.basePath, this.tableType});
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("HoodieTableMetaClient{");
        sb.append("basePath='").append(this.basePath).append('\'');
        sb.append(", metaPath='").append(this.metaPath).append('\'');
        sb.append(", tableType=").append((Object)this.tableType);
        sb.append('}');
        return sb.toString();
    }

    public void setBasePath(String basePath) {
        this.basePath = basePath;
    }

    public void setMetaPath(String metaPath) {
        this.metaPath = metaPath;
    }

    public void setActiveTimeline(HoodieActiveTimeline activeTimeline) {
        this.activeTimeline = activeTimeline;
    }

    public void setTableConfig(HoodieTableConfig tableConfig) {
        this.tableConfig = tableConfig;
    }
}

