/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.hive;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.IMetaStoreClient;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.api.Partition;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.hadoop.hive.ql.Driver;
import org.apache.hadoop.hive.ql.metadata.Hive;
import org.apache.hadoop.hive.ql.metadata.HiveException;
import org.apache.hadoop.hive.ql.processors.CommandProcessorResponse;
import org.apache.hadoop.hive.ql.session.SessionState;
import org.apache.hive.jdbc.HiveDriver;
import org.apache.hudi.common.model.HoodieCommitMetadata;
import org.apache.hudi.common.model.HoodieFileFormat;
import org.apache.hudi.common.model.HoodieTableType;
import org.apache.hudi.common.storage.StorageSchemes;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.HoodieTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.util.FSUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.InvalidTableException;
import org.apache.hudi.hive.HiveSyncConfig;
import org.apache.hudi.hive.HoodieHiveSyncException;
import org.apache.hudi.hive.PartitionValueExtractor;
import org.apache.hudi.hive.util.SchemaUtil;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.parquet.format.converter.ParquetMetadataConverter;
import org.apache.parquet.hadoop.ParquetFileReader;
import org.apache.parquet.hadoop.metadata.ParquetMetadata;
import org.apache.parquet.schema.MessageType;
import org.apache.thrift.TException;

public class HoodieHiveClient {
    private static final String HOODIE_LAST_COMMIT_TIME_SYNC = "last_commit_time_sync";
    private static String driverName = HiveDriver.class.getName();
    private static final String HIVE_ESCAPE_CHARACTER = "`";
    private static final Logger LOG;
    private final HoodieTableMetaClient metaClient;
    private final HoodieTableType tableType;
    private final PartitionValueExtractor partitionValueExtractor;
    private IMetaStoreClient client;
    private HiveSyncConfig syncConfig;
    private FileSystem fs;
    private Connection connection;
    private HoodieTimeline activeTimeline;
    private HiveConf configuration;

    public HoodieHiveClient(HiveSyncConfig cfg, HiveConf configuration, FileSystem fs) {
        this.syncConfig = cfg;
        this.fs = fs;
        this.metaClient = new HoodieTableMetaClient(fs.getConf(), cfg.basePath, true);
        this.tableType = this.metaClient.getTableType();
        this.configuration = configuration;
        if (cfg.useJdbc.booleanValue()) {
            LOG.info((Object)("Creating hive connection " + cfg.jdbcUrl));
            this.createHiveConnection();
        }
        try {
            this.client = Hive.get((HiveConf)configuration).getMSC();
        }
        catch (MetaException | HiveException e) {
            throw new HoodieHiveSyncException("Failed to create HiveMetaStoreClient", e);
        }
        try {
            this.partitionValueExtractor = (PartitionValueExtractor)Class.forName(cfg.partitionValueExtractorClass).newInstance();
        }
        catch (Exception e) {
            throw new HoodieHiveSyncException("Failed to initialize PartitionValueExtractor class " + cfg.partitionValueExtractorClass, e);
        }
        this.activeTimeline = this.metaClient.getActiveTimeline().getCommitsTimeline().filterCompletedInstants();
    }

    public HoodieTimeline getActiveTimeline() {
        return this.activeTimeline;
    }

    void addPartitionsToTable(String tableName, List<String> partitionsToAdd) {
        if (partitionsToAdd.isEmpty()) {
            LOG.info((Object)("No partitions to add for " + tableName));
            return;
        }
        LOG.info((Object)("Adding partitions " + partitionsToAdd.size() + " to table " + tableName));
        String sql = this.constructAddPartitions(tableName, partitionsToAdd);
        this.updateHiveSQL(sql);
    }

    void updatePartitionsToTable(String tableName, List<String> changedPartitions) {
        if (changedPartitions.isEmpty()) {
            LOG.info((Object)("No partitions to change for " + tableName));
            return;
        }
        LOG.info((Object)("Changing partitions " + changedPartitions.size() + " on " + tableName));
        List<String> sqls = this.constructChangePartitions(tableName, changedPartitions);
        for (String sql : sqls) {
            this.updateHiveSQL(sql);
        }
    }

    private String constructAddPartitions(String tableName, List<String> partitions) {
        StringBuilder alterSQL = new StringBuilder("ALTER TABLE ");
        alterSQL.append(HIVE_ESCAPE_CHARACTER).append(this.syncConfig.databaseName).append(HIVE_ESCAPE_CHARACTER).append(".").append(HIVE_ESCAPE_CHARACTER).append(tableName).append(HIVE_ESCAPE_CHARACTER).append(" ADD IF NOT EXISTS ");
        for (String partition : partitions) {
            String partitionClause = this.getPartitionClause(partition);
            String fullPartitionPath = FSUtils.getPartitionPath(this.syncConfig.basePath, partition).toString();
            alterSQL.append("  PARTITION (").append(partitionClause).append(") LOCATION '").append(fullPartitionPath).append("' ");
        }
        return alterSQL.toString();
    }

    private String getPartitionClause(String partition) {
        List<String> partitionValues = this.partitionValueExtractor.extractPartitionValuesInPath(partition);
        Preconditions.checkArgument((this.syncConfig.partitionFields.size() == partitionValues.size() ? 1 : 0) != 0, (Object)("Partition key parts " + this.syncConfig.partitionFields + " does not match with partition values " + partitionValues + ". Check partition strategy. "));
        ArrayList<String> partBuilder = new ArrayList<String>();
        for (int i = 0; i < this.syncConfig.partitionFields.size(); ++i) {
            partBuilder.add(HIVE_ESCAPE_CHARACTER + this.syncConfig.partitionFields.get(i) + "`='" + partitionValues.get(i) + "'");
        }
        return partBuilder.stream().collect(Collectors.joining(","));
    }

    private List<String> constructChangePartitions(String tableName, List<String> partitions) {
        ArrayList changePartitions = Lists.newArrayList();
        String useDatabase = "USE `" + this.syncConfig.databaseName + HIVE_ESCAPE_CHARACTER;
        changePartitions.add(useDatabase);
        String alterTable = "ALTER TABLE `" + tableName + HIVE_ESCAPE_CHARACTER;
        for (String partition : partitions) {
            String partitionClause = this.getPartitionClause(partition);
            Path partitionPath = FSUtils.getPartitionPath(this.syncConfig.basePath, partition);
            String fullPartitionPath = partitionPath.toUri().getScheme().equals(StorageSchemes.HDFS.getScheme()) ? FSUtils.getDFSFullPartitionPath(this.fs, partitionPath) : partitionPath.toString();
            String changePartition = alterTable + " PARTITION (" + partitionClause + ") SET LOCATION '" + fullPartitionPath + "'";
            changePartitions.add(changePartition);
        }
        return changePartitions;
    }

    List<PartitionEvent> getPartitionEvents(List<Partition> tablePartitions, List<String> partitionStoragePartitions) {
        HashMap paths = Maps.newHashMap();
        for (Partition tablePartition : tablePartitions) {
            List hivePartitionValues = tablePartition.getValues();
            Collections.sort(hivePartitionValues);
            String fullTablePartitionPath = Path.getPathWithoutSchemeAndAuthority((Path)new Path(tablePartition.getSd().getLocation())).toUri().getPath();
            paths.put(String.join((CharSequence)", ", hivePartitionValues), fullTablePartitionPath);
        }
        ArrayList events = Lists.newArrayList();
        for (String storagePartition : partitionStoragePartitions) {
            Path storagePartitionPath = FSUtils.getPartitionPath(this.syncConfig.basePath, storagePartition);
            String fullStoragePartitionPath = Path.getPathWithoutSchemeAndAuthority((Path)storagePartitionPath).toUri().getPath();
            List<String> storagePartitionValues = this.partitionValueExtractor.extractPartitionValuesInPath(storagePartition);
            Collections.sort(storagePartitionValues);
            if (storagePartitionValues.isEmpty()) continue;
            String storageValue = String.join((CharSequence)", ", storagePartitionValues);
            if (!paths.containsKey(storageValue)) {
                events.add(PartitionEvent.newPartitionAddEvent(storagePartition));
                continue;
            }
            if (((String)paths.get(storageValue)).equals(fullStoragePartitionPath)) continue;
            events.add(PartitionEvent.newPartitionUpdateEvent(storagePartition));
        }
        return events;
    }

    public List<Partition> scanTablePartitions(String tableName) throws TException {
        return this.client.listPartitions(this.syncConfig.databaseName, tableName, (short)-1);
    }

    void updateTableDefinition(String tableName, MessageType newSchema) {
        try {
            String newSchemaStr = SchemaUtil.generateSchemaString(newSchema, this.syncConfig.partitionFields);
            String cascadeClause = this.syncConfig.partitionFields.size() > 0 ? " cascade" : "";
            StringBuilder sqlBuilder = new StringBuilder("ALTER TABLE ").append(HIVE_ESCAPE_CHARACTER).append(this.syncConfig.databaseName).append(HIVE_ESCAPE_CHARACTER).append(".").append(HIVE_ESCAPE_CHARACTER).append(tableName).append(HIVE_ESCAPE_CHARACTER).append(" REPLACE COLUMNS(").append(newSchemaStr).append(" )").append(cascadeClause);
            LOG.info((Object)("Updating table definition with " + sqlBuilder));
            this.updateHiveSQL(sqlBuilder.toString());
        }
        catch (IOException e) {
            throw new HoodieHiveSyncException("Failed to update table for " + tableName, e);
        }
    }

    void createTable(String tableName, MessageType storageSchema, String inputFormatClass, String outputFormatClass, String serdeClass) {
        try {
            String createSQLQuery = SchemaUtil.generateCreateDDL(tableName, storageSchema, this.syncConfig, inputFormatClass, outputFormatClass, serdeClass);
            LOG.info((Object)("Creating table with " + createSQLQuery));
            this.updateHiveSQL(createSQLQuery);
        }
        catch (IOException e) {
            throw new HoodieHiveSyncException("Failed to create table " + tableName, e);
        }
    }

    public Map<String, String> getTableSchema(String tableName) {
        if (this.syncConfig.useJdbc.booleanValue()) {
            HashMap hashMap;
            if (!this.doesTableExist(tableName)) {
                throw new IllegalArgumentException("Failed to get schema for table " + tableName + " does not exist");
            }
            HashMap schema = Maps.newHashMap();
            ResultSet result = null;
            try {
                DatabaseMetaData databaseMetaData = this.connection.getMetaData();
                result = databaseMetaData.getColumns(null, this.syncConfig.databaseName, tableName, null);
                while (result.next()) {
                    String columnName = result.getString(4);
                    String columnType = result.getString(6);
                    if ("DECIMAL".equals(columnType)) {
                        int columnSize = result.getInt("COLUMN_SIZE");
                        int decimalDigits = result.getInt("DECIMAL_DIGITS");
                        columnType = columnType + String.format("(%s,%s)", columnSize, decimalDigits);
                    }
                    schema.put(columnName, columnType);
                }
                hashMap = schema;
            }
            catch (SQLException e) {
                try {
                    throw new HoodieHiveSyncException("Failed to get table schema for " + tableName, e);
                }
                catch (Throwable throwable) {
                    HoodieHiveClient.closeQuietly(result, null);
                    throw throwable;
                }
            }
            HoodieHiveClient.closeQuietly(result, null);
            return hashMap;
        }
        return this.getTableSchemaUsingMetastoreClient(tableName);
    }

    public Map<String, String> getTableSchemaUsingMetastoreClient(String tableName) {
        try {
            long start = System.currentTimeMillis();
            Table table = this.client.getTable(this.syncConfig.databaseName, tableName);
            Map<String, String> partitionKeysMap = table.getPartitionKeys().stream().collect(Collectors.toMap(FieldSchema::getName, f -> f.getType().toUpperCase()));
            Map<String, String> columnsMap = table.getSd().getCols().stream().collect(Collectors.toMap(FieldSchema::getName, f -> f.getType().toUpperCase()));
            HashMap<String, String> schema = new HashMap<String, String>();
            schema.putAll(columnsMap);
            schema.putAll(partitionKeysMap);
            long end = System.currentTimeMillis();
            LOG.info((Object)String.format("Time taken to getTableSchema: %s ms", end - start));
            return schema;
        }
        catch (Exception e) {
            throw new HoodieHiveSyncException("Failed to get table schema for : " + tableName, e);
        }
    }

    public MessageType getDataSchema() {
        try {
            switch (this.tableType) {
                case COPY_ON_WRITE: {
                    HoodieInstant lastCommit = this.activeTimeline.lastInstant().orElseThrow(() -> new InvalidTableException(this.syncConfig.basePath));
                    HoodieCommitMetadata commitMetadata = HoodieCommitMetadata.fromBytes(this.activeTimeline.getInstantDetails(lastCommit).get(), HoodieCommitMetadata.class);
                    String filePath = commitMetadata.getFileIdAndFullPaths(this.metaClient.getBasePath()).values().stream().findAny().orElseThrow(() -> new IllegalArgumentException("Could not find any data file written for commit " + lastCommit + ", could not get schema for table " + this.metaClient.getBasePath() + ", Metadata :" + commitMetadata));
                    return this.readSchemaFromBaseFile(new Path(filePath));
                }
                case MERGE_ON_READ: {
                    Option<HoodieInstant> lastCompactionCommit = this.metaClient.getActiveTimeline().getCommitTimeline().filterCompletedInstants().lastInstant();
                    LOG.info((Object)("Found the last compaction commit as " + lastCompactionCommit));
                    Option<HoodieInstant> lastDeltaCommit = lastCompactionCommit.isPresent() ? this.metaClient.getActiveTimeline().getDeltaCommitTimeline().filterCompletedInstants().findInstantsAfter(lastCompactionCommit.get().getTimestamp(), Integer.MAX_VALUE).lastInstant() : this.metaClient.getActiveTimeline().getDeltaCommitTimeline().filterCompletedInstants().lastInstant();
                    LOG.info((Object)("Found the last delta commit " + lastDeltaCommit));
                    if (lastDeltaCommit.isPresent()) {
                        HoodieInstant lastDeltaInstant = lastDeltaCommit.get();
                        HoodieCommitMetadata commitMetadata = HoodieCommitMetadata.fromBytes(this.activeTimeline.getInstantDetails(lastDeltaInstant).get(), HoodieCommitMetadata.class);
                        Pair filePathWithFormat = commitMetadata.getFileIdAndFullPaths(this.metaClient.getBasePath()).values().stream().filter(s -> s.contains(".log")).findAny().map(f -> Pair.of(f, HoodieFileFormat.HOODIE_LOG)).orElseGet(() -> commitMetadata.getFileIdAndFullPaths(this.metaClient.getBasePath()).values().stream().filter(s -> s.contains(this.metaClient.getTableConfig().getBaseFileFormat().getFileExtension())).findAny().map(f -> Pair.of(f, HoodieFileFormat.PARQUET)).orElseThrow(() -> new IllegalArgumentException("Could not find any data file written for commit " + lastDeltaInstant + ", could not get schema for table " + this.metaClient.getBasePath() + ", CommitMetadata :" + commitMetadata)));
                        switch ((HoodieFileFormat)((Object)filePathWithFormat.getRight())) {
                            case HOODIE_LOG: {
                                return this.readSchemaFromLogFile(lastCompactionCommit, new Path((String)filePathWithFormat.getLeft()));
                            }
                            case PARQUET: {
                                return this.readSchemaFromBaseFile(new Path((String)filePathWithFormat.getLeft()));
                            }
                        }
                        throw new IllegalArgumentException("Unknown file format :" + filePathWithFormat.getRight() + " for file " + (String)filePathWithFormat.getLeft());
                    }
                    return this.readSchemaFromLastCompaction(lastCompactionCommit);
                }
            }
            LOG.error((Object)("Unknown table type " + (Object)((Object)this.tableType)));
            throw new InvalidTableException(this.syncConfig.basePath);
        }
        catch (IOException e) {
            throw new HoodieHiveSyncException("Failed to read data schema", e);
        }
    }

    private MessageType readSchemaFromLastCompaction(Option<HoodieInstant> lastCompactionCommitOpt) throws IOException {
        HoodieInstant lastCompactionCommit = lastCompactionCommitOpt.orElseThrow(() -> new HoodieHiveSyncException("Could not read schema from last compaction, no compaction commits found on path " + this.syncConfig.basePath));
        HoodieCommitMetadata compactionMetadata = HoodieCommitMetadata.fromBytes(this.activeTimeline.getInstantDetails(lastCompactionCommit).get(), HoodieCommitMetadata.class);
        String filePath = compactionMetadata.getFileIdAndFullPaths(this.metaClient.getBasePath()).values().stream().findAny().orElseThrow(() -> new IllegalArgumentException("Could not find any data file written for compaction " + lastCompactionCommit + ", could not get schema for table " + this.metaClient.getBasePath()));
        return this.readSchemaFromBaseFile(new Path(filePath));
    }

    private MessageType readSchemaFromLogFile(Option<HoodieInstant> lastCompactionCommitOpt, Path path) throws IOException {
        MessageType messageType = SchemaUtil.readSchemaFromLogFile(this.fs, path);
        if (messageType == null) {
            LOG.info((Object)("Falling back to read the schema from last compaction " + lastCompactionCommitOpt));
            return this.readSchemaFromLastCompaction(lastCompactionCommitOpt);
        }
        return messageType;
    }

    private MessageType readSchemaFromBaseFile(Path parquetFilePath) throws IOException {
        LOG.info((Object)("Reading schema from " + parquetFilePath));
        if (!this.fs.exists(parquetFilePath)) {
            throw new IllegalArgumentException("Failed to read schema from data file " + parquetFilePath + ". File does not exist.");
        }
        ParquetMetadata fileFooter = ParquetFileReader.readFooter((Configuration)this.fs.getConf(), (Path)parquetFilePath, (ParquetMetadataConverter.MetadataFilter)ParquetMetadataConverter.NO_FILTER);
        return fileFooter.getFileMetaData().getSchema();
    }

    public boolean doesTableExist(String tableName) {
        try {
            return this.client.tableExists(this.syncConfig.databaseName, tableName);
        }
        catch (TException e) {
            throw new HoodieHiveSyncException("Failed to check if table exists " + tableName, e);
        }
    }

    public void updateHiveSQL(String s) {
        if (this.syncConfig.useJdbc.booleanValue()) {
            Statement stmt = null;
            try {
                stmt = this.connection.createStatement();
                LOG.info((Object)("Executing SQL " + s));
                stmt.execute(s);
            }
            catch (SQLException e) {
                throw new HoodieHiveSyncException("Failed in executing SQL " + s, e);
            }
            finally {
                HoodieHiveClient.closeQuietly(null, stmt);
            }
        } else {
            this.updateHiveSQLUsingHiveDriver(s);
        }
    }

    public CommandProcessorResponse updateHiveSQLUsingHiveDriver(String sql) throws HoodieHiveSyncException {
        List<CommandProcessorResponse> responses = this.updateHiveSQLs(Arrays.asList(sql));
        return responses.get(responses.size() - 1);
    }

    private List<CommandProcessorResponse> updateHiveSQLs(List<String> sqls) throws HoodieHiveSyncException {
        SessionState ss = null;
        Driver hiveDriver = null;
        ArrayList<CommandProcessorResponse> responses = new ArrayList<CommandProcessorResponse>();
        try {
            long startTime = System.currentTimeMillis();
            ss = SessionState.start((HiveConf)this.configuration);
            hiveDriver = new Driver(this.configuration);
            long endTime = System.currentTimeMillis();
            LOG.info((Object)String.format("Time taken to start SessionState and create Driver: %s ms", endTime - startTime));
            for (String sql : sqls) {
                long start = System.currentTimeMillis();
                responses.add(hiveDriver.run(sql));
                long end = System.currentTimeMillis();
                LOG.info((Object)String.format("Time taken to execute [%s]: %s ms", sql, end - start));
            }
        }
        catch (Exception e) {
            throw new HoodieHiveSyncException("Failed in executing SQL", e);
        }
        finally {
            if (ss != null) {
                try {
                    ss.close();
                }
                catch (IOException ie) {
                    LOG.error((Object)"Error while closing SessionState", (Throwable)ie);
                }
            }
            if (hiveDriver != null) {
                try {
                    hiveDriver.close();
                }
                catch (Exception e) {
                    LOG.error((Object)"Error while closing hiveDriver", (Throwable)e);
                }
            }
        }
        return responses;
    }

    private void createHiveConnection() {
        if (this.connection == null) {
            try {
                Class.forName(HiveDriver.class.getCanonicalName());
            }
            catch (ClassNotFoundException e) {
                LOG.error((Object)"Unable to load Hive driver class", (Throwable)e);
                return;
            }
            try {
                this.connection = DriverManager.getConnection(this.syncConfig.jdbcUrl, this.syncConfig.hiveUser, this.syncConfig.hivePass);
                LOG.info((Object)("Successfully established Hive connection to  " + this.syncConfig.jdbcUrl));
            }
            catch (SQLException e) {
                throw new HoodieHiveSyncException("Cannot create hive connection " + this.getHiveJdbcUrlWithDefaultDBName(), e);
            }
        }
    }

    private String getHiveJdbcUrlWithDefaultDBName() {
        String hiveJdbcUrl = this.syncConfig.jdbcUrl;
        String urlAppend = null;
        if (hiveJdbcUrl.contains(";")) {
            urlAppend = hiveJdbcUrl.substring(hiveJdbcUrl.indexOf(";"));
            hiveJdbcUrl = hiveJdbcUrl.substring(0, hiveJdbcUrl.indexOf(";"));
        }
        if (!hiveJdbcUrl.endsWith("/")) {
            hiveJdbcUrl = hiveJdbcUrl + "/";
        }
        return hiveJdbcUrl + (urlAppend == null ? "" : urlAppend);
    }

    private static void closeQuietly(ResultSet resultSet, Statement stmt) {
        try {
            if (stmt != null) {
                stmt.close();
            }
        }
        catch (SQLException e) {
            LOG.error((Object)"Could not close the statement opened ", (Throwable)e);
        }
        try {
            if (resultSet != null) {
                resultSet.close();
            }
        }
        catch (SQLException e) {
            LOG.error((Object)"Could not close the resultset opened ", (Throwable)e);
        }
    }

    public String getBasePath() {
        return this.metaClient.getBasePath();
    }

    HoodieTableType getTableType() {
        return this.tableType;
    }

    public FileSystem getFs() {
        return this.fs;
    }

    public Option<String> getLastCommitTimeSynced(String tableName) {
        try {
            Table database = this.client.getTable(this.syncConfig.databaseName, tableName);
            return Option.ofNullable(database.getParameters().getOrDefault(HOODIE_LAST_COMMIT_TIME_SYNC, null));
        }
        catch (Exception e) {
            throw new HoodieHiveSyncException("Failed to get the last commit time synced from the database", e);
        }
    }

    public void close() {
        try {
            if (this.connection != null) {
                this.connection.close();
            }
            if (this.client != null) {
                Hive.closeCurrent();
                this.client = null;
            }
        }
        catch (SQLException e) {
            LOG.error((Object)"Could not close connection ", (Throwable)e);
        }
    }

    List<String> getPartitionsWrittenToSince(Option<String> lastCommitTimeSynced) {
        if (!lastCommitTimeSynced.isPresent()) {
            LOG.info((Object)("Last commit time synced is not known, listing all partitions in " + this.syncConfig.basePath + ",FS :" + this.fs));
            try {
                return FSUtils.getAllPartitionPaths(this.fs, this.syncConfig.basePath, this.syncConfig.assumeDatePartitioning);
            }
            catch (IOException e) {
                throw new HoodieIOException("Failed to list all partitions in " + this.syncConfig.basePath, e);
            }
        }
        LOG.info((Object)("Last commit time synced is " + lastCommitTimeSynced.get() + ", Getting commits since then"));
        HoodieTimeline timelineToSync = this.activeTimeline.findInstantsAfter(lastCommitTimeSynced.get(), Integer.MAX_VALUE);
        return timelineToSync.getInstants().map(s -> {
            try {
                return HoodieCommitMetadata.fromBytes(this.activeTimeline.getInstantDetails((HoodieInstant)s).get(), HoodieCommitMetadata.class);
            }
            catch (IOException e) {
                throw new HoodieIOException("Failed to get partitions written since " + lastCommitTimeSynced, e);
            }
        }).flatMap(s -> s.getPartitionToWriteStats().keySet().stream()).distinct().collect(Collectors.toList());
    }

    List<String> getAllTables(String db) throws Exception {
        return this.client.getAllTables(db);
    }

    void updateLastCommitTimeSynced(String tableName) {
        String lastCommitSynced = this.activeTimeline.lastInstant().get().getTimestamp();
        try {
            Table table = this.client.getTable(this.syncConfig.databaseName, tableName);
            table.putToParameters(HOODIE_LAST_COMMIT_TIME_SYNC, lastCommitSynced);
            this.client.alter_table(this.syncConfig.databaseName, tableName, table);
        }
        catch (Exception e) {
            throw new HoodieHiveSyncException("Failed to get update last commit time synced to " + lastCommitSynced, e);
        }
    }

    static {
        try {
            Class.forName(driverName);
        }
        catch (ClassNotFoundException e) {
            throw new IllegalStateException("Could not find " + driverName + " in classpath. ", e);
        }
        LOG = LogManager.getLogger(HoodieHiveClient.class);
    }

    static class PartitionEvent {
        PartitionEventType eventType;
        String storagePartition;

        PartitionEvent(PartitionEventType eventType, String storagePartition) {
            this.eventType = eventType;
            this.storagePartition = storagePartition;
        }

        static PartitionEvent newPartitionAddEvent(String storagePartition) {
            return new PartitionEvent(PartitionEventType.ADD, storagePartition);
        }

        static PartitionEvent newPartitionUpdateEvent(String storagePartition) {
            return new PartitionEvent(PartitionEventType.UPDATE, storagePartition);
        }

        public static enum PartitionEventType {
            ADD,
            UPDATE;

        }
    }
}

