/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.io;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.hudi.avro.model.HoodieCleanMetadata;
import org.apache.hudi.common.model.CompactionOperation;
import org.apache.hudi.common.model.FileSlice;
import org.apache.hudi.common.model.HoodieBaseFile;
import org.apache.hudi.common.model.HoodieCleaningPolicy;
import org.apache.hudi.common.model.HoodieCommitMetadata;
import org.apache.hudi.common.model.HoodieFileGroup;
import org.apache.hudi.common.model.HoodieFileGroupId;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.model.HoodieRecordPayload;
import org.apache.hudi.common.model.HoodieTableType;
import org.apache.hudi.common.table.HoodieTimeline;
import org.apache.hudi.common.table.SyncableFileSystemView;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.util.AvroUtils;
import org.apache.hudi.common.util.FSUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.config.HoodieWriteConfig;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.table.HoodieTable;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class HoodieCleanHelper<T extends HoodieRecordPayload<T>>
implements Serializable {
    private static final Logger LOG = LogManager.getLogger(HoodieCleanHelper.class);
    private final SyncableFileSystemView fileSystemView;
    private final HoodieTimeline commitTimeline;
    private final Map<HoodieFileGroupId, CompactionOperation> fgIdToPendingCompactionOperations;
    private HoodieTable<T> hoodieTable;
    private HoodieWriteConfig config;

    public HoodieCleanHelper(HoodieTable<T> hoodieTable, HoodieWriteConfig config) {
        this.hoodieTable = hoodieTable;
        this.fileSystemView = hoodieTable.getHoodieView();
        this.commitTimeline = hoodieTable.getCompletedCommitTimeline();
        this.config = config;
        this.fgIdToPendingCompactionOperations = ((SyncableFileSystemView)hoodieTable.getSliceView()).getPendingCompactionOperations().map(entry -> Pair.of(new HoodieFileGroupId(((CompactionOperation)entry.getValue()).getPartitionPath(), ((CompactionOperation)entry.getValue()).getFileId()), entry.getValue())).collect(Collectors.toMap(Pair::getKey, Pair::getValue));
    }

    public List<String> getPartitionPathsToClean(Option<HoodieInstant> newInstantToRetain) throws IOException {
        HoodieCleanMetadata cleanMetadata;
        Option<HoodieInstant> lastClean;
        if (this.config.incrementalCleanerModeEnabled() && newInstantToRetain.isPresent() && HoodieCleaningPolicy.KEEP_LATEST_COMMITS == this.config.getCleanerPolicy() && (lastClean = this.hoodieTable.getCleanTimeline().filterCompletedInstants().lastInstant()).isPresent() && (cleanMetadata = AvroUtils.deserializeHoodieCleanMetadata(this.hoodieTable.getActiveTimeline().getInstantDetails(lastClean.get()).get())).getEarliestCommitToRetain() != null && cleanMetadata.getEarliestCommitToRetain().length() > 0) {
            LOG.warn((Object)("Incremental Cleaning mode is enabled. Looking up partition-paths that have since changed since last cleaned at " + cleanMetadata.getEarliestCommitToRetain() + ". New Instant to retain : " + newInstantToRetain));
            return this.hoodieTable.getCompletedCommitsTimeline().getInstants().filter(instant -> HoodieTimeline.compareTimestamps(instant.getTimestamp(), cleanMetadata.getEarliestCommitToRetain(), HoodieTimeline.GREATER_OR_EQUAL) && HoodieTimeline.compareTimestamps(instant.getTimestamp(), ((HoodieInstant)newInstantToRetain.get()).getTimestamp(), HoodieTimeline.LESSER)).flatMap(instant -> {
                try {
                    HoodieCommitMetadata commitMetadata = HoodieCommitMetadata.fromBytes(this.hoodieTable.getActiveTimeline().getInstantDetails((HoodieInstant)instant).get(), HoodieCommitMetadata.class);
                    return commitMetadata.getPartitionToWriteStats().keySet().stream();
                }
                catch (IOException e) {
                    throw new HoodieIOException(e.getMessage(), e);
                }
            }).distinct().collect(Collectors.toList());
        }
        return FSUtils.getAllPartitionPaths(this.hoodieTable.getMetaClient().getFs(), this.hoodieTable.getMetaClient().getBasePath(), this.config.shouldAssumeDatePartitioning());
    }

    private List<String> getFilesToCleanKeepingLatestVersions(String partitionPath) {
        LOG.info((Object)("Cleaning " + partitionPath + ", retaining latest " + this.config.getCleanerFileVersionsRetained() + " file versions. "));
        List fileGroups = this.fileSystemView.getAllFileGroups(partitionPath).collect(Collectors.toList());
        ArrayList<String> deletePaths = new ArrayList<String>();
        List savepointedFiles = this.hoodieTable.getSavepoints().stream().flatMap(s -> this.hoodieTable.getSavepointedDataFiles((String)s)).collect(Collectors.toList());
        for (HoodieFileGroup fileGroup : fileGroups) {
            Serializable dataFile;
            FileSlice nextSlice;
            int keepVersions = this.config.getCleanerFileVersionsRetained();
            Iterator fileSliceIterator = fileGroup.getAllFileSlices().filter(fs -> !this.isFileSliceNeededForPendingCompaction((FileSlice)fs)).iterator();
            if (this.isFileGroupInPendingCompaction(fileGroup)) {
                --keepVersions;
            }
            while (fileSliceIterator.hasNext() && keepVersions > 0) {
                nextSlice = (FileSlice)fileSliceIterator.next();
                dataFile = nextSlice.getBaseFile();
                if (((Option)dataFile).isPresent() && savepointedFiles.contains(((Option)dataFile).get().getFileName())) continue;
                --keepVersions;
            }
            while (fileSliceIterator.hasNext()) {
                nextSlice = (FileSlice)fileSliceIterator.next();
                if (nextSlice.getBaseFile().isPresent()) {
                    dataFile = nextSlice.getBaseFile().get();
                    deletePaths.add(((HoodieBaseFile)dataFile).getFileName());
                }
                if (this.hoodieTable.getMetaClient().getTableType() != HoodieTableType.MERGE_ON_READ) continue;
                deletePaths.addAll(nextSlice.getLogFiles().map(HoodieLogFile::getFileName).collect(Collectors.toList()));
            }
        }
        return deletePaths;
    }

    private List<String> getFilesToCleanKeepingLatestCommits(String partitionPath) {
        int commitsRetained = this.config.getCleanerCommitsRetained();
        LOG.info((Object)("Cleaning " + partitionPath + ", retaining latest " + commitsRetained + " commits. "));
        ArrayList<String> deletePaths = new ArrayList<String>();
        List savepointedFiles = this.hoodieTable.getSavepoints().stream().flatMap(s -> this.hoodieTable.getSavepointedDataFiles((String)s)).collect(Collectors.toList());
        if (this.commitTimeline.countInstants() > commitsRetained) {
            HoodieInstant earliestCommitToRetain = this.getEarliestCommitToRetain().get();
            List fileGroups = this.fileSystemView.getAllFileGroups(partitionPath).collect(Collectors.toList());
            for (HoodieFileGroup fileGroup : fileGroups) {
                List<FileSlice> fileSliceList = fileGroup.getAllFileSlices().collect(Collectors.toList());
                if (fileSliceList.isEmpty()) continue;
                String lastVersion = ((FileSlice)fileSliceList.get(0)).getBaseInstantTime();
                String lastVersionBeforeEarliestCommitToRetain = this.getLatestVersionBeforeCommit(fileSliceList, earliestCommitToRetain);
                for (FileSlice aSlice : fileSliceList) {
                    Option<HoodieBaseFile> aFile = aSlice.getBaseFile();
                    String fileCommitTime = aSlice.getBaseInstantTime();
                    if (aFile.isPresent() && savepointedFiles.contains(aFile.get().getFileName()) || fileCommitTime.equals(lastVersion) || fileCommitTime.equals(lastVersionBeforeEarliestCommitToRetain) || this.isFileSliceNeededForPendingCompaction(aSlice) || !HoodieTimeline.compareTimestamps(earliestCommitToRetain.getTimestamp(), fileCommitTime, HoodieTimeline.GREATER)) continue;
                    aFile.ifPresent(hoodieDataFile -> deletePaths.add(hoodieDataFile.getFileName()));
                    if (this.hoodieTable.getMetaClient().getTableType() != HoodieTableType.MERGE_ON_READ) continue;
                    deletePaths.addAll(aSlice.getLogFiles().map(HoodieLogFile::getFileName).collect(Collectors.toList()));
                }
            }
        }
        return deletePaths;
    }

    private String getLatestVersionBeforeCommit(List<FileSlice> fileSliceList, HoodieInstant commitTime) {
        for (FileSlice file : fileSliceList) {
            String fileCommitTime = file.getBaseInstantTime();
            if (!HoodieTimeline.compareTimestamps(commitTime.getTimestamp(), fileCommitTime, HoodieTimeline.GREATER)) continue;
            return fileCommitTime;
        }
        return null;
    }

    public List<String> getDeletePaths(String partitionPath) {
        List<String> deletePaths;
        HoodieCleaningPolicy policy = this.config.getCleanerPolicy();
        if (policy == HoodieCleaningPolicy.KEEP_LATEST_COMMITS) {
            deletePaths = this.getFilesToCleanKeepingLatestCommits(partitionPath);
        } else if (policy == HoodieCleaningPolicy.KEEP_LATEST_FILE_VERSIONS) {
            deletePaths = this.getFilesToCleanKeepingLatestVersions(partitionPath);
        } else {
            throw new IllegalArgumentException("Unknown cleaning policy : " + policy.name());
        }
        LOG.info((Object)(deletePaths.size() + " patterns used to delete in partition path:" + partitionPath));
        return deletePaths;
    }

    public Option<HoodieInstant> getEarliestCommitToRetain() {
        Option<HoodieInstant> earliestCommitToRetain = Option.empty();
        int commitsRetained = this.config.getCleanerCommitsRetained();
        if (this.config.getCleanerPolicy() == HoodieCleaningPolicy.KEEP_LATEST_COMMITS && this.commitTimeline.countInstants() > commitsRetained) {
            earliestCommitToRetain = this.commitTimeline.nthInstant(this.commitTimeline.countInstants() - commitsRetained);
        }
        return earliestCommitToRetain;
    }

    private boolean isFileSliceNeededForPendingCompaction(FileSlice fileSlice) {
        CompactionOperation op = this.fgIdToPendingCompactionOperations.get(fileSlice.getFileGroupId());
        if (null != op) {
            return HoodieTimeline.compareTimestamps(fileSlice.getBaseInstantTime(), op.getBaseInstantTime(), HoodieTimeline.GREATER_OR_EQUAL);
        }
        return false;
    }

    private boolean isFileGroupInPendingCompaction(HoodieFileGroup fg) {
        return this.fgIdToPendingCompactionOperations.containsKey(fg.getFileGroupId());
    }
}

