/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.io;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.avro.Schema;
import org.apache.avro.generic.IndexedRecord;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.avro.model.HoodieArchivedMetaEntry;
import org.apache.hudi.avro.model.HoodieCommitMetadata;
import org.apache.hudi.avro.model.HoodieCompactionPlan;
import org.apache.hudi.avro.model.HoodieRollbackMetadata;
import org.apache.hudi.avro.model.HoodieSavepointMetadata;
import org.apache.hudi.common.model.ActionType;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.HoodieTimeline;
import org.apache.hudi.common.table.log.HoodieLogFormat;
import org.apache.hudi.common.table.log.block.HoodieAvroDataBlock;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.common.table.timeline.HoodieActiveTimeline;
import org.apache.hudi.common.table.timeline.HoodieArchivedTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.util.AvroUtils;
import org.apache.hudi.common.util.CleanerUtils;
import org.apache.hudi.common.util.CompactionUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.config.HoodieWriteConfig;
import org.apache.hudi.exception.HoodieCommitException;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.table.HoodieTable;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.spark.api.java.JavaSparkContext;

public class HoodieCommitArchiveLog {
    private static final Logger LOG = LogManager.getLogger(HoodieCommitArchiveLog.class);
    private final Path archiveFilePath;
    private final HoodieTableMetaClient metaClient;
    private final HoodieWriteConfig config;
    private HoodieLogFormat.Writer writer;

    public HoodieCommitArchiveLog(HoodieWriteConfig config, HoodieTableMetaClient metaClient) {
        this.config = config;
        this.metaClient = metaClient;
        this.archiveFilePath = HoodieArchivedTimeline.getArchiveLogPath(metaClient.getArchivePath());
    }

    private HoodieLogFormat.Writer openWriter() {
        try {
            if (this.writer == null) {
                return HoodieLogFormat.newWriterBuilder().onParentPath(this.archiveFilePath.getParent()).withFileId(this.archiveFilePath.getName()).withFileExtension(".archive").withFs(this.metaClient.getFs()).overBaseCommit("").build();
            }
            return this.writer;
        }
        catch (IOException | InterruptedException e) {
            throw new HoodieException("Unable to initialize HoodieLogFormat writer", e);
        }
    }

    private void close() {
        try {
            if (this.writer != null) {
                this.writer.close();
            }
        }
        catch (IOException e) {
            throw new HoodieException("Unable to close HoodieLogFormat writer", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean archiveIfRequired(JavaSparkContext jsc) throws IOException {
        try {
            List<HoodieInstant> instantsToArchive = this.getInstantsToArchive(jsc).collect(Collectors.toList());
            boolean success = true;
            if (!instantsToArchive.isEmpty()) {
                this.writer = this.openWriter();
                LOG.info((Object)("Archiving instants " + instantsToArchive));
                this.archive(instantsToArchive);
                LOG.info((Object)("Deleting archived instants " + instantsToArchive));
                success = this.deleteArchivedInstants(instantsToArchive);
            } else {
                LOG.info((Object)"No Instants to archive");
            }
            boolean bl = success;
            return bl;
        }
        finally {
            this.close();
        }
    }

    private Stream<HoodieInstant> getInstantsToArchive(JavaSparkContext jsc) {
        int maxCommitsToKeep = this.config.getMaxCommitsToKeep();
        int minCommitsToKeep = this.config.getMinCommitsToKeep();
        HoodieTable table = HoodieTable.getHoodieTable(this.metaClient, this.config, jsc);
        HoodieTimeline cleanAndRollbackTimeline = table.getActiveTimeline().getTimelineOfActions(Sets.newHashSet((Object[])new String[]{"clean"})).filterCompletedInstants();
        Stream<Object> instants = cleanAndRollbackTimeline.getInstants().collect(Collectors.groupingBy(s -> s.getAction())).entrySet().stream().map(i -> {
            if (((List)i.getValue()).size() > maxCommitsToKeep) {
                return ((List)i.getValue()).subList(0, ((List)i.getValue()).size() - minCommitsToKeep);
            }
            return new ArrayList();
        }).flatMap(Collection::stream);
        HoodieTimeline commitTimeline = table.getCompletedCommitsTimeline();
        Option<HoodieInstant> oldestPendingCompactionInstant = table.getActiveTimeline().filterPendingCompactionTimeline().firstInstant();
        Option<HoodieInstant> firstSavepoint = table.getCompletedSavepointTimeline().firstInstant();
        if (!commitTimeline.empty() && commitTimeline.countInstants() > maxCommitsToKeep) {
            instants = Stream.concat(instants, commitTimeline.getInstants().filter(s -> !firstSavepoint.isPresent() || !HoodieTimeline.compareTimestamps(((HoodieInstant)firstSavepoint.get()).getTimestamp(), s.getTimestamp(), HoodieTimeline.LESSER_OR_EQUAL)).filter(s -> oldestPendingCompactionInstant.map(instant -> HoodieTimeline.compareTimestamps(instant.getTimestamp(), s.getTimestamp(), HoodieTimeline.GREATER)).orElse(true)).limit(commitTimeline.countInstants() - minCommitsToKeep));
        }
        HoodieActiveTimeline rawActiveTimeline = new HoodieActiveTimeline(this.metaClient, false);
        Map<Pair, List<HoodieInstant>> groupByTsAction = rawActiveTimeline.getInstants().collect(Collectors.groupingBy(i -> Pair.of(i.getTimestamp(), HoodieInstant.getComparableAction(i.getAction()))));
        return instants.flatMap(hoodieInstant -> ((List)groupByTsAction.get(Pair.of(hoodieInstant.getTimestamp(), HoodieInstant.getComparableAction(hoodieInstant.getAction())))).stream());
    }

    private boolean deleteArchivedInstants(List<HoodieInstant> archivedInstants) throws IOException {
        LOG.info((Object)("Deleting instants " + archivedInstants));
        boolean success = true;
        for (HoodieInstant archivedInstant : archivedInstants) {
            Path commitFile = new Path(this.metaClient.getMetaPath(), archivedInstant.getFileName());
            try {
                if (!this.metaClient.getFs().exists(commitFile)) continue;
                success &= this.metaClient.getFs().delete(commitFile, false);
                LOG.info((Object)("Archived and deleted instant file " + commitFile));
            }
            catch (IOException e) {
                throw new HoodieIOException("Failed to delete archived instant " + archivedInstant, e);
            }
        }
        Option<HoodieInstant> latestCommitted = Option.fromJavaOptional(archivedInstants.stream().filter(i -> i.isCompleted() && (i.getAction().equals("commit") || i.getAction().equals("deltacommit"))).max(Comparator.comparing(HoodieInstant::getTimestamp)));
        LOG.info((Object)("Latest Committed Instant=" + latestCommitted));
        if (latestCommitted.isPresent()) {
            success &= this.deleteAllInstantsOlderorEqualsInAuxMetaFolder(latestCommitted.get());
        }
        return success;
    }

    private boolean deleteAllInstantsOlderorEqualsInAuxMetaFolder(HoodieInstant thresholdInstant) throws IOException {
        List<HoodieInstant> instants = this.metaClient.scanHoodieInstantsFromFileSystem(new Path(this.metaClient.getMetaAuxiliaryPath()), HoodieActiveTimeline.VALID_EXTENSIONS_IN_ACTIVE_TIMELINE, false);
        List instantsToBeDeleted = instants.stream().filter(instant1 -> HoodieTimeline.compareTimestamps(instant1.getTimestamp(), thresholdInstant.getTimestamp(), HoodieTimeline.LESSER_OR_EQUAL)).collect(Collectors.toList());
        boolean success = true;
        for (HoodieInstant deleteInstant : instantsToBeDeleted) {
            LOG.info((Object)("Deleting instant " + deleteInstant + " in auxiliary meta path " + this.metaClient.getMetaAuxiliaryPath()));
            Path metaFile = new Path(this.metaClient.getMetaAuxiliaryPath(), deleteInstant.getFileName());
            if (!this.metaClient.getFs().exists(metaFile)) continue;
            success &= this.metaClient.getFs().delete(metaFile, false);
            LOG.info((Object)("Deleted instant file in auxiliary metapath : " + metaFile));
        }
        return success;
    }

    public void archive(List<HoodieInstant> instants) throws HoodieCommitException {
        try {
            HoodieTimeline commitTimeline = this.metaClient.getActiveTimeline().getAllCommitsTimeline().filterCompletedInstants();
            Schema wrapperSchema = HoodieArchivedMetaEntry.getClassSchema();
            LOG.info((Object)("Wrapper schema " + wrapperSchema.toString()));
            ArrayList<IndexedRecord> records = new ArrayList<IndexedRecord>();
            for (HoodieInstant hoodieInstant : instants) {
                try {
                    records.add(this.convertToAvroRecord(commitTimeline, hoodieInstant));
                    if (records.size() < this.config.getCommitArchivalBatchSize()) continue;
                    this.writeToFile(wrapperSchema, records);
                }
                catch (Exception e) {
                    LOG.error((Object)("Failed to archive commits, .commit file: " + hoodieInstant.getFileName()), (Throwable)e);
                    if (!this.config.isFailOnTimelineArchivingEnabled()) continue;
                    throw e;
                }
            }
            this.writeToFile(wrapperSchema, records);
        }
        catch (Exception e) {
            throw new HoodieCommitException("Failed to archive commits", e);
        }
    }

    public Path getArchiveFilePath() {
        return this.archiveFilePath;
    }

    private void writeToFile(Schema wrapperSchema, List<IndexedRecord> records) throws Exception {
        if (records.size() > 0) {
            HashMap header = Maps.newHashMap();
            header.put(HoodieLogBlock.HeaderMetadataType.SCHEMA, wrapperSchema.toString());
            HoodieAvroDataBlock block = new HoodieAvroDataBlock(records, header);
            this.writer = this.writer.appendBlock(block);
            records.clear();
        }
    }

    private IndexedRecord convertToAvroRecord(HoodieTimeline commitTimeline, HoodieInstant hoodieInstant) throws IOException {
        HoodieArchivedMetaEntry archivedMetaWrapper = new HoodieArchivedMetaEntry();
        archivedMetaWrapper.setCommitTime(hoodieInstant.getTimestamp());
        archivedMetaWrapper.setActionState(hoodieInstant.getState().name());
        switch (hoodieInstant.getAction()) {
            case "clean": {
                if (hoodieInstant.isCompleted()) {
                    archivedMetaWrapper.setHoodieCleanMetadata(CleanerUtils.getCleanerMetadata(this.metaClient, hoodieInstant));
                } else {
                    archivedMetaWrapper.setHoodieCleanerPlan(CleanerUtils.getCleanerPlan(this.metaClient, hoodieInstant));
                }
                archivedMetaWrapper.setActionType(ActionType.clean.name());
                break;
            }
            case "commit": {
                org.apache.hudi.common.model.HoodieCommitMetadata commitMetadata = org.apache.hudi.common.model.HoodieCommitMetadata.fromBytes(commitTimeline.getInstantDetails(hoodieInstant).get(), org.apache.hudi.common.model.HoodieCommitMetadata.class);
                archivedMetaWrapper.setHoodieCommitMetadata(this.commitMetadataConverter(commitMetadata));
                archivedMetaWrapper.setActionType(ActionType.commit.name());
                break;
            }
            case "rollback": {
                archivedMetaWrapper.setHoodieRollbackMetadata(AvroUtils.deserializeAvroMetadata(commitTimeline.getInstantDetails(hoodieInstant).get(), HoodieRollbackMetadata.class));
                archivedMetaWrapper.setActionType(ActionType.rollback.name());
                break;
            }
            case "savepoint": {
                archivedMetaWrapper.setHoodieSavePointMetadata(AvroUtils.deserializeAvroMetadata(commitTimeline.getInstantDetails(hoodieInstant).get(), HoodieSavepointMetadata.class));
                archivedMetaWrapper.setActionType(ActionType.savepoint.name());
                break;
            }
            case "deltacommit": {
                org.apache.hudi.common.model.HoodieCommitMetadata commitMetadata = org.apache.hudi.common.model.HoodieCommitMetadata.fromBytes(commitTimeline.getInstantDetails(hoodieInstant).get(), org.apache.hudi.common.model.HoodieCommitMetadata.class);
                archivedMetaWrapper.setHoodieCommitMetadata(this.commitMetadataConverter(commitMetadata));
                archivedMetaWrapper.setActionType(ActionType.commit.name());
                break;
            }
            case "compaction": {
                HoodieCompactionPlan plan = CompactionUtils.getCompactionPlan(this.metaClient, hoodieInstant.getTimestamp());
                archivedMetaWrapper.setHoodieCompactionPlan(plan);
                archivedMetaWrapper.setActionType(ActionType.compaction.name());
                break;
            }
            default: {
                throw new UnsupportedOperationException("Action not fully supported yet");
            }
        }
        return archivedMetaWrapper;
    }

    private HoodieCommitMetadata commitMetadataConverter(org.apache.hudi.common.model.HoodieCommitMetadata hoodieCommitMetadata) {
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        HoodieCommitMetadata avroMetaData = (HoodieCommitMetadata)((Object)mapper.convertValue((Object)hoodieCommitMetadata, HoodieCommitMetadata.class));
        avroMetaData.getExtraMetadata().put("ROLLING_STAT", "");
        return avroMetaData;
    }
}

