/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import org.apache.hadoop.hbase.testclassification.SmallTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Order;
import org.apache.hadoop.hbase.util.OrderedBytes;
import org.apache.hadoop.hbase.util.PositionedByteRange;
import org.apache.hadoop.hbase.util.SimplePositionedMutableByteRange;
import org.junit.Assert;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={SmallTests.class})
public class TestOrderedBytes {
    static final Long[] I_VALS = new Long[]{0L, 1L, 10L, 99L, 100L, 1234L, 9999L, 10000L, 10001L, 12345L, 123450L, Long.MAX_VALUE, -1L, -10L, -99L, -100L, -123L, -999L, -10000L, -10001L, -12345L, -123450L, Long.MIN_VALUE};
    static final int[] I_LENGTHS = new int[]{1, 2, 2, 2, 2, 3, 3, 2, 4, 4, 4, 11, 2, 2, 2, 2, 3, 3, 2, 4, 4, 4, 11};
    static final Double[] D_VALS = new Double[]{0.0, 0.00123, 0.0123, 0.123, 1.0, 10.0, 12.345, 99.0, 99.01, 99.0001, 100.0, 100.01, 100.1, 1234.0, 1234.5, 9999.0, 9999.000001, 9999.000009, 9999.00001, 9999.00009, 9999.000099, 9999.0001, 9999.001, 9999.01, 9999.1, 10000.0, 10001.0, 12345.0, 123450.0, Double.NEGATIVE_INFINITY, Double.POSITIVE_INFINITY, Double.NaN, Double.MAX_VALUE, -0.00123, -0.0123, -0.123, -1.0, -10.0, -12.345, -99.0, -99.01, -99.0001, -100.0, -100.01, -100.1, -1234.0, -1234.5, -9999.0, -9999.000001, -9999.000009, -9999.00001, -9999.00009, -9999.000099, -9999.0001, -9999.001, -9999.01, -9999.1, -10000.0, -10001.0, -12345.0, -123450.0};
    static final int[] D_LENGTHS = new int[]{1, 4, 4, 4, 2, 2, 4, 2, 3, 4, 2, 4, 4, 3, 4, 3, 6, 6, 6, 6, 6, 5, 5, 4, 4, 2, 4, 4, 4, 1, 1, 1, 11, 4, 4, 4, 2, 2, 4, 2, 3, 4, 2, 4, 4, 3, 4, 3, 6, 6, 6, 6, 6, 5, 5, 4, 4, 2, 4, 4, 4};
    static final BigDecimal[] BD_VALS = new BigDecimal[]{null, BigDecimal.valueOf(Long.MAX_VALUE), BigDecimal.valueOf(Long.MIN_VALUE), BigDecimal.valueOf(Double.MAX_VALUE), BigDecimal.valueOf(Double.MIN_VALUE), BigDecimal.valueOf(Long.MAX_VALUE).multiply(BigDecimal.valueOf(100L))};
    static final int[] BD_LENGTHS = new int[]{1, 11, 11, 11, 4, 12};
    static final double MIN_EPSILON = 1.0E-6;

    @Test
    public void testVerifyTestIntegrity() {
        for (int i = 0; i < I_VALS.length; ++i) {
            for (int d = 0; d < D_VALS.length; ++d) {
                if (!(Math.abs((double)I_VALS[i].longValue() - D_VALS[d]) < 1.0E-6)) continue;
                Assert.assertEquals((String)("Test inconsistency detected: expected lengths for " + I_VALS[i] + " do not match."), (long)I_LENGTHS[i], (long)D_LENGTHS[d]);
            }
        }
    }

    @Test
    public void testVaruint64Boundaries() {
        long[] vals = new long[]{239L, 240L, 2286L, 2287L, 67822L, 67823L, 0xFFFFFEL, 0xFFFFFFL, 0xFFFFFFFEL, 0xFFFFFFFFL, 0xFFFFFFFFFEL, 0xFFFFFFFFFFL, 0xFFFFFFFFFFFEL, 0xFFFFFFFFFFFFL, 0xFFFFFFFFFFFFFEL, 0xFFFFFFFFFFFFFFL, 0x7FFFFFFFFFFFFFFEL, Long.MAX_VALUE, -9223372036854775807L, Long.MIN_VALUE, -2L, -1L};
        int[] lens = new int[]{1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 9, 9, 9, 9};
        Assert.assertEquals((String)"Broken test!", (long)vals.length, (long)lens.length);
        for (boolean comp : new boolean[]{true, false}) {
            for (int i = 0; i < vals.length; ++i) {
                byte[] a = new byte[lens[i] + 2];
                SimplePositionedMutableByteRange buf = new SimplePositionedMutableByteRange(a, 1, lens[i]);
                Assert.assertEquals((String)"Surprising return value.", (long)lens[i], (long)OrderedBytes.putVaruint64(buf, vals[i], comp));
                Assert.assertEquals((String)"Surprising serialized length.", (long)lens[i], (long)buf.getPosition());
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf.setPosition(0);
                Assert.assertEquals((String)"Surprising return value.", (long)lens[i], (long)OrderedBytes.skipVaruint64(buf, comp));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)lens[i], (long)buf.getPosition());
                buf.setPosition(0);
                Assert.assertEquals((String)"Deserialization failed.", (long)vals[i], (long)OrderedBytes.getVaruint64(buf, comp));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)lens[i], (long)buf.getPosition());
            }
        }
    }

    @Test
    public void testNumericInt() {
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < I_VALS.length; ++i) {
                byte[] a = new byte[I_LENGTHS[i] + 3];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, I_LENGTHS[i] + 1);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)I_LENGTHS[i], (long)OrderedBytes.encodeNumeric((PositionedByteRange)buf1, I_VALS[i], ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)I_LENGTHS[i], (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)I_LENGTHS[i], (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)I_LENGTHS[i], (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertEquals((String)"Deserialization failed.", (long)I_VALS[i], (long)OrderedBytes.decodeNumericAsLong(buf1));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)I_LENGTHS[i], (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[I_VALS.length][];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < I_VALS.length; ++i) {
                encoded[i] = new byte[I_LENGTHS[i]];
                OrderedBytes.encodeNumeric(pbr.set(encoded[i]), I_VALS[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            Object[] sortedVals = Arrays.copyOf(I_VALS, I_VALS.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder());
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                pbr.set(encoded[i]);
                long decoded = OrderedBytes.decodeNumericAsLong(pbr);
                Assert.assertEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{sortedVals[i], decoded, ord}), (long)((Long)sortedVals[i]), (long)decoded);
            }
        }
    }

    @Test
    public void testNumericReal() {
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < D_VALS.length; ++i) {
                byte[] a = new byte[D_LENGTHS[i] + 3];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, D_LENGTHS[i] + 1);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)D_LENGTHS[i], (long)OrderedBytes.encodeNumeric((PositionedByteRange)buf1, D_VALS[i], ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)D_LENGTHS[i], (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)D_LENGTHS[i], (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)D_LENGTHS[i], (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertEquals((String)"Deserialization failed.", (double)D_VALS[i], (double)OrderedBytes.decodeNumericAsDouble(buf1), (double)1.0E-6);
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)D_LENGTHS[i], (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[D_VALS.length][];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < D_VALS.length; ++i) {
                encoded[i] = new byte[D_LENGTHS[i]];
                OrderedBytes.encodeNumeric(pbr.set(encoded[i]), D_VALS[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            Object[] sortedVals = Arrays.copyOf(D_VALS, D_VALS.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder());
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                pbr.set(encoded[i]);
                double decoded = OrderedBytes.decodeNumericAsDouble(pbr);
                Assert.assertEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{sortedVals[i], decoded, ord}), (double)((Double)sortedVals[i]), (double)decoded, (double)1.0E-6);
            }
        }
    }

    @Test
    public void testNumericOther() {
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < BD_VALS.length; ++i) {
                byte[] a = new byte[BD_LENGTHS[i] + 3];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, BD_LENGTHS[i] + 1);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)BD_LENGTHS[i], (long)OrderedBytes.encodeNumeric((PositionedByteRange)buf1, BD_VALS[i], ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)BD_LENGTHS[i], (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)BD_LENGTHS[i], (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)BD_LENGTHS[i], (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                BigDecimal decoded = OrderedBytes.decodeNumericAsBigDecimal(buf1);
                if (null == BD_VALS[i]) {
                    Assert.assertEquals((Object)BD_VALS[i], (Object)decoded);
                } else {
                    Assert.assertEquals((String)"Deserialization failed.", (long)0L, (long)BD_VALS[i].compareTo(decoded));
                }
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)BD_LENGTHS[i], (long)(buf1.getPosition() - 1));
            }
        }
    }

    @Test
    public void testNumericIntRealCompatibility() {
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < I_VALS.length; ++i) {
                SimplePositionedMutableByteRange pbri = new SimplePositionedMutableByteRange(I_LENGTHS[i]);
                SimplePositionedMutableByteRange pbrr = new SimplePositionedMutableByteRange(I_LENGTHS[i]);
                OrderedBytes.encodeNumeric((PositionedByteRange)pbri, I_VALS[i], ord);
                OrderedBytes.encodeNumeric((PositionedByteRange)pbrr, I_VALS[i], ord);
                Assert.assertArrayEquals((String)"Integer and real encodings differ.", (byte[])pbri.getBytes(), (byte[])pbrr.getBytes());
                pbri.setPosition(0);
                pbrr.setPosition(0);
                Assert.assertEquals((long)I_VALS[i], (long)OrderedBytes.decodeNumericAsLong(pbri));
                Assert.assertEquals((long)I_VALS[i], (long)((long)OrderedBytes.decodeNumericAsDouble(pbrr)));
                BigDecimal bd = BigDecimal.valueOf(I_VALS[i]);
                SimplePositionedMutableByteRange pbrbd = new SimplePositionedMutableByteRange(I_LENGTHS[i]);
                OrderedBytes.encodeNumeric((PositionedByteRange)pbrbd, bd, ord);
                Assert.assertArrayEquals((String)"Integer and BigDecimal encodings differ.", (byte[])pbri.getBytes(), (byte[])pbrbd.getBytes());
                pbri.setPosition(0);
                Assert.assertEquals((String)"Value not preserved when decoding as Long", (long)0L, (long)bd.compareTo(BigDecimal.valueOf(OrderedBytes.decodeNumericAsLong(pbri))));
            }
        }
    }

    @Test
    public void testInt8() {
        Byte[] vals = new Byte[]{(byte)-128, (byte)-64, (byte)0, (byte)63, (byte)127};
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < vals.length; ++i) {
                byte[] a = new byte[5];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, 3);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)2L, (long)OrderedBytes.encodeInt8(buf1, vals[i], ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)2L, (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)2L, (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)2L, (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertEquals((String)"Deserialization failed.", (long)vals[i].byteValue(), (long)OrderedBytes.decodeInt8(buf1));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)2L, (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[vals.length][2];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < vals.length; ++i) {
                OrderedBytes.encodeInt8(pbr.set(encoded[i]), vals[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            Object[] sortedVals = Arrays.copyOf(vals, vals.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder());
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                byte decoded = OrderedBytes.decodeInt8(pbr.set(encoded[i]));
                Assert.assertEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{sortedVals[i], (int)decoded, ord}), (long)((Byte)sortedVals[i]).byteValue(), (long)decoded);
            }
        }
    }

    @Test
    public void testInt16() {
        Short[] vals = new Short[]{(short)Short.MIN_VALUE, (short)-16384, (short)0, (short)16383, (short)Short.MAX_VALUE};
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < vals.length; ++i) {
                byte[] a = new byte[6];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, 4);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)3L, (long)OrderedBytes.encodeInt16(buf1, vals[i], ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)3L, (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)3L, (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)3L, (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertEquals((String)"Deserialization failed.", (long)vals[i].shortValue(), (long)OrderedBytes.decodeInt16(buf1));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)3L, (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[vals.length][3];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < vals.length; ++i) {
                OrderedBytes.encodeInt16(pbr.set(encoded[i]), vals[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            Object[] sortedVals = Arrays.copyOf(vals, vals.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder());
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                short decoded = OrderedBytes.decodeInt16(pbr.set(encoded[i]));
                Assert.assertEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{sortedVals[i], (int)decoded, ord}), (long)((Short)sortedVals[i]).shortValue(), (long)decoded);
            }
        }
    }

    @Test
    public void testInt32() {
        Integer[] vals = new Integer[]{Integer.MIN_VALUE, -1073741824, 0, 0x3FFFFFFF, Integer.MAX_VALUE};
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < vals.length; ++i) {
                byte[] a = new byte[8];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, 6);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)5L, (long)OrderedBytes.encodeInt32(buf1, vals[i], ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)5L, (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)5L, (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)5L, (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertEquals((String)"Deserialization failed.", (long)vals[i].intValue(), (long)OrderedBytes.decodeInt32(buf1));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)5L, (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[vals.length][5];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < vals.length; ++i) {
                OrderedBytes.encodeInt32(pbr.set(encoded[i]), vals[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            Object[] sortedVals = Arrays.copyOf(vals, vals.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder());
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                int decoded = OrderedBytes.decodeInt32(pbr.set(encoded[i]));
                Assert.assertEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{sortedVals[i], decoded, ord}), (long)((Integer)sortedVals[i]).intValue(), (long)decoded);
            }
        }
    }

    @Test
    public void testInt64() {
        Long[] vals = new Long[]{Long.MIN_VALUE, -4611686018427387904L, 0L, 0x3FFFFFFFFFFFFFFFL, Long.MAX_VALUE};
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < vals.length; ++i) {
                byte[] a = new byte[12];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, 10);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)9L, (long)OrderedBytes.encodeInt64(buf1, vals[i], ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)9L, (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)9L, (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)9L, (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertEquals((String)"Deserialization failed.", (long)vals[i], (long)OrderedBytes.decodeInt64(buf1));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)9L, (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[vals.length][9];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < vals.length; ++i) {
                OrderedBytes.encodeInt64(pbr.set(encoded[i]), vals[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            Object[] sortedVals = Arrays.copyOf(vals, vals.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder());
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                long decoded = OrderedBytes.decodeInt64(pbr.set(encoded[i]));
                Assert.assertEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{sortedVals[i], decoded, ord}), (long)((Long)sortedVals[i]), (long)decoded);
            }
        }
    }

    @Test
    public void testFloat32() {
        Float[] vals = new Float[]{Float.valueOf(Float.MIN_VALUE), Float.valueOf(1.0f), Float.valueOf(0.0f), Float.valueOf(1.7014117E38f), Float.valueOf(Float.MAX_VALUE)};
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < vals.length; ++i) {
                byte[] a = new byte[8];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, 6);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)5L, (long)OrderedBytes.encodeFloat32(buf1, vals[i].floatValue(), ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)5L, (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)5L, (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)5L, (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertEquals((String)"Deserialization failed.", (long)Float.floatToIntBits(vals[i].floatValue()), (long)Float.floatToIntBits(OrderedBytes.decodeFloat32(buf1)));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)5L, (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[vals.length][5];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < vals.length; ++i) {
                OrderedBytes.encodeFloat32(pbr.set(encoded[i]), vals[i].floatValue(), ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            Object[] sortedVals = Arrays.copyOf(vals, vals.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder());
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                float decoded = OrderedBytes.decodeFloat32(pbr.set(encoded[i]));
                Assert.assertEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{sortedVals[i], Float.valueOf(decoded), ord}), (long)Float.floatToIntBits(((Float)sortedVals[i]).floatValue()), (long)Float.floatToIntBits(decoded));
            }
        }
    }

    @Test
    public void testFloat64() {
        Double[] vals = new Double[]{Double.MIN_VALUE, 1.0, 0.0, 8.988465674311579E307, Double.MAX_VALUE};
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < vals.length; ++i) {
                byte[] a = new byte[12];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, 10);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)9L, (long)OrderedBytes.encodeFloat64(buf1, vals[i], ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)9L, (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)9L, (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)9L, (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertEquals((String)"Deserialization failed.", (long)Double.doubleToLongBits(vals[i]), (long)Double.doubleToLongBits(OrderedBytes.decodeFloat64(buf1)));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)9L, (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[vals.length][9];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < vals.length; ++i) {
                OrderedBytes.encodeFloat64(pbr.set(encoded[i]), vals[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            Object[] sortedVals = Arrays.copyOf(vals, vals.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder());
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                double decoded = OrderedBytes.decodeFloat64(pbr.set(encoded[i]));
                Assert.assertEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{sortedVals[i], decoded, ord}), (long)Double.doubleToLongBits((Double)sortedVals[i]), (long)Double.doubleToLongBits(decoded));
            }
        }
    }

    @Test
    public void testString() {
        String[] vals = new String[]{"foo", "baaaar", "bazz"};
        int[] expectedLengths = new int[]{5, 8, 6};
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (int i = 0; i < vals.length; ++i) {
                byte[] a = new byte[expectedLengths[i] + 3];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, expectedLengths[i] + 1);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)expectedLengths[i], (long)OrderedBytes.encodeString(buf1, vals[i], ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)expectedLengths[i], (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)expectedLengths[i], (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)expectedLengths[i], (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertEquals((String)"Deserialization failed.", (Object)vals[i], (Object)OrderedBytes.decodeString(buf1));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)expectedLengths[i], (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[vals.length][];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < vals.length; ++i) {
                encoded[i] = new byte[expectedLengths[i]];
                OrderedBytes.encodeString(pbr.set(encoded[i]), vals[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            Object[] sortedVals = Arrays.copyOf(vals, vals.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder());
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                pbr.set(encoded[i]);
                String decoded = OrderedBytes.decodeString(pbr);
                Assert.assertEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{sortedVals[i], decoded, ord}), (Object)sortedVals[i], (Object)decoded);
            }
        }
    }

    @Test(expected=IllegalArgumentException.class)
    public void testStringNoNullChars() {
        SimplePositionedMutableByteRange buff = new SimplePositionedMutableByteRange(3);
        OrderedBytes.encodeString(buff, "\u0000", Order.ASCENDING);
    }

    @Test
    public void testBlobVarLencodedLength() {
        int[][] values2;
        for (int[] pair : values2 = new int[][]{{1, 3}, {2, 4}, {3, 5}, {4, 6}, {5, 7}, {6, 8}, {7, 9}, {8, 11}}) {
            Assert.assertEquals((long)pair[1], (long)OrderedBytes.blobVarEncodedLength(pair[0]));
            Assert.assertEquals((long)pair[0], (long)OrderedBytes.blobVarDecodedLength(pair[1]));
        }
    }

    @Test
    public void testBlobVar() {
        byte[][] vals = new byte[][]{"".getBytes(), "foo".getBytes(), "foobarbazbub".getBytes(), {-86, -86, -86, -86, -86, -86, -86}, {-86, -86, -86, -86, -86, -86, -86, -86, -86, -86, -86, -86}, {-86, -86, -86, -86, -86, -86, -86, -86, -86, -86, -86, -86, -86, -86}, {85, 85, 85, 85, 85, 85, 85}, {85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85}, {85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85}, "1".getBytes(), "22".getBytes(), "333".getBytes(), "4444".getBytes(), "55555".getBytes(), "666666".getBytes(), "7777777".getBytes(), "88888888".getBytes()};
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (byte[] val : vals) {
                int expectedLen = OrderedBytes.blobVarEncodedLength(val.length);
                byte[] a = new byte[expectedLen + 3];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, expectedLen + 1);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)expectedLen, (long)OrderedBytes.encodeBlobVar(buf1, val, ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)expectedLen, (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)expectedLen, (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)expectedLen, (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertArrayEquals((String)"Deserialization failed.", (byte[])val, (byte[])OrderedBytes.decodeBlobVar(buf1));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)expectedLen, (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[vals.length][];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < vals.length; ++i) {
                encoded[i] = new byte[OrderedBytes.blobVarEncodedLength(vals[i].length)];
                OrderedBytes.encodeBlobVar(pbr.set(encoded[i]), vals[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            byte[][] sortedVals = (byte[][])Arrays.copyOf(vals, vals.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals, Bytes.BYTES_COMPARATOR);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder(Bytes.BYTES_COMPARATOR));
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                pbr.set(encoded[i]);
                byte[] decoded = OrderedBytes.decodeBlobVar(pbr);
                Assert.assertArrayEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{Arrays.toString(sortedVals[i]), Arrays.toString(decoded), ord}), (byte[])sortedVals[i], (byte[])decoded);
            }
        }
    }

    @Test
    public void testBlobCopy() {
        byte[][] vals = new byte[][]{"".getBytes(), "foo".getBytes(), "foobarbazbub".getBytes(), {-86, -86, -86, -86, -86, -86, -86, -86, -86, -86, -86, -86}, {85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85}};
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            for (byte[] val : vals) {
                int expectedLen = val.length + (Order.ASCENDING == ord ? 1 : 2);
                byte[] a = new byte[expectedLen + 3];
                SimplePositionedMutableByteRange buf1 = new SimplePositionedMutableByteRange(a, 1, expectedLen + 1);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)expectedLen, (long)OrderedBytes.encodeBlobCopy(buf1, val, ord));
                Assert.assertEquals((String)"Broken test: serialization did not consume entire buffer.", (long)buf1.getLength(), (long)buf1.getPosition());
                Assert.assertEquals((String)"Surprising serialized length.", (long)expectedLen, (long)(buf1.getPosition() - 1));
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[0]);
                Assert.assertEquals((String)"Buffer underflow.", (long)0L, (long)a[1]);
                Assert.assertEquals((String)"Buffer overflow.", (long)0L, (long)a[a.length - 1]);
                buf1.setPosition(1);
                Assert.assertEquals((String)"Surprising return value.", (long)expectedLen, (long)OrderedBytes.skip(buf1));
                Assert.assertEquals((String)"Did not skip enough bytes.", (long)expectedLen, (long)(buf1.getPosition() - 1));
                buf1.setPosition(1);
                Assert.assertArrayEquals((String)"Deserialization failed.", (byte[])val, (byte[])OrderedBytes.decodeBlobCopy(buf1));
                Assert.assertEquals((String)"Did not consume enough bytes.", (long)expectedLen, (long)(buf1.getPosition() - 1));
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[][] encoded = new byte[vals.length][];
            SimplePositionedMutableByteRange pbr = new SimplePositionedMutableByteRange();
            for (int i = 0; i < vals.length; ++i) {
                encoded[i] = new byte[vals[i].length + (Order.ASCENDING == ord ? 1 : 2)];
                OrderedBytes.encodeBlobCopy(pbr.set(encoded[i]), vals[i], ord);
            }
            Arrays.sort(encoded, Bytes.BYTES_COMPARATOR);
            byte[][] sortedVals = (byte[][])Arrays.copyOf(vals, vals.length);
            if (ord == Order.ASCENDING) {
                Arrays.sort(sortedVals, Bytes.BYTES_COMPARATOR);
            } else {
                Arrays.sort(sortedVals, Collections.reverseOrder(Bytes.BYTES_COMPARATOR));
            }
            for (int i = 0; i < sortedVals.length; ++i) {
                pbr.set(encoded[i]);
                byte[] decoded = OrderedBytes.decodeBlobCopy(pbr);
                Assert.assertArrayEquals((String)String.format("Encoded representations do not preserve natural order: <%s>, <%s>, %s", new Object[]{Arrays.toString(sortedVals[i]), Arrays.toString(decoded), ord}), (byte[])sortedVals[i], (byte[])decoded);
            }
        }
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            byte[] a = new byte[3 + (Order.ASCENDING == ord ? 1 : 2) + 2];
            SimplePositionedMutableByteRange buf = new SimplePositionedMutableByteRange(a, 1, 3 + (Order.ASCENDING == ord ? 1 : 2));
            OrderedBytes.encodeBlobCopy(buf, "foobarbaz".getBytes(), 3, 3, ord);
            buf.setPosition(0);
            Assert.assertArrayEquals((byte[])"bar".getBytes(), (byte[])OrderedBytes.decodeBlobCopy(buf));
        }
    }

    @Test(expected=IllegalArgumentException.class)
    public void testBlobCopyNoZeroBytes() {
        byte[] val = new byte[]{1, 2, 0, 3};
        byte[] ascExpected = new byte[]{56, 1, 2, 0, 3};
        SimplePositionedMutableByteRange buf = new SimplePositionedMutableByteRange(val.length + 1);
        OrderedBytes.encodeBlobCopy(buf, val, Order.ASCENDING);
        Assert.assertArrayEquals((byte[])ascExpected, (byte[])buf.getBytes());
        buf.set(val.length + 2);
        OrderedBytes.encodeBlobCopy(buf, val, Order.DESCENDING);
        Assert.fail((String)"test should never get here.");
    }

    @Test
    public void testSkip() {
        BigDecimal longMax = BigDecimal.valueOf(Long.MAX_VALUE);
        double negInf = Double.NEGATIVE_INFINITY;
        BigDecimal negLarge = longMax.multiply(longMax).negate();
        BigDecimal negMed = new BigDecimal("-10.0");
        BigDecimal negSmall = new BigDecimal("-0.0010");
        long zero = 0L;
        BigDecimal posSmall = negSmall.negate();
        BigDecimal posMed = negMed.negate();
        BigDecimal posLarge = negLarge.negate();
        double posInf = Double.POSITIVE_INFINITY;
        double nan = Double.NaN;
        byte int8 = 100;
        short int16 = 100;
        int int32 = 100;
        long int64 = 100L;
        float float32 = 100.0f;
        double float64 = 100.0;
        String text = "hello world.";
        byte[] blobVar = Bytes.toBytes("foo");
        byte[] blobCopy = Bytes.toBytes("bar");
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            SimplePositionedMutableByteRange buff = new SimplePositionedMutableByteRange(30);
            int o = OrderedBytes.encodeNull(buff, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, negInf, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, negLarge, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, negMed, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, negSmall, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, zero, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, posSmall, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, posMed, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, posLarge, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, posInf, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, nan, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeInt8(buff, int8, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeInt16(buff, int16, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeInt32(buff, int32, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeInt64(buff, int64, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeFloat32(buff, float32, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeFloat64(buff, float64, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeString(buff, text, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.setPosition(0);
            o = OrderedBytes.encodeBlobVar(buff, blobVar, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
            buff.set(blobCopy.length + (Order.ASCENDING == ord ? 1 : 2));
            o = OrderedBytes.encodeBlobCopy(buff, blobCopy, ord);
            buff.setPosition(0);
            Assert.assertEquals((long)o, (long)OrderedBytes.skip(buff));
        }
    }

    @Test
    public void testEncodedValueCheck() {
        BigDecimal longMax = BigDecimal.valueOf(Long.MAX_VALUE);
        double negInf = Double.NEGATIVE_INFINITY;
        BigDecimal negLarge = longMax.multiply(longMax).negate();
        BigDecimal negMed = new BigDecimal("-10.0");
        BigDecimal negSmall = new BigDecimal("-0.0010");
        long zero = 0L;
        BigDecimal posSmall = negSmall.negate();
        BigDecimal posMed = negMed.negate();
        BigDecimal posLarge = negLarge.negate();
        double posInf = Double.POSITIVE_INFINITY;
        double nan = Double.NaN;
        byte int8 = 100;
        short int16 = 100;
        int int32 = 100;
        long int64 = 100L;
        float float32 = 100.0f;
        double float64 = 100.0;
        String text = "hello world.";
        byte[] blobVar = Bytes.toBytes("foo");
        int cnt = 0;
        SimplePositionedMutableByteRange buff = new SimplePositionedMutableByteRange(1024);
        for (Order ord : new Order[]{Order.ASCENDING, Order.DESCENDING}) {
            int o = OrderedBytes.encodeNull(buff, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, negInf, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, negLarge, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, negMed, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, negSmall, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, zero, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, posSmall, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, posMed, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, posLarge, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, posInf, ord);
            ++cnt;
            o = OrderedBytes.encodeNumeric((PositionedByteRange)buff, nan, ord);
            ++cnt;
            o = OrderedBytes.encodeInt8(buff, int8, ord);
            ++cnt;
            o = OrderedBytes.encodeInt16(buff, int16, ord);
            ++cnt;
            o = OrderedBytes.encodeInt32(buff, int32, ord);
            ++cnt;
            o = OrderedBytes.encodeInt64(buff, int64, ord);
            ++cnt;
            o = OrderedBytes.encodeFloat32(buff, float32, ord);
            ++cnt;
            o = OrderedBytes.encodeFloat64(buff, float64, ord);
            ++cnt;
            o = OrderedBytes.encodeString(buff, text, ord);
            ++cnt;
            o = OrderedBytes.encodeBlobVar(buff, blobVar, ord);
            ++cnt;
        }
        buff.setPosition(0);
        Assert.assertEquals((long)OrderedBytes.length(buff), (long)cnt);
        for (int i = 0; i < cnt; ++i) {
            Assert.assertEquals((Object)OrderedBytes.isEncodedValue(buff), (Object)true);
            OrderedBytes.skip(buff);
        }
    }
}

