/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.table;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hudi.common.config.SerializableConfiguration;
import org.apache.hudi.common.engine.HoodieEngineContext;
import org.apache.hudi.common.fs.FSUtils;
import org.apache.hudi.common.model.IOType;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.table.HoodieTable;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class MarkerFiles
implements Serializable {
    private static final Logger LOG = LogManager.getLogger(MarkerFiles.class);
    private final String instantTime;
    private final transient FileSystem fs;
    private final transient Path markerDirPath;
    private final String basePath;

    public MarkerFiles(FileSystem fs, String basePath, String markerFolderPath, String instantTime) {
        this.instantTime = instantTime;
        this.fs = fs;
        this.markerDirPath = new Path(markerFolderPath);
        this.basePath = basePath;
    }

    public MarkerFiles(HoodieTable table, String instantTime) {
        this(table.getMetaClient().getFs(), table.getMetaClient().getBasePath(), table.getMetaClient().getMarkerFolderPath(instantTime), instantTime);
    }

    public void quietDeleteMarkerDir(HoodieEngineContext context, int parallelism) {
        try {
            this.deleteMarkerDir(context, parallelism);
        }
        catch (HoodieIOException ioe) {
            LOG.warn((Object)("Error deleting marker directory for instant " + this.instantTime), (Throwable)ioe);
        }
    }

    public boolean deleteMarkerDir(HoodieEngineContext context, int parallelism) {
        try {
            if (this.fs.exists(this.markerDirPath)) {
                FileStatus[] fileStatuses = this.fs.listStatus(this.markerDirPath);
                List markerDirSubPaths = Arrays.stream(fileStatuses).map(fileStatus -> fileStatus.getPath().toString()).collect(Collectors.toList());
                if (markerDirSubPaths.size() > 0) {
                    SerializableConfiguration conf = new SerializableConfiguration(this.fs.getConf());
                    parallelism = Math.min(markerDirSubPaths.size(), parallelism);
                    context.foreach(markerDirSubPaths, subPathStr -> {
                        Path subPath = new Path(subPathStr);
                        FileSystem fileSystem = subPath.getFileSystem(conf.get());
                        fileSystem.delete(subPath, true);
                    }, parallelism);
                }
                boolean result = this.fs.delete(this.markerDirPath, true);
                LOG.info((Object)("Removing marker directory at " + this.markerDirPath));
                return result;
            }
        }
        catch (IOException ioe) {
            throw new HoodieIOException(ioe.getMessage(), ioe);
        }
        return false;
    }

    public boolean doesMarkerDirExist() throws IOException {
        return this.fs.exists(this.markerDirPath);
    }

    public Set<String> createdAndMergedDataPaths(HoodieEngineContext context, int parallelism) throws IOException {
        HashSet<String> dataFiles = new HashSet<String>();
        FileStatus[] topLevelStatuses = this.fs.listStatus(this.markerDirPath);
        ArrayList<String> subDirectories = new ArrayList<String>();
        for (FileStatus topLevelStatus : topLevelStatuses) {
            if (topLevelStatus.isFile()) {
                String pathStr = topLevelStatus.getPath().toString();
                if (!pathStr.contains(".marker") || pathStr.endsWith(IOType.APPEND.name())) continue;
                dataFiles.add(this.translateMarkerToDataPath(pathStr));
                continue;
            }
            subDirectories.add(topLevelStatus.getPath().toString());
        }
        if (subDirectories.size() > 0) {
            parallelism = Math.min(subDirectories.size(), parallelism);
            SerializableConfiguration serializedConf = new SerializableConfiguration(this.fs.getConf());
            context.setJobStatus(this.getClass().getSimpleName(), "Obtaining marker files for all created, merged paths");
            dataFiles.addAll(context.flatMap(subDirectories, directory -> {
                Path path = new Path(directory);
                FileSystem fileSystem = path.getFileSystem(serializedConf.get());
                RemoteIterator itr = fileSystem.listFiles(path, true);
                ArrayList<String> result = new ArrayList<String>();
                while (itr.hasNext()) {
                    FileStatus status = (FileStatus)itr.next();
                    String pathStr = status.getPath().toString();
                    if (!pathStr.contains(".marker") || pathStr.endsWith(IOType.APPEND.name())) continue;
                    result.add(this.translateMarkerToDataPath(pathStr));
                }
                return result.stream();
            }, parallelism));
        }
        return dataFiles;
    }

    private String translateMarkerToDataPath(String markerPath) {
        String rPath = this.stripMarkerFolderPrefix(markerPath);
        return MarkerFiles.stripMarkerSuffix(rPath);
    }

    public static String stripMarkerSuffix(String path) {
        return path.substring(0, path.indexOf(".marker"));
    }

    public List<String> allMarkerFilePaths() throws IOException {
        ArrayList<String> markerFiles = new ArrayList<String>();
        FSUtils.processFiles(this.fs, this.markerDirPath.toString(), fileStatus -> {
            markerFiles.add(this.stripMarkerFolderPrefix(fileStatus.getPath().toString()));
            return true;
        }, false);
        return markerFiles;
    }

    private String stripMarkerFolderPrefix(String fullMarkerPath) {
        ValidationUtils.checkArgument(fullMarkerPath.contains(".marker"));
        String markerRootPath = Path.getPathWithoutSchemeAndAuthority((Path)new Path(String.format("%s/%s/%s", this.basePath, HoodieTableMetaClient.TEMPFOLDER_NAME, this.instantTime))).toString();
        int begin = fullMarkerPath.indexOf(markerRootPath);
        ValidationUtils.checkArgument(begin >= 0, "Not in marker dir. Marker Path=" + fullMarkerPath + ", Expected Marker Root=" + markerRootPath);
        return fullMarkerPath.substring(begin + markerRootPath.length() + 1);
    }

    public Path create(String partitionPath, String dataFileName, IOType type) {
        Path path = FSUtils.getPartitionPath(this.markerDirPath, partitionPath);
        try {
            this.fs.mkdirs(path);
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to make dir " + path, e);
        }
        String markerFileName = String.format("%s%s.%s", dataFileName, ".marker", type.name());
        Path markerPath = new Path(path, markerFileName);
        try {
            LOG.info((Object)("Creating Marker Path=" + markerPath));
            this.fs.create(markerPath, false).close();
        }
        catch (IOException e) {
            throw new HoodieException("Failed to create marker file " + markerPath, e);
        }
        return markerPath;
    }
}

