/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.util.collection;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hudi.common.util.FileIOUtils;
import org.apache.hudi.common.util.HoodieTimer;
import org.apache.hudi.common.util.SerializationUtils;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.rocksdb.AbstractImmutableNativeReference;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.DBOptions;
import org.rocksdb.InfoLogLevel;
import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.Statistics;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;

public class RocksDBDAO {
    private static final Logger LOG = LogManager.getLogger(RocksDBDAO.class);
    private transient ConcurrentHashMap<String, ColumnFamilyHandle> managedHandlesMap;
    private transient ConcurrentHashMap<String, ColumnFamilyDescriptor> managedDescriptorMap;
    private transient RocksDB rocksDB;
    private boolean closed = false;
    private final String rocksDBBasePath;
    private long totalBytesWritten;

    public RocksDBDAO(String basePath, String rocksDBBasePath) {
        this.rocksDBBasePath = String.format("%s/%s/%s", rocksDBBasePath, basePath.replace("/", "_"), UUID.randomUUID().toString());
        this.init();
        this.totalBytesWritten = 0L;
    }

    private RocksDB getRocksDB() {
        return this.rocksDB;
    }

    private void init() {
        try {
            LOG.info((Object)("DELETING RocksDB persisted at " + this.rocksDBBasePath));
            FileIOUtils.deleteDirectory(new File(this.rocksDBBasePath));
            this.managedHandlesMap = new ConcurrentHashMap();
            this.managedDescriptorMap = new ConcurrentHashMap();
            DBOptions dbOptions = new DBOptions().setCreateIfMissing(true).setCreateMissingColumnFamilies(true).setWalDir(this.rocksDBBasePath).setStatsDumpPeriodSec(300).setStatistics(new Statistics());
            dbOptions.setLogger(new org.rocksdb.Logger(dbOptions){

                @Override
                protected void log(InfoLogLevel infoLogLevel, String logMsg) {
                    LOG.info((Object)("From Rocks DB : " + logMsg));
                }
            });
            List<ColumnFamilyDescriptor> managedColumnFamilies = this.loadManagedColumnFamilies(dbOptions);
            ArrayList<ColumnFamilyHandle> managedHandles = new ArrayList<ColumnFamilyHandle>();
            FileIOUtils.mkdir(new File(this.rocksDBBasePath));
            this.rocksDB = RocksDB.open(dbOptions, this.rocksDBBasePath, managedColumnFamilies, managedHandles);
            ValidationUtils.checkArgument(managedHandles.size() == managedColumnFamilies.size(), "Unexpected number of handles are returned");
            for (int index = 0; index < managedHandles.size(); ++index) {
                ColumnFamilyHandle handle = (ColumnFamilyHandle)managedHandles.get(index);
                ColumnFamilyDescriptor descriptor = managedColumnFamilies.get(index);
                String familyNameFromHandle = new String(handle.getName());
                String familyNameFromDescriptor = new String(descriptor.getName());
                ValidationUtils.checkArgument(familyNameFromDescriptor.equals(familyNameFromHandle), "Family Handles not in order with descriptors");
                this.managedHandlesMap.put(familyNameFromHandle, handle);
                this.managedDescriptorMap.put(familyNameFromDescriptor, descriptor);
            }
        }
        catch (IOException | RocksDBException re) {
            LOG.error((Object)"Got exception opening Rocks DB instance ", (Throwable)re);
            throw new HoodieException(re);
        }
    }

    private List<ColumnFamilyDescriptor> loadManagedColumnFamilies(DBOptions dbOptions) throws RocksDBException {
        ArrayList<ColumnFamilyDescriptor> managedColumnFamilies = new ArrayList<ColumnFamilyDescriptor>();
        Options options = new Options(dbOptions, new ColumnFamilyOptions());
        List<byte[]> existing = RocksDB.listColumnFamilies(options, this.rocksDBBasePath);
        if (existing.isEmpty()) {
            LOG.info((Object)"No column family found. Loading default");
            managedColumnFamilies.add(RocksDBDAO.getColumnFamilyDescriptor(RocksDB.DEFAULT_COLUMN_FAMILY));
        } else {
            LOG.info((Object)("Loading column families :" + existing.stream().map(String::new).collect(Collectors.toList())));
            managedColumnFamilies.addAll(existing.stream().map(RocksDBDAO::getColumnFamilyDescriptor).collect(Collectors.toList()));
        }
        return managedColumnFamilies;
    }

    private static ColumnFamilyDescriptor getColumnFamilyDescriptor(byte[] columnFamilyName) {
        return new ColumnFamilyDescriptor(columnFamilyName, new ColumnFamilyOptions());
    }

    public void writeBatch(BatchHandler handler) {
        try (WriteBatch batch = new WriteBatch();){
            handler.apply(batch);
            this.getRocksDB().write(new WriteOptions(), batch);
        }
        catch (RocksDBException re) {
            throw new HoodieException(re);
        }
    }

    public <T extends Serializable> void putInBatch(WriteBatch batch, String columnFamilyName, String key, T value) {
        try {
            byte[] payload = this.serializePayload(value);
            batch.put(this.managedHandlesMap.get(columnFamilyName), key.getBytes(), payload);
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <K extends Serializable, T extends Serializable> void putInBatch(WriteBatch batch, String columnFamilyName, K key, T value) {
        try {
            byte[] keyBytes = SerializationUtils.serialize(key);
            byte[] payload = this.serializePayload(value);
            batch.put(this.managedHandlesMap.get(columnFamilyName), keyBytes, payload);
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <T extends Serializable> void put(String columnFamilyName, String key, T value) {
        try {
            byte[] payload = this.serializePayload(value);
            this.getRocksDB().put(this.managedHandlesMap.get(columnFamilyName), key.getBytes(), payload);
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <K extends Serializable, T extends Serializable> void put(String columnFamilyName, K key, T value) {
        try {
            byte[] payload = this.serializePayload(value);
            this.getRocksDB().put(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key), payload);
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public void deleteInBatch(WriteBatch batch, String columnFamilyName, String key) {
        try {
            batch.delete(this.managedHandlesMap.get(columnFamilyName), key.getBytes());
        }
        catch (RocksDBException e) {
            throw new HoodieException(e);
        }
    }

    public <K extends Serializable> void deleteInBatch(WriteBatch batch, String columnFamilyName, K key) {
        try {
            batch.delete(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key));
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public void delete(String columnFamilyName, String key) {
        try {
            this.getRocksDB().delete(this.managedHandlesMap.get(columnFamilyName), key.getBytes());
        }
        catch (RocksDBException e) {
            throw new HoodieException(e);
        }
    }

    public <K extends Serializable> void delete(String columnFamilyName, K key) {
        try {
            this.getRocksDB().delete(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key));
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <T extends Serializable> T get(String columnFamilyName, String key) {
        ValidationUtils.checkArgument(!this.closed);
        try {
            byte[] val = this.getRocksDB().get(this.managedHandlesMap.get(columnFamilyName), key.getBytes());
            return (T)(val == null ? null : (Serializable)SerializationUtils.deserialize(val));
        }
        catch (RocksDBException e) {
            throw new HoodieException(e);
        }
    }

    public <K extends Serializable, T extends Serializable> T get(String columnFamilyName, K key) {
        ValidationUtils.checkArgument(!this.closed);
        try {
            byte[] val = this.getRocksDB().get(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key));
            return (T)(val == null ? null : (Serializable)SerializationUtils.deserialize(val));
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <T extends Serializable> Stream<Pair<String, T>> prefixSearch(String columnFamilyName, String prefix) {
        ValidationUtils.checkArgument(!this.closed);
        HoodieTimer timer = new HoodieTimer();
        timer.startTimer();
        long timeTakenMicro = 0L;
        LinkedList<Pair<String, Serializable>> results = new LinkedList<Pair<String, Serializable>>();
        try (RocksIterator it = this.getRocksDB().newIterator(this.managedHandlesMap.get(columnFamilyName));){
            it.seek(prefix.getBytes());
            while (it.isValid() && new String(it.key()).startsWith(prefix)) {
                long beginTs = System.nanoTime();
                Serializable val = (Serializable)SerializationUtils.deserialize(it.value());
                timeTakenMicro += (System.nanoTime() - beginTs) / 1000L;
                results.add(Pair.of(new String(it.key()), val));
                it.next();
            }
        }
        LOG.info((Object)("Prefix Search for (query=" + prefix + ") on " + columnFamilyName + ". Total Time Taken (msec)=" + timer.endTimer() + ". Serialization Time taken(micro)=" + timeTakenMicro + ", num entries=" + results.size()));
        return results.stream();
    }

    public <T extends Serializable> Iterator<T> iterator(String columnFamilyName) {
        return new IteratorWrapper(this.getRocksDB().newIterator(this.managedHandlesMap.get(columnFamilyName)));
    }

    public <T extends Serializable> void prefixDelete(String columnFamilyName, String prefix) {
        ValidationUtils.checkArgument(!this.closed);
        LOG.info((Object)("Prefix DELETE (query=" + prefix + ") on " + columnFamilyName));
        RocksIterator it = this.getRocksDB().newIterator(this.managedHandlesMap.get(columnFamilyName));
        it.seek(prefix.getBytes());
        String firstEntry = null;
        String lastEntry = null;
        while (it.isValid() && new String(it.key()).startsWith(prefix)) {
            String result = new String(it.key());
            it.next();
            if (firstEntry == null) {
                firstEntry = result;
            }
            lastEntry = result;
        }
        it.close();
        if (null != firstEntry) {
            try {
                this.getRocksDB().deleteRange(this.managedHandlesMap.get(columnFamilyName), firstEntry.getBytes(), lastEntry.getBytes());
                this.getRocksDB().delete(lastEntry.getBytes());
            }
            catch (RocksDBException e) {
                LOG.error((Object)"Got exception performing range delete");
                throw new HoodieException(e);
            }
        }
    }

    public void addColumnFamily(String columnFamilyName) {
        ValidationUtils.checkArgument(!this.closed);
        this.managedDescriptorMap.computeIfAbsent(columnFamilyName, colFamilyName -> {
            try {
                ColumnFamilyDescriptor descriptor = RocksDBDAO.getColumnFamilyDescriptor(colFamilyName.getBytes());
                ColumnFamilyHandle handle = this.getRocksDB().createColumnFamily(descriptor);
                this.managedHandlesMap.put((String)colFamilyName, handle);
                return descriptor;
            }
            catch (RocksDBException e) {
                throw new HoodieException(e);
            }
        });
    }

    public void dropColumnFamily(String columnFamilyName) {
        ValidationUtils.checkArgument(!this.closed);
        this.managedDescriptorMap.computeIfPresent(columnFamilyName, (colFamilyName, descriptor) -> {
            ColumnFamilyHandle handle = this.managedHandlesMap.get(colFamilyName);
            try {
                this.getRocksDB().dropColumnFamily(handle);
                handle.close();
            }
            catch (RocksDBException e) {
                throw new HoodieException(e);
            }
            this.managedHandlesMap.remove(columnFamilyName);
            return null;
        });
    }

    public synchronized void close() {
        if (!this.closed) {
            this.closed = true;
            this.managedHandlesMap.values().forEach(AbstractImmutableNativeReference::close);
            this.managedHandlesMap.clear();
            this.managedDescriptorMap.clear();
            this.getRocksDB().close();
            try {
                FileIOUtils.deleteDirectory(new File(this.rocksDBBasePath));
            }
            catch (IOException e) {
                throw new HoodieIOException(e.getMessage(), e);
            }
        }
    }

    public long getTotalBytesWritten() {
        return this.totalBytesWritten;
    }

    private <T extends Serializable> byte[] serializePayload(T value) throws IOException {
        byte[] payload = SerializationUtils.serialize(value);
        this.totalBytesWritten += (long)payload.length;
        return payload;
    }

    String getRocksDBBasePath() {
        return this.rocksDBBasePath;
    }

    public static interface BatchHandler {
        public void apply(WriteBatch var1);
    }

    private static class IteratorWrapper<R>
    implements Iterator<R> {
        private final RocksIterator iterator;

        public IteratorWrapper(RocksIterator iterator2) {
            this.iterator = iterator2;
            iterator2.seekToFirst();
        }

        @Override
        public boolean hasNext() {
            return this.iterator.isValid();
        }

        @Override
        public R next() {
            if (!this.hasNext()) {
                throw new IllegalStateException("next() called on rocksDB with no more valid entries");
            }
            Object val = SerializationUtils.deserialize(this.iterator.value());
            this.iterator.next();
            return (R)val;
        }
    }
}

