/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.fs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.avro.model.HoodieFileStatus;
import org.apache.hudi.avro.model.HoodiePath;
import org.apache.hudi.common.config.HoodieMetadataConfig;
import org.apache.hudi.common.engine.HoodieEngineContext;
import org.apache.hudi.common.model.HoodieFileFormat;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.table.HoodieTableConfig;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.collection.ImmutablePair;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.HoodieValidationException;
import org.apache.hudi.exception.InvalidHoodieFileNameException;
import org.apache.hudi.exception.InvalidHoodiePathException;
import org.apache.hudi.metadata.HoodieTableMetadata;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.HoodieStorageUtils;
import org.apache.hudi.storage.StorageConfiguration;
import org.apache.hudi.storage.StoragePath;
import org.apache.hudi.storage.StoragePathFilter;
import org.apache.hudi.storage.StoragePathInfo;
import org.apache.hudi.storage.inline.InLineFSUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FSUtils {
    private static final Logger LOG = LoggerFactory.getLogger(FSUtils.class);
    public static final String PATH_SEPARATOR = "/";
    public static final Pattern LOG_FILE_PATTERN = Pattern.compile("^\\.(.+)_(.*)\\.(log|archive)\\.(\\d+)(_((\\d+)-(\\d+)-(\\d+))(.cdc)?)?");
    public static final Pattern PREFIX_BY_FILE_ID_PATTERN = Pattern.compile("^(.+)-(\\d+)");
    private static final Pattern BASE_FILE_PATTERN = Pattern.compile("[a-zA-Z0-9-]+_[a-zA-Z0-9-]+_[0-9]+\\.[a-zA-Z0-9]+");
    private static final String LOG_FILE_EXTENSION = ".log";
    private static final StoragePathFilter ALLOW_ALL_FILTER = file -> true;

    public static boolean isTableExists(String path, HoodieStorage storage2) throws IOException {
        return storage2.exists(new StoragePath(path, ".hoodie"));
    }

    public static StoragePath makeQualified(HoodieStorage storage2, StoragePath path) {
        return path.makeQualified(storage2.getUri());
    }

    public static String makeWriteToken(int taskPartitionId, int stageId, long taskAttemptId) {
        return String.format("%d-%d-%d", taskPartitionId, stageId, taskAttemptId);
    }

    public static String makeBaseFileName(String instantTime, String writeToken, String fileId, String fileExtension2) {
        return String.format("%s_%s_%s%s", fileId, writeToken, instantTime, fileExtension2);
    }

    public static String makeBootstrapIndexFileName(String instantTime, String fileId, String ext) {
        return String.format("%s_%s_%s%s", fileId, "1-0-1", instantTime, ext);
    }

    public static String maskWithoutFileId(String instantTime, int taskPartitionId) {
        return String.format("*_%s_%s%s", taskPartitionId, instantTime, HoodieTableConfig.BASE_FILE_FORMAT.defaultValue().getFileExtension());
    }

    public static String getCommitTime(String fullFileName) {
        try {
            if (FSUtils.isLogFile(fullFileName)) {
                return fullFileName.split("_")[1].split("\\.", 2)[0];
            }
            return fullFileName.split("_")[2].split("\\.", 2)[0];
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new HoodieException("Failed to get commit time from filename: " + fullFileName, e);
        }
    }

    public static String getCommitTimeWithFullPath(String path) {
        String fullFileName = path.contains(PATH_SEPARATOR) ? path.substring(path.lastIndexOf(PATH_SEPARATOR) + 1) : path;
        return FSUtils.getCommitTime(fullFileName);
    }

    public static long getFileSize(HoodieStorage storage2, StoragePath path) throws IOException {
        return storage2.getPathInfo(path).getLength();
    }

    public static String getFileId(String fullFileName) {
        return fullFileName.split("_", 2)[0];
    }

    public static String getFileNameFromPath(String filePath) {
        return filePath.substring(filePath.lastIndexOf(PATH_SEPARATOR) + 1);
    }

    public static List<String> getAllPartitionFoldersThreeLevelsDown(HoodieStorage storage2, String basePath) throws IOException {
        ArrayList<String> datePartitions = new ArrayList<String>();
        StoragePathFilter filter = FSUtils.getExcludeMetaPathFilter();
        List<StoragePathInfo> folders = storage2.globEntries(new StoragePath(basePath + "/*/*/*"), filter);
        for (StoragePathInfo pathInfo : folders) {
            StoragePath path = pathInfo.getPath();
            datePartitions.add(String.format("%s/%s/%s", path.getParent().getParent().getName(), path.getParent().getName(), path.getName()));
        }
        return datePartitions;
    }

    public static String getRelativePartitionPath(StoragePath basePath, StoragePath fullPartitionPath) {
        String basePathString;
        basePath = FSUtils.getPathWithoutSchemeAndAuthority(basePath);
        String fullPartitionPathStr = (fullPartitionPath = FSUtils.getPathWithoutSchemeAndAuthority(fullPartitionPath)).toString();
        if (!fullPartitionPathStr.startsWith(basePathString = basePath.toString())) {
            throw new IllegalArgumentException("Partition path \"" + fullPartitionPathStr + "\" does not belong to base-path \"" + basePath + "\"");
        }
        return fullPartitionPathStr.length() == basePathString.length() ? "" : fullPartitionPathStr.substring(basePathString.length() + 1);
    }

    public static StoragePath getPathWithoutSchemeAndAuthority(StoragePath path) {
        return path.getPathWithoutSchemeAndAuthority();
    }

    public static void processFiles(HoodieStorage storage2, String basePathStr, Function<StoragePathInfo, Boolean> consumer, boolean excludeMetaFolder) throws IOException {
        StoragePathFilter pathFilter = excludeMetaFolder ? FSUtils.getExcludeMetaPathFilter() : ALLOW_ALL_FILTER;
        List<StoragePathInfo> topLevelInfoList = storage2.listDirectEntries(new StoragePath(basePathStr));
        for (StoragePathInfo child : topLevelInfoList) {
            if (child.isFile()) {
                boolean success = consumer.apply(child);
                if (success) continue;
                throw new HoodieException("Failed to process file-status=" + child);
            }
            if (!pathFilter.accept(child.getPath())) continue;
            List<StoragePathInfo> list = storage2.listFiles(child.getPath());
            for (StoragePathInfo pathInfo : list) {
                boolean success = consumer.apply(pathInfo);
                if (success) continue;
                throw new HoodieException("Failed to process StoragePathInfo=" + pathInfo);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static List<String> getAllPartitionPaths(HoodieEngineContext engineContext, HoodieStorage storage2, String basePathStr, boolean useFileListingFromMetadata) {
        HoodieMetadataConfig metadataConfig = HoodieMetadataConfig.newBuilder().enable(useFileListingFromMetadata).build();
        try (HoodieTableMetadata tableMetadata = HoodieTableMetadata.create(engineContext, storage2, metadataConfig, basePathStr);){
            List<String> list = tableMetadata.getAllPartitionPaths();
            return list;
        }
        catch (Exception e) {
            throw new HoodieException("Error fetching partition paths from metadata table", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static List<String> getAllPartitionPaths(HoodieEngineContext engineContext, HoodieStorage storage2, HoodieMetadataConfig metadataConfig, String basePathStr) {
        try (HoodieTableMetadata tableMetadata = HoodieTableMetadata.create(engineContext, storage2, metadataConfig, basePathStr);){
            List<String> list = tableMetadata.getAllPartitionPaths();
            return list;
        }
        catch (Exception e) {
            throw new HoodieException("Error fetching partition paths from metadata table", e);
        }
    }

    public static List<String> getAllPartitionPaths(HoodieEngineContext engineContext, HoodieStorage storage2, StoragePath basePath, boolean useFileListingFromMetadata) {
        return FSUtils.getAllPartitionPaths(engineContext, storage2, basePath.toString(), useFileListingFromMetadata);
    }

    public static List<String> getAllPartitionPaths(HoodieEngineContext engineContext, HoodieStorage storage2, HoodieMetadataConfig metadataConfig, StoragePath basePath) {
        return FSUtils.getAllPartitionPaths(engineContext, storage2, metadataConfig, basePath.toString());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Map<String, List<StoragePathInfo>> getFilesInPartitions(HoodieEngineContext engineContext, HoodieStorage storage2, HoodieMetadataConfig metadataConfig, String basePathStr, String[] partitionPaths) {
        try (HoodieTableMetadata tableMetadata = HoodieTableMetadata.create(engineContext, storage2, metadataConfig, basePathStr);){
            Map<String, List<StoragePathInfo>> map2 = tableMetadata.getAllFilesInPartitions(Arrays.asList(partitionPaths));
            return map2;
        }
        catch (Exception ex) {
            throw new HoodieException("Error get files in partitions: " + String.join((CharSequence)",", partitionPaths), ex);
        }
    }

    public static String getFileExtension(String fullName) {
        Objects.requireNonNull(fullName);
        String fileName = new File(fullName).getName();
        int dotIndex = fileName.lastIndexOf(46);
        return dotIndex == -1 ? "" : fileName.substring(dotIndex);
    }

    private static StoragePathFilter getExcludeMetaPathFilter() {
        return path -> !path.toString().contains(".hoodie");
    }

    public static String createNewFileIdPfx() {
        return UUID.randomUUID().toString();
    }

    public static String getFileIdPfxFromFileId(String fileId) {
        Matcher matcher = PREFIX_BY_FILE_ID_PATTERN.matcher(fileId);
        if (!matcher.find()) {
            throw new HoodieValidationException("Failed to get prefix from " + fileId);
        }
        return matcher.group(1);
    }

    public static String createNewFileId(String idPfx, int id) {
        return String.format("%s-%d", idPfx, id);
    }

    public static String getFileExtensionFromLog(StoragePath logPath) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(logPath.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(logPath.toString(), "LogFile");
        }
        return matcher.group(3);
    }

    public static String getFileIdFromFileName(String fileName) {
        if (FSUtils.isLogFile(fileName)) {
            Matcher matcher = LOG_FILE_PATTERN.matcher(fileName);
            if (!matcher.find()) {
                throw new InvalidHoodieFileNameException(fileName, "LogFile");
            }
            return matcher.group(1);
        }
        return FSUtils.getFileId(fileName);
    }

    public static String getFileIdFromLogPath(StoragePath path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        return matcher.group(1);
    }

    public static String getFileIdFromFilePath(StoragePath filePath) {
        if (FSUtils.isLogFile(filePath)) {
            return FSUtils.getFileIdFromLogPath(filePath);
        }
        return FSUtils.getFileId(filePath.getName());
    }

    public static String getDeltaCommitTimeFromLogPath(StoragePath path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path.toString(), "LogFile");
        }
        return matcher.group(2);
    }

    public static Integer getTaskPartitionIdFromLogPath(StoragePath path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path.toString(), "LogFile");
        }
        String val = matcher.group(7);
        return val == null ? null : Integer.valueOf(Integer.parseInt(val));
    }

    public static String getWriteTokenFromLogPath(StoragePath path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path.toString(), "LogFile");
        }
        return matcher.group(6);
    }

    public static Integer getStageIdFromLogPath(StoragePath path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path.toString(), "LogFile");
        }
        String val = matcher.group(8);
        return val == null ? null : Integer.valueOf(Integer.parseInt(val));
    }

    public static Integer getTaskAttemptIdFromLogPath(StoragePath path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path.toString(), "LogFile");
        }
        String val = matcher.group(9);
        return val == null ? null : Integer.valueOf(Integer.parseInt(val));
    }

    public static int getFileVersionFromLog(StoragePath logPath) {
        return FSUtils.getFileVersionFromLog(logPath.getName());
    }

    public static int getFileVersionFromLog(String logFileName) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(logFileName);
        if (!matcher.find()) {
            throw new HoodieIOException("Invalid log file name: " + logFileName);
        }
        return Integer.parseInt(matcher.group(4));
    }

    public static String makeLogFileName(String fileId, String logFileExtension, String deltaCommitTime, int version, String writeToken) {
        String suffix = writeToken == null ? String.format("%s_%s%s.%d", fileId, deltaCommitTime, logFileExtension, version) : String.format("%s_%s%s.%d_%s", fileId, deltaCommitTime, logFileExtension, version, writeToken);
        return "." + suffix;
    }

    public static boolean isBaseFile(StoragePath path) {
        String extension = FSUtils.getFileExtension(path.getName());
        if (HoodieFileFormat.BASE_FILE_EXTENSIONS.contains(extension)) {
            return BASE_FILE_PATTERN.matcher(path.getName()).matches();
        }
        return false;
    }

    public static boolean isLogFile(StoragePath logPath) {
        String scheme2 = logPath.toUri().getScheme();
        return FSUtils.isLogFile("inlinefs".equals(scheme2) ? InLineFSUtils.getOuterFilePathFromInlinePath(logPath).getName() : logPath.getName());
    }

    public static boolean isLogFile(String fileName) {
        if (fileName.contains(LOG_FILE_EXTENSION)) {
            Matcher matcher = LOG_FILE_PATTERN.matcher(fileName);
            return matcher.find();
        }
        return false;
    }

    public static boolean isDataFile(StoragePath path) {
        return FSUtils.isBaseFile(path) || FSUtils.isLogFile(path);
    }

    public static List<StoragePathInfo> getAllDataFilesInPartition(HoodieStorage storage2, StoragePath partitionPath) throws IOException {
        Set validFileExtensions = Arrays.stream(HoodieFileFormat.values()).map(HoodieFileFormat::getFileExtension).collect(Collectors.toCollection(HashSet::new));
        String logFileExtension = HoodieFileFormat.HOODIE_LOG.getFileExtension();
        try {
            return storage2.listDirectEntries(partitionPath, path -> {
                String extension = FSUtils.getFileExtension(path.getName());
                return validFileExtensions.contains(extension) || path.getName().contains(logFileExtension);
            }).stream().filter(StoragePathInfo::isFile).collect(Collectors.toList());
        }
        catch (IOException e) {
            if (!storage2.exists(partitionPath)) {
                return Collections.emptyList();
            }
            throw e;
        }
    }

    public static Option<HoodieLogFile> getLatestLogFile(HoodieStorage storage2, StoragePath partitionPath, String fileId, String logFileExtension, String deltaCommitTime) throws IOException {
        return FSUtils.getLatestLogFile(FSUtils.getAllLogFiles(storage2, partitionPath, fileId, logFileExtension, deltaCommitTime));
    }

    public static Stream<HoodieLogFile> getAllLogFiles(HoodieStorage storage2, StoragePath partitionPath, String fileId, String logFileExtension, String deltaCommitTime) throws IOException {
        try {
            StoragePathFilter pathFilter = path -> path.getName().startsWith("." + fileId) && path.getName().contains(logFileExtension);
            return storage2.listDirectEntries(partitionPath, pathFilter).stream().map(HoodieLogFile::new).filter(s -> s.getDeltaCommitTime().equals(deltaCommitTime));
        }
        catch (FileNotFoundException e) {
            return Stream.of(new HoodieLogFile[0]);
        }
    }

    public static Option<Pair<Integer, String>> getLatestLogVersion(HoodieStorage storage2, StoragePath partitionPath, String fileId, String logFileExtension, String deltaCommitTime) throws IOException {
        Option<HoodieLogFile> latestLogFile = FSUtils.getLatestLogFile(FSUtils.getAllLogFiles(storage2, partitionPath, fileId, logFileExtension, deltaCommitTime));
        if (latestLogFile.isPresent()) {
            return Option.of(Pair.of(latestLogFile.get().getLogVersion(), latestLogFile.get().getLogWriteToken()));
        }
        return Option.empty();
    }

    public static void createPathIfNotExists(HoodieStorage storage2, StoragePath partitionPath) throws IOException {
        if (!storage2.exists(partitionPath)) {
            storage2.createDirectory(partitionPath);
        }
    }

    public static Long getSizeInMB(long sizeInBytes) {
        return sizeInBytes / 0x100000L;
    }

    public static StoragePath constructAbsolutePath(String basePath, String relativePartitionPath) {
        if (StringUtils.isNullOrEmpty(relativePartitionPath)) {
            return new StoragePath(basePath);
        }
        String properPartitionPath = relativePartitionPath.startsWith(PATH_SEPARATOR) ? relativePartitionPath.substring(1) : relativePartitionPath;
        return FSUtils.constructAbsolutePath(new StoragePath(basePath), properPartitionPath);
    }

    public static StoragePath constructAbsolutePath(StoragePath basePath, String relativePartitionPath) {
        return StringUtils.isNullOrEmpty(relativePartitionPath) ? basePath : new StoragePath(basePath, relativePartitionPath);
    }

    public static String getFileName(String filePathWithPartition, String partition) {
        int offset = StringUtils.isNullOrEmpty(partition) ? (filePathWithPartition.startsWith(PATH_SEPARATOR) ? 1 : 0) : partition.length() + 1;
        return filePathWithPartition.substring(offset);
    }

    public static List<StoragePathInfo> getGlobStatusExcludingMetaFolder(HoodieStorage storage2, StoragePath globPath) throws IOException {
        List<StoragePathInfo> statuses = storage2.globEntries(globPath);
        return statuses.stream().filter(fileStatus -> !fileStatus.getPath().toString().contains(".hoodie")).collect(Collectors.toList());
    }

    public static boolean deleteDir(HoodieEngineContext hoodieEngineContext, HoodieStorage storage2, StoragePath dirPath, int parallelism) {
        try {
            if (storage2.exists(dirPath)) {
                FSUtils.parallelizeSubPathProcess(hoodieEngineContext, storage2, dirPath, parallelism, e -> true, pairOfSubPathAndConf -> FSUtils.deleteSubPath((String)pairOfSubPathAndConf.getKey(), (StorageConfiguration)pairOfSubPathAndConf.getValue(), true));
                boolean result2 = storage2.deleteDirectory(dirPath);
                LOG.info("Removed directory at " + dirPath);
                return result2;
            }
        }
        catch (IOException ioe) {
            throw new HoodieIOException(ioe.getMessage(), ioe);
        }
        return false;
    }

    public static HoodiePath fromStoragePath(StoragePath path) {
        if (null == path) {
            return null;
        }
        return HoodiePath.newBuilder().setUri(path.toString()).build();
    }

    public static HoodieFileStatus fromPathInfo(StoragePathInfo pathInfo) {
        if (null == pathInfo) {
            return null;
        }
        HoodieFileStatus fStatus = new HoodieFileStatus();
        fStatus.setPath(FSUtils.fromStoragePath(pathInfo.getPath()));
        fStatus.setLength(pathInfo.getLength());
        fStatus.setIsDir(pathInfo.isDirectory());
        fStatus.setBlockReplication(Integer.valueOf(pathInfo.getBlockReplication()));
        fStatus.setBlockSize(pathInfo.getBlockSize());
        fStatus.setModificationTime(pathInfo.getModificationTime());
        fStatus.setAccessTime(pathInfo.getModificationTime());
        return fStatus;
    }

    public static <T> Map<String, T> parallelizeSubPathProcess(HoodieEngineContext hoodieEngineContext, HoodieStorage storage2, StoragePath dirPath, int parallelism, Predicate<StoragePathInfo> subPathPredicate, SerializableFunction<Pair<String, StorageConfiguration<?>>, T> pairFunction) {
        Map<Object, Object> result2 = new HashMap();
        try {
            List<StoragePathInfo> pathInfoList = storage2.listDirectEntries(dirPath);
            List<String> subPaths = pathInfoList.stream().filter(subPathPredicate).map(fileStatus -> fileStatus.getPath().toString()).collect(Collectors.toList());
            result2 = FSUtils.parallelizeFilesProcess(hoodieEngineContext, storage2, parallelism, pairFunction, subPaths);
        }
        catch (IOException ioe) {
            throw new HoodieIOException(ioe.getMessage(), ioe);
        }
        return result2;
    }

    public static <T> Map<String, T> parallelizeFilesProcess(HoodieEngineContext hoodieEngineContext, HoodieStorage storage2, int parallelism, SerializableFunction<Pair<String, StorageConfiguration<?>>, T> pairFunction, List<String> subPaths) {
        Map result2 = new HashMap();
        if (subPaths.size() > 0) {
            StorageConfiguration<?> storageConf = storage2.getConf();
            int actualParallelism = Math.min(subPaths.size(), parallelism);
            hoodieEngineContext.setJobStatus(FSUtils.class.getSimpleName(), "Parallel listing paths " + String.join((CharSequence)",", subPaths));
            result2 = hoodieEngineContext.mapToPair(subPaths, subPath -> new ImmutablePair((String)subPath, pairFunction.apply(new ImmutablePair<String, StorageConfiguration>((String)subPath, storageConf))), actualParallelism);
        }
        return result2;
    }

    public static boolean deleteSubPath(String subPathStr, StorageConfiguration<?> conf, boolean recursive) {
        try {
            StoragePath subPath = new StoragePath(subPathStr);
            HoodieStorage storage2 = HoodieStorageUtils.getStorage(subPath, conf);
            if (recursive) {
                return storage2.deleteDirectory(subPath);
            }
            return storage2.deleteFile(subPath);
        }
        catch (IOException e) {
            throw new HoodieIOException(e.getMessage(), e);
        }
    }

    public static List<StoragePathInfo> getAllDataPathInfo(HoodieStorage storage2, StoragePath path) throws IOException {
        ArrayList<StoragePathInfo> pathInfoList = new ArrayList<StoragePathInfo>();
        for (StoragePathInfo pathInfo : storage2.listDirectEntries(path)) {
            if (pathInfo.getPath().toString().contains(".hoodie")) continue;
            if (pathInfo.isDirectory()) {
                pathInfoList.addAll(FSUtils.getAllDataPathInfo(storage2, pathInfo.getPath()));
                continue;
            }
            pathInfoList.add(pathInfo);
        }
        return pathInfoList;
    }

    public static boolean comparePathsWithoutScheme(String pathStr1, String pathStr2) {
        Path pathWithoutScheme1 = FSUtils.getPathWithoutScheme(new Path(pathStr1));
        Path pathWithoutScheme2 = FSUtils.getPathWithoutScheme(new Path(pathStr2));
        return pathWithoutScheme1.equals((Object)pathWithoutScheme2);
    }

    public static Path getPathWithoutScheme(Path path) {
        return path.isUriPathAbsolute() ? new Path(null, path.toUri().getAuthority(), path.toUri().getPath()) : path;
    }

    private static Option<HoodieLogFile> getLatestLogFile(Stream<HoodieLogFile> logFiles) {
        return Option.fromJavaOptional(logFiles.min(HoodieLogFile.getReverseLogFileComparator()));
    }

    public static interface SerializableFunction<T, R>
    extends Function<T, R>,
    Serializable {
    }
}

