/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.util.collection;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hudi.common.util.FileIOUtils;
import org.apache.hudi.common.util.HoodieTimer;
import org.apache.hudi.common.util.SerializationUtils;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.rocksdb.AbstractImmutableNativeReference;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.DBOptions;
import org.rocksdb.InfoLogLevel;
import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.Statistics;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RocksDBDAO {
    private static final Logger LOG = LoggerFactory.getLogger(RocksDBDAO.class);
    private transient ConcurrentHashMap<String, ColumnFamilyHandle> managedHandlesMap;
    private transient ConcurrentHashMap<String, ColumnFamilyDescriptor> managedDescriptorMap;
    private transient RocksDB rocksDB;
    private boolean closed = false;
    private final String rocksDBBasePath;
    private long totalBytesWritten;

    public RocksDBDAO(String basePath, String rocksDBBasePath) {
        this.rocksDBBasePath = String.format("%s/%s/%s", rocksDBBasePath, URI.create(basePath).getPath().replace(":", "").replace("/", "_"), UUID.randomUUID().toString());
        this.init();
        this.totalBytesWritten = 0L;
    }

    private RocksDB getRocksDB() {
        return this.rocksDB;
    }

    private void init() {
        try {
            LOG.info("DELETING RocksDB persisted at " + this.rocksDBBasePath);
            FileIOUtils.deleteDirectory(new File(this.rocksDBBasePath));
            this.managedHandlesMap = new ConcurrentHashMap();
            this.managedDescriptorMap = new ConcurrentHashMap();
            DBOptions dbOptions = new DBOptions().setCreateIfMissing(true).setCreateMissingColumnFamilies(true).setWalDir(this.rocksDBBasePath).setStatsDumpPeriodSec(300).setStatistics(new Statistics());
            dbOptions.setLogger(new org.rocksdb.Logger(dbOptions){

                @Override
                protected void log(InfoLogLevel infoLogLevel, String logMsg) {
                    switch (infoLogLevel) {
                        case DEBUG_LEVEL: {
                            LOG.debug("From Rocks DB : {}", (Object)logMsg);
                            break;
                        }
                        case WARN_LEVEL: {
                            LOG.warn("From Rocks DB : {}", (Object)logMsg);
                            break;
                        }
                        case ERROR_LEVEL: 
                        case FATAL_LEVEL: {
                            LOG.error("From Rocks DB : {}", (Object)logMsg);
                            break;
                        }
                        default: {
                            LOG.info("From Rocks DB : {}", (Object)logMsg);
                        }
                    }
                }
            });
            List<ColumnFamilyDescriptor> managedColumnFamilies = this.loadManagedColumnFamilies(dbOptions);
            ArrayList<ColumnFamilyHandle> managedHandles = new ArrayList<ColumnFamilyHandle>(managedColumnFamilies.size());
            FileIOUtils.mkdir(new File(this.rocksDBBasePath));
            this.rocksDB = RocksDB.open(dbOptions, this.rocksDBBasePath, managedColumnFamilies, managedHandles);
            ValidationUtils.checkArgument(managedHandles.size() == managedColumnFamilies.size(), "Unexpected number of handles are returned");
            for (int index = 0; index < managedHandles.size(); ++index) {
                ColumnFamilyHandle handle2 = (ColumnFamilyHandle)managedHandles.get(index);
                ColumnFamilyDescriptor descriptor2 = managedColumnFamilies.get(index);
                String familyNameFromHandle = new String(handle2.getName());
                String familyNameFromDescriptor = new String(descriptor2.getName());
                ValidationUtils.checkArgument(familyNameFromDescriptor.equals(familyNameFromHandle), "Family Handles not in order with descriptors");
                this.managedHandlesMap.put(familyNameFromHandle, handle2);
                this.managedDescriptorMap.put(familyNameFromDescriptor, descriptor2);
            }
        }
        catch (IOException | RocksDBException re) {
            LOG.error("Got exception opening Rocks DB instance ", (Throwable)re);
            throw new HoodieException(re);
        }
    }

    private List<ColumnFamilyDescriptor> loadManagedColumnFamilies(DBOptions dbOptions) throws RocksDBException {
        ArrayList<ColumnFamilyDescriptor> managedColumnFamilies = new ArrayList<ColumnFamilyDescriptor>();
        Options options = new Options(dbOptions, new ColumnFamilyOptions());
        List<byte[]> existing = RocksDB.listColumnFamilies(options, this.rocksDBBasePath);
        if (existing.isEmpty()) {
            LOG.info("No column family found. Loading default");
            managedColumnFamilies.add(RocksDBDAO.getColumnFamilyDescriptor(RocksDB.DEFAULT_COLUMN_FAMILY));
        } else {
            LOG.info("Loading column families :" + existing.stream().map(String::new).collect(Collectors.toList()));
            managedColumnFamilies.addAll(existing.stream().map(RocksDBDAO::getColumnFamilyDescriptor).collect(Collectors.toList()));
        }
        return managedColumnFamilies;
    }

    private static ColumnFamilyDescriptor getColumnFamilyDescriptor(byte[] columnFamilyName) {
        return new ColumnFamilyDescriptor(columnFamilyName, new ColumnFamilyOptions());
    }

    public void writeBatch(BatchHandler handler) {
        try (WriteBatch batch = new WriteBatch();){
            handler.apply(batch);
            this.getRocksDB().write(new WriteOptions(), batch);
        }
        catch (RocksDBException re) {
            throw new HoodieException(re);
        }
    }

    public <T extends Serializable> void putInBatch(WriteBatch batch, String columnFamilyName, String key, T value) {
        try {
            byte[] payload = this.serializePayload(value);
            batch.put(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes(key), payload);
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <K extends Serializable, T extends Serializable> void putInBatch(WriteBatch batch, String columnFamilyName, K key, T value) {
        try {
            byte[] keyBytes = SerializationUtils.serialize(key);
            byte[] payload = this.serializePayload(value);
            batch.put(this.managedHandlesMap.get(columnFamilyName), keyBytes, payload);
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <T extends Serializable> void put(String columnFamilyName, String key, T value) {
        try {
            byte[] payload = this.serializePayload(value);
            this.getRocksDB().put(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes(key), payload);
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <K extends Serializable, T extends Serializable> void put(String columnFamilyName, K key, T value) {
        try {
            byte[] payload = this.serializePayload(value);
            this.getRocksDB().put(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key), payload);
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public void deleteInBatch(WriteBatch batch, String columnFamilyName, String key) {
        try {
            batch.delete(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes(key));
        }
        catch (RocksDBException e) {
            throw new HoodieException(e);
        }
    }

    public <K extends Serializable> void deleteInBatch(WriteBatch batch, String columnFamilyName, K key) {
        try {
            batch.delete(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key));
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public void delete(String columnFamilyName, String key) {
        try {
            this.getRocksDB().delete(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes(key));
        }
        catch (RocksDBException e) {
            throw new HoodieException(e);
        }
    }

    public <K extends Serializable> void delete(String columnFamilyName, K key) {
        try {
            this.getRocksDB().delete(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key));
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <T extends Serializable> T get(String columnFamilyName, String key) {
        return this.get(columnFamilyName, this.getKeyBytes(key));
    }

    public <K extends Serializable, T extends Serializable> T get(String columnFamilyName, K key) {
        return this.get(columnFamilyName, this.getKeyBytes(key));
    }

    public <K extends Serializable, T extends Serializable> T get(String columnFamilyName, byte[] key) {
        ValidationUtils.checkArgument(!this.closed);
        try {
            byte[] val = this.getRocksDB().get(this.managedHandlesMap.get(columnFamilyName), key);
            return (T)(val == null ? null : (Serializable)SerializationUtils.deserialize(val));
        }
        catch (Exception e) {
            throw new HoodieException(e);
        }
    }

    public <T extends Serializable> Stream<Pair<String, T>> prefixSearch(String columnFamilyName, String prefix) {
        ValidationUtils.checkArgument(!this.closed);
        HoodieTimer timer = HoodieTimer.start();
        long timeTakenMicro = 0L;
        LinkedList<Pair<String, Serializable>> results = new LinkedList<Pair<String, Serializable>>();
        try (RocksIterator it = this.getRocksDB().newIterator(this.managedHandlesMap.get(columnFamilyName));){
            it.seek(StringUtils.getUTF8Bytes(prefix));
            while (it.isValid() && new String(it.key()).startsWith(prefix)) {
                long beginTs = System.nanoTime();
                Serializable val = (Serializable)SerializationUtils.deserialize(it.value());
                timeTakenMicro += (System.nanoTime() - beginTs) / 1000L;
                results.add(Pair.of(new String(it.key()), val));
                it.next();
            }
        }
        LOG.info("Prefix Search for (query=" + prefix + ") on " + columnFamilyName + ". Total Time Taken (msec)=" + timer.endTimer() + ". Serialization Time taken(micro)=" + timeTakenMicro + ", num entries=" + results.size());
        return results.stream();
    }

    public <T extends Serializable, R extends Serializable> Iterator<Pair<T, R>> iterator(String columnFamilyName) {
        return new IteratorWrapper(this.getRocksDB().newIterator(this.managedHandlesMap.get(columnFamilyName)));
    }

    public <T extends Serializable> void prefixDelete(String columnFamilyName, String prefix) {
        ValidationUtils.checkArgument(!this.closed);
        LOG.info("Prefix DELETE (query=" + prefix + ") on " + columnFamilyName);
        RocksIterator it = this.getRocksDB().newIterator(this.managedHandlesMap.get(columnFamilyName));
        it.seek(StringUtils.getUTF8Bytes(prefix));
        String firstEntry = null;
        String lastEntry = null;
        while (it.isValid() && new String(it.key()).startsWith(prefix)) {
            String result2 = new String(it.key());
            it.next();
            if (firstEntry == null) {
                firstEntry = result2;
            }
            lastEntry = result2;
        }
        it.close();
        if (null != firstEntry) {
            try {
                this.getRocksDB().deleteRange(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes(firstEntry), StringUtils.getUTF8Bytes(lastEntry));
                this.getRocksDB().delete(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes(lastEntry));
            }
            catch (RocksDBException e) {
                LOG.error("Got exception performing range delete");
                throw new HoodieException(e);
            }
        }
    }

    public void addColumnFamily(String columnFamilyName) {
        ValidationUtils.checkArgument(!this.closed);
        this.managedDescriptorMap.computeIfAbsent(columnFamilyName, colFamilyName -> {
            try {
                ColumnFamilyDescriptor descriptor2 = RocksDBDAO.getColumnFamilyDescriptor(StringUtils.getUTF8Bytes(colFamilyName));
                ColumnFamilyHandle handle2 = this.getRocksDB().createColumnFamily(descriptor2);
                this.managedHandlesMap.put((String)colFamilyName, handle2);
                return descriptor2;
            }
            catch (RocksDBException e) {
                throw new HoodieException(e);
            }
        });
    }

    public void dropColumnFamily(String columnFamilyName) {
        ValidationUtils.checkArgument(!this.closed);
        this.managedDescriptorMap.computeIfPresent(columnFamilyName, (colFamilyName, descriptor2) -> {
            ColumnFamilyHandle handle2 = this.managedHandlesMap.get(colFamilyName);
            try {
                this.getRocksDB().dropColumnFamily(handle2);
                handle2.close();
            }
            catch (RocksDBException e) {
                throw new HoodieException(e);
            }
            this.managedHandlesMap.remove(columnFamilyName);
            return null;
        });
    }

    public synchronized void close() {
        if (!this.closed) {
            this.closed = true;
            this.managedHandlesMap.values().forEach(AbstractImmutableNativeReference::close);
            this.managedHandlesMap.clear();
            this.managedDescriptorMap.clear();
            this.getRocksDB().close();
            try {
                FileIOUtils.deleteDirectory(new File(this.rocksDBBasePath));
            }
            catch (IOException e) {
                throw new HoodieIOException(e.getMessage(), e);
            }
        }
    }

    public long getTotalBytesWritten() {
        return this.totalBytesWritten;
    }

    private <T extends Serializable> byte[] serializePayload(T value) throws IOException {
        byte[] payload = SerializationUtils.serialize(value);
        this.totalBytesWritten += (long)payload.length;
        return payload;
    }

    private byte[] getKeyBytes(String key) {
        return StringUtils.getUTF8Bytes(key);
    }

    private <K extends Serializable> byte[] getKeyBytes(K key) {
        try {
            return SerializationUtils.serialize(key);
        }
        catch (IOException e) {
            throw new HoodieException(e);
        }
    }

    String getRocksDBBasePath() {
        return this.rocksDBBasePath;
    }

    public static interface BatchHandler {
        public void apply(WriteBatch var1);
    }

    private static class IteratorWrapper<T, R>
    implements Iterator<Pair<T, R>> {
        private final RocksIterator iterator;

        public IteratorWrapper(RocksIterator iterator2) {
            this.iterator = iterator2;
            iterator2.seekToFirst();
        }

        @Override
        public boolean hasNext() {
            return this.iterator.isValid();
        }

        @Override
        public Pair<T, R> next() {
            if (!this.hasNext()) {
                throw new IllegalStateException("next() called on rocksDB with no more valid entries");
            }
            Object key = SerializationUtils.deserialize(this.iterator.key());
            Object val = SerializationUtils.deserialize(this.iterator.value());
            this.iterator.next();
            return Pair.of(key, val);
        }
    }
}

