/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.index.bucket;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.hudi.common.fs.FSUtils;
import org.apache.hudi.common.model.ConsistentHashingNode;
import org.apache.hudi.common.model.HoodieConsistentHashingMetadata;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.util.FileIOUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.HoodieIndexException;
import org.apache.hudi.index.bucket.ConsistentBucketIdentifier;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;
import org.apache.hudi.storage.StoragePathInfo;
import org.apache.hudi.table.HoodieTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsistentBucketIndexUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ConsistentBucketIndexUtils.class);

    public static HoodieConsistentHashingMetadata loadOrCreateMetadata(HoodieTable table, String partition, int numBuckets) {
        Option<HoodieConsistentHashingMetadata> metadataOption = ConsistentBucketIndexUtils.loadMetadata(table, partition);
        if (metadataOption.isPresent()) {
            return metadataOption.get();
        }
        LOG.info("Failed to load metadata, try to create one. Partition: {}.", (Object)partition);
        HoodieConsistentHashingMetadata metadata2 = new HoodieConsistentHashingMetadata(partition, numBuckets);
        if (ConsistentBucketIndexUtils.saveMetadata(table, metadata2)) {
            return metadata2;
        }
        metadataOption = ConsistentBucketIndexUtils.loadMetadata(table, partition);
        ValidationUtils.checkState(metadataOption.isPresent(), "Failed to load or create metadata, partition: " + partition);
        return metadataOption.get();
    }

    public static Option<HoodieConsistentHashingMetadata> loadMetadata(HoodieTable table, String partition) {
        HoodieTableMetaClient metaClient = table.getMetaClient();
        StoragePath metadataPath = FSUtils.constructAbsolutePath(metaClient.getHashingMetadataPath(), partition);
        try {
            StoragePathInfo maxMetadataFile;
            Predicate<StoragePathInfo> hashingMetaCommitFilePredicate = pathInfo -> {
                String filename = pathInfo.getPath().getName();
                return filename.endsWith(".commit");
            };
            Predicate<StoragePathInfo> hashingMetadataFilePredicate = pathInfo -> {
                String filename = pathInfo.getPath().getName();
                return filename.endsWith(".hashing_meta");
            };
            List<StoragePathInfo> metaFiles = metaClient.getStorage().listDirectEntries(metadataPath);
            TreeSet commitMetaTss = metaFiles.stream().filter(hashingMetaCommitFilePredicate).map(commitFile -> HoodieConsistentHashingMetadata.getTimestampFromFile(commitFile.getPath().getName())).sorted().collect(Collectors.toCollection(TreeSet::new));
            List<StoragePathInfo> hashingMetaFiles = metaFiles.stream().filter(hashingMetadataFilePredicate).sorted(Comparator.comparing(f -> f.getPath().getName())).collect(Collectors.toList());
            String maxCommitMetaFileTs = commitMetaTss.isEmpty() ? null : (String)commitMetaTss.last();
            StoragePathInfo storagePathInfo = maxMetadataFile = hashingMetaFiles.isEmpty() ? null : (StoragePathInfo)hashingMetaFiles.get(hashingMetaFiles.size() - 1);
            if (maxMetadataFile != null && HoodieConsistentHashingMetadata.getTimestampFromFile(maxMetadataFile.getPath().getName()).equals("00000000000000")) {
                return ConsistentBucketIndexUtils.loadMetadataFromGivenFile(table, maxMetadataFile);
            }
            if (maxCommitMetaFileTs != null && maxMetadataFile != null && maxCommitMetaFileTs.equals(HoodieConsistentHashingMetadata.getTimestampFromFile(maxMetadataFile.getPath().getName()))) {
                return ConsistentBucketIndexUtils.loadMetadataFromGivenFile(table, maxMetadataFile);
            }
            HoodieTimeline completedCommits = metaClient.getActiveTimeline().getCommitAndReplaceTimeline().filterCompletedInstants();
            ArrayList fixed = new ArrayList();
            Option<Object> maxCommittedMetadataFileOpt = Option.empty();
            if (maxCommitMetaFileTs != null) {
                maxCommittedMetadataFileOpt = Option.fromJavaOptional(hashingMetaFiles.stream().filter(hashingMetaFile -> {
                    String timestamp = HoodieConsistentHashingMetadata.getTimestampFromFile(hashingMetaFile.getPath().getName());
                    return maxCommitMetaFileTs.equals(timestamp);
                }).findFirst());
                ValidationUtils.checkState(maxCommittedMetadataFileOpt.isPresent(), "Failed to find max committed metadata file but commit marker file exist with instant: " + maxCommittedMetadataFileOpt);
            }
            hashingMetaFiles.forEach(hashingMetaFile -> {
                boolean isRehashingCommitted;
                StoragePath path = hashingMetaFile.getPath();
                String timestamp = HoodieConsistentHashingMetadata.getTimestampFromFile(path.getName());
                if (maxCommitMetaFileTs != null && timestamp.compareTo(maxCommitMetaFileTs) <= 0) {
                    return;
                }
                boolean bl = isRehashingCommitted = completedCommits.containsInstant(timestamp) || timestamp.equals("00000000000000");
                if (isRehashingCommitted) {
                    if (!commitMetaTss.contains(timestamp)) {
                        try {
                            ConsistentBucketIndexUtils.createCommitMarker(table, path, metadataPath);
                        }
                        catch (IOException e) {
                            throw new HoodieIOException("Exception while creating marker file " + path.getName() + " for partition " + partition, e);
                        }
                    }
                    fixed.add(hashingMetaFile);
                } else if (ConsistentBucketIndexUtils.recommitMetadataFile(table, hashingMetaFile, partition)) {
                    fixed.add(hashingMetaFile);
                }
            });
            if (!fixed.isEmpty()) {
                return ConsistentBucketIndexUtils.loadMetadataFromGivenFile(table, (StoragePathInfo)fixed.get(fixed.size() - 1));
            }
            return maxCommittedMetadataFileOpt.isPresent() ? ConsistentBucketIndexUtils.loadMetadataFromGivenFile(table, (StoragePathInfo)maxCommittedMetadataFileOpt.get()) : Option.empty();
        }
        catch (FileNotFoundException e) {
            return Option.empty();
        }
        catch (IOException e) {
            LOG.error("Error when loading hashing metadata, partition: " + partition, (Throwable)e);
            throw new HoodieIndexException("Error while loading hashing metadata", e);
        }
    }

    public static boolean saveMetadata(HoodieTable table, HoodieConsistentHashingMetadata metadata2) {
        HoodieStorage storage2 = table.getStorage();
        StoragePath dir = FSUtils.constructAbsolutePath(table.getMetaClient().getHashingMetadataPath(), metadata2.getPartitionPath());
        StoragePath fullPath = new StoragePath(dir, metadata2.getFilename());
        try {
            if (storage2.exists(fullPath)) {
                return true;
            }
            storage2.createImmutableFileInPath(fullPath, Option.of(metadata2.toBytes()), true);
            return true;
        }
        catch (IOException e1) {
            try {
                if (storage2.exists(fullPath)) {
                    return true;
                }
            }
            catch (IOException e2) {
                LOG.warn("Failed to check the existence of bucket metadata file: " + fullPath, (Throwable)e2);
            }
            LOG.warn("Failed to update bucket metadata: " + metadata2, (Throwable)e1);
            return false;
        }
    }

    private static void createCommitMarker(HoodieTable table, StoragePath path, StoragePath metadataPath) throws IOException {
        StoragePath fullPath;
        HoodieStorage storage2 = table.getStorage();
        if (storage2.exists(fullPath = new StoragePath(metadataPath, HoodieConsistentHashingMetadata.getTimestampFromFile(path.getName()) + ".commit"))) {
            return;
        }
        try {
            FileIOUtils.createFileInPath(storage2, fullPath, Option.of(StringUtils.getUTF8Bytes("")));
        }
        catch (HoodieIOException e) {
            if (!storage2.exists(fullPath)) {
                throw e;
            }
            LOG.warn("Failed to create marker but " + fullPath + " exists", (Throwable)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static Option<HoodieConsistentHashingMetadata> loadMetadataFromGivenFile(HoodieTable table, StoragePathInfo metaFile) {
        if (metaFile == null) {
            return Option.empty();
        }
        try (InputStream is = table.getStorage().open(metaFile.getPath());){
            byte[] content = FileIOUtils.readAsByteArray(is);
            Option<HoodieConsistentHashingMetadata> option = Option.of(HoodieConsistentHashingMetadata.fromBytes(content));
            return option;
        }
        catch (FileNotFoundException e) {
            return Option.empty();
        }
        catch (IOException e) {
            LOG.error("Error when loading hashing metadata, for path: " + metaFile.getPath().getName(), (Throwable)e);
            throw new HoodieIndexException("Error while loading hashing metadata", e);
        }
    }

    private static boolean recommitMetadataFile(HoodieTable table, StoragePathInfo metaFile, String partition) {
        StoragePath metadataPath = FSUtils.constructAbsolutePath(table.getMetaClient().getHashingMetadataPath(), partition);
        String timestamp = HoodieConsistentHashingMetadata.getTimestampFromFile(metaFile.getPath().getName());
        if (table.getPendingCommitsTimeline().containsInstant(timestamp)) {
            return false;
        }
        Option<HoodieConsistentHashingMetadata> hoodieConsistentHashingMetadataOption = ConsistentBucketIndexUtils.loadMetadataFromGivenFile(table, metaFile);
        if (!hoodieConsistentHashingMetadataOption.isPresent()) {
            return false;
        }
        HoodieConsistentHashingMetadata hoodieConsistentHashingMetadata = hoodieConsistentHashingMetadataOption.get();
        Predicate<String> hoodieFileGroupIdPredicate = hoodieBaseFile -> hoodieConsistentHashingMetadata.getNodes().stream().anyMatch(node -> node.getFileIdPrefix().equals(hoodieBaseFile));
        if (table.getBaseFileOnlyView().getLatestBaseFiles(partition).map(fileIdPrefix -> FSUtils.getFileIdPfxFromFileId(fileIdPrefix.getFileId())).anyMatch(hoodieFileGroupIdPredicate)) {
            try {
                ConsistentBucketIndexUtils.createCommitMarker(table, metaFile.getPath(), metadataPath);
                return true;
            }
            catch (IOException e) {
                throw new HoodieIOException("Exception while creating marker file " + metaFile.getPath().getName() + " for partition " + partition, e);
            }
        }
        return false;
    }

    public static Map<String, Map<String, Integer>> generatePartitionToFileIdPfxIdxMap(Map<String, ConsistentBucketIdentifier> partitionToIdentifier) {
        HashMap<String, Map<String, Integer>> partitionToFileIdPfxIdxMap = new HashMap<String, Map<String, Integer>>(partitionToIdentifier.size() * 2);
        int count = 0;
        for (ConsistentBucketIdentifier identifier : partitionToIdentifier.values()) {
            HashMap<String, Integer> fileIdPfxToIdx = new HashMap<String, Integer>();
            for (ConsistentHashingNode node : identifier.getNodes()) {
                fileIdPfxToIdx.put(node.getFileIdPrefix(), count++);
            }
            partitionToFileIdPfxIdxMap.put(identifier.getMetadata().getPartitionPath(), fileIdPfxToIdx);
        }
        return partitionToFileIdPfxIdxMap;
    }
}

