/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.fs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.apache.hudi.common.fs.ConsistencyGuard;
import org.apache.hudi.common.fs.ConsistencyGuardConfig;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;
import org.apache.hudi.storage.StoragePathInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FailSafeConsistencyGuard
implements ConsistencyGuard {
    private static final Logger LOG = LoggerFactory.getLogger(FailSafeConsistencyGuard.class);
    protected final HoodieStorage storage;
    protected final ConsistencyGuardConfig consistencyGuardConfig;

    public FailSafeConsistencyGuard(HoodieStorage storage2, ConsistencyGuardConfig consistencyGuardConfig) {
        this.storage = storage2;
        this.consistencyGuardConfig = consistencyGuardConfig;
        ValidationUtils.checkArgument(consistencyGuardConfig.isConsistencyCheckEnabled());
    }

    @Override
    public void waitTillFileAppears(StoragePath filePath) throws TimeoutException {
        this.waitForFileVisibility(filePath, ConsistencyGuard.FileVisibility.APPEAR);
    }

    @Override
    public void waitTillFileDisappears(StoragePath filePath) throws TimeoutException {
        this.waitForFileVisibility(filePath, ConsistencyGuard.FileVisibility.DISAPPEAR);
    }

    @Override
    public void waitTillAllFilesAppear(String dirPath, List<String> files) throws TimeoutException {
        this.waitForFilesVisibility(dirPath, files, ConsistencyGuard.FileVisibility.APPEAR);
    }

    @Override
    public void waitTillAllFilesDisappear(String dirPath, List<String> files) throws TimeoutException {
        this.waitForFilesVisibility(dirPath, files, ConsistencyGuard.FileVisibility.DISAPPEAR);
    }

    public void waitForFilesVisibility(String dirPath, List<String> files, ConsistencyGuard.FileVisibility event) throws TimeoutException {
        StoragePath dir = new StoragePath(dirPath);
        List<String> filesWithoutSchemeAndAuthority = this.getFilesWithoutSchemeAndAuthority(files);
        this.retryTillSuccess(dir, filesWithoutSchemeAndAuthority, event);
    }

    protected boolean checkFileVisibility(StoragePath filePath, ConsistencyGuard.FileVisibility visibility) throws IOException {
        try {
            StoragePathInfo pathInfo = this.storage.getPathInfo(filePath);
            switch (visibility) {
                case APPEAR: {
                    return pathInfo != null;
                }
            }
            return pathInfo == null;
        }
        catch (FileNotFoundException nfe) {
            switch (visibility) {
                case APPEAR: {
                    return false;
                }
            }
            return true;
        }
    }

    private void waitForFileVisibility(StoragePath filePath, ConsistencyGuard.FileVisibility visibility) throws TimeoutException {
        long waitMs = this.consistencyGuardConfig.getInitialConsistencyCheckIntervalMs();
        for (int attempt = 0; attempt < this.consistencyGuardConfig.getMaxConsistencyChecks(); ++attempt) {
            try {
                if (this.checkFileVisibility(filePath, visibility)) {
                    return;
                }
            }
            catch (IOException ioe) {
                LOG.warn("Got IOException waiting for file visibility. Retrying", (Throwable)ioe);
            }
            this.sleepSafe(waitMs);
            waitMs *= 2L;
            waitMs = Math.min(waitMs, (long)this.consistencyGuardConfig.getMaxConsistencyCheckIntervalMs());
        }
        throw new TimeoutException("Timed-out waiting for the file to " + visibility.name());
    }

    private void retryTillSuccess(StoragePath dir, List<String> files, ConsistencyGuard.FileVisibility event) throws TimeoutException {
        long waitMs = this.consistencyGuardConfig.getInitialConsistencyCheckIntervalMs();
        LOG.info("Max Attempts=" + this.consistencyGuardConfig.getMaxConsistencyChecks());
        for (int attempt = 0; attempt < this.consistencyGuardConfig.getMaxConsistencyChecks(); ++attempt) {
            boolean success = this.checkFilesVisibility(attempt, dir, files, event);
            if (success) {
                return;
            }
            this.sleepSafe(waitMs);
            waitMs *= 2L;
            waitMs = Math.min(waitMs, (long)this.consistencyGuardConfig.getMaxConsistencyCheckIntervalMs());
        }
        throw new TimeoutException("Timed out waiting for files to adhere to event " + event.name());
    }

    protected boolean checkFilesVisibility(int retryNum, StoragePath dir, List<String> files, ConsistencyGuard.FileVisibility event) {
        try {
            LOG.info("Trying " + retryNum);
            List<StoragePathInfo> entries = this.storage.listDirectEntries(dir);
            List gotFiles = entries.stream().map(e -> e.getPath().getPathWithoutSchemeAndAuthority()).map(StoragePath::toString).collect(Collectors.toList());
            ArrayList<String> candidateFiles = new ArrayList<String>(files);
            boolean altered = candidateFiles.removeAll(gotFiles);
            switch (event) {
                case DISAPPEAR: {
                    LOG.info("Following files are visible" + candidateFiles);
                    return !altered;
                }
            }
            return candidateFiles.isEmpty();
        }
        catch (IOException ioe) {
            LOG.warn("Got IOException waiting for file event. Have tried " + retryNum + " time(s)", (Throwable)ioe);
            return false;
        }
    }

    protected List<String> getFilesWithoutSchemeAndAuthority(List<String> files) {
        return files.stream().map(f -> new StoragePath((String)f).getPathWithoutSchemeAndAuthority()).map(StoragePath::toString).collect(Collectors.toList());
    }

    private void sleepSafe(long waitMs) {
        try {
            Thread.sleep(waitMs);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }
}

