/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.org.apache.hadoop.hbase.quotas;

import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hudi.org.apache.hadoop.hbase.ClusterMetrics;
import org.apache.hudi.org.apache.hadoop.hbase.MetaTableAccessor;
import org.apache.hudi.org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hudi.org.apache.hadoop.hbase.Stoppable;
import org.apache.hudi.org.apache.hadoop.hbase.TableName;
import org.apache.hudi.org.apache.hadoop.hbase.client.Get;
import org.apache.hudi.org.apache.hadoop.hbase.quotas.NoopQuotaLimiter;
import org.apache.hudi.org.apache.hadoop.hbase.quotas.QuotaLimiter;
import org.apache.hudi.org.apache.hadoop.hbase.quotas.QuotaState;
import org.apache.hudi.org.apache.hadoop.hbase.quotas.QuotaUtil;
import org.apache.hudi.org.apache.hadoop.hbase.quotas.UserQuotaState;
import org.apache.hudi.org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hudi.org.apache.hadoop.hbase.regionserver.RegionServerServices;
import org.apache.hudi.org.apache.hadoop.hbase.util.ConcurrentMapUtils;
import org.apache.hudi.org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class QuotaCache
implements Stoppable {
    private static final Logger LOG = LoggerFactory.getLogger(QuotaCache.class);
    public static final String REFRESH_CONF_KEY = "hbase.quota.refresh.period";
    private static final int REFRESH_DEFAULT_PERIOD = 300000;
    private static final int EVICT_PERIOD_FACTOR = 5;
    static boolean TEST_FORCE_REFRESH = false;
    private final ConcurrentHashMap<String, QuotaState> namespaceQuotaCache = new ConcurrentHashMap();
    private final ConcurrentHashMap<TableName, QuotaState> tableQuotaCache = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, UserQuotaState> userQuotaCache = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, QuotaState> regionServerQuotaCache = new ConcurrentHashMap();
    private volatile boolean exceedThrottleQuotaEnabled = false;
    private volatile double machineQuotaFactor = 1.0;
    private final ConcurrentHashMap<TableName, Double> tableMachineQuotaFactors = new ConcurrentHashMap();
    private final RegionServerServices rsServices;
    private QuotaRefresherChore refreshChore;
    private boolean stopped = true;

    public QuotaCache(RegionServerServices rsServices) {
        this.rsServices = rsServices;
    }

    public void start() throws IOException {
        this.stopped = false;
        Configuration conf = this.rsServices.getConfiguration();
        int period = conf.getInt(REFRESH_CONF_KEY, 300000);
        this.refreshChore = new QuotaRefresherChore(period, this);
        this.rsServices.getChoreService().scheduleChore(this.refreshChore);
    }

    @Override
    public void stop(String why) {
        if (this.refreshChore != null) {
            LOG.debug("Stopping QuotaRefresherChore chore.");
            this.refreshChore.shutdown(true);
        }
        this.stopped = true;
    }

    @Override
    public boolean isStopped() {
        return this.stopped;
    }

    public QuotaLimiter getUserLimiter(UserGroupInformation ugi, TableName table) {
        if (table.isSystemTable()) {
            return NoopQuotaLimiter.get();
        }
        return this.getUserQuotaState(ugi).getTableLimiter(table);
    }

    public UserQuotaState getUserQuotaState(UserGroupInformation ugi) {
        return ConcurrentMapUtils.computeIfAbsent(this.userQuotaCache, ugi.getShortUserName(), UserQuotaState::new, this::triggerCacheRefresh);
    }

    public QuotaLimiter getTableLimiter(TableName table) {
        return this.getQuotaState(this.tableQuotaCache, table).getGlobalLimiter();
    }

    public QuotaLimiter getNamespaceLimiter(String namespace) {
        return this.getQuotaState(this.namespaceQuotaCache, namespace).getGlobalLimiter();
    }

    public QuotaLimiter getRegionServerQuotaLimiter(String regionServer) {
        return this.getQuotaState(this.regionServerQuotaCache, regionServer).getGlobalLimiter();
    }

    protected boolean isExceedThrottleQuotaEnabled() {
        return this.exceedThrottleQuotaEnabled;
    }

    private <K> QuotaState getQuotaState(ConcurrentHashMap<K, QuotaState> quotasMap, K key) {
        return ConcurrentMapUtils.computeIfAbsent(quotasMap, key, QuotaState::new, this::triggerCacheRefresh);
    }

    void triggerCacheRefresh() {
        this.refreshChore.triggerNow();
    }

    long getLastUpdate() {
        return this.refreshChore.lastUpdate;
    }

    Map<String, QuotaState> getNamespaceQuotaCache() {
        return this.namespaceQuotaCache;
    }

    Map<String, QuotaState> getRegionServerQuotaCache() {
        return this.regionServerQuotaCache;
    }

    Map<TableName, QuotaState> getTableQuotaCache() {
        return this.tableQuotaCache;
    }

    Map<String, UserQuotaState> getUserQuotaCache() {
        return this.userQuotaCache;
    }

    static interface Fetcher<Key, Value> {
        public Get makeGet(Map.Entry<Key, Value> var1);

        public Map<Key, Value> fetchEntries(List<Get> var1) throws IOException;
    }

    private class QuotaRefresherChore
    extends ScheduledChore {
        private long lastUpdate;

        public QuotaRefresherChore(int period, Stoppable stoppable) {
            super("QuotaRefresherChore", stoppable, period);
            this.lastUpdate = 0L;
        }

        @Override
        @SuppressWarnings(value={"GC_UNRELATED_TYPES"}, justification="I do not understand why the complaints, it looks good to me -- FIX")
        protected void chore() {
            for (TableName table : ((HRegionServer)QuotaCache.this.rsServices).getOnlineTables()) {
                if (table.isSystemTable()) continue;
                if (!QuotaCache.this.tableQuotaCache.containsKey(table)) {
                    QuotaCache.this.tableQuotaCache.putIfAbsent(table, new QuotaState());
                }
                String ns = table.getNamespaceAsString();
                if (QuotaCache.this.namespaceQuotaCache.containsKey(ns)) continue;
                QuotaCache.this.namespaceQuotaCache.putIfAbsent(ns, new QuotaState());
            }
            QuotaCache.this.regionServerQuotaCache.putIfAbsent("all", new QuotaState());
            this.updateQuotaFactors();
            this.fetchNamespaceQuotaState();
            this.fetchTableQuotaState();
            this.fetchUserQuotaState();
            this.fetchRegionServerQuotaState();
            this.fetchExceedThrottleQuota();
            this.lastUpdate = EnvironmentEdgeManager.currentTime();
        }

        private void fetchNamespaceQuotaState() {
            this.fetch("namespace", QuotaCache.this.namespaceQuotaCache, new Fetcher<String, QuotaState>(){

                @Override
                public Get makeGet(Map.Entry<String, QuotaState> entry) {
                    return QuotaUtil.makeGetForNamespaceQuotas(entry.getKey());
                }

                @Override
                public Map<String, QuotaState> fetchEntries(List<Get> gets) throws IOException {
                    return QuotaUtil.fetchNamespaceQuotas(QuotaCache.this.rsServices.getConnection(), gets, QuotaCache.this.machineQuotaFactor);
                }
            });
        }

        private void fetchTableQuotaState() {
            this.fetch("table", QuotaCache.this.tableQuotaCache, new Fetcher<TableName, QuotaState>(){

                @Override
                public Get makeGet(Map.Entry<TableName, QuotaState> entry) {
                    return QuotaUtil.makeGetForTableQuotas(entry.getKey());
                }

                @Override
                public Map<TableName, QuotaState> fetchEntries(List<Get> gets) throws IOException {
                    return QuotaUtil.fetchTableQuotas(QuotaCache.this.rsServices.getConnection(), gets, QuotaCache.this.tableMachineQuotaFactors);
                }
            });
        }

        private void fetchUserQuotaState() {
            final Set namespaces = QuotaCache.this.namespaceQuotaCache.keySet();
            final Set tables = QuotaCache.this.tableQuotaCache.keySet();
            this.fetch("user", QuotaCache.this.userQuotaCache, new Fetcher<String, UserQuotaState>(){

                @Override
                public Get makeGet(Map.Entry<String, UserQuotaState> entry) {
                    return QuotaUtil.makeGetForUserQuotas(entry.getKey(), tables, namespaces);
                }

                @Override
                public Map<String, UserQuotaState> fetchEntries(List<Get> gets) throws IOException {
                    return QuotaUtil.fetchUserQuotas(QuotaCache.this.rsServices.getConnection(), gets, QuotaCache.this.tableMachineQuotaFactors, QuotaCache.this.machineQuotaFactor);
                }
            });
        }

        private void fetchRegionServerQuotaState() {
            this.fetch("regionServer", QuotaCache.this.regionServerQuotaCache, new Fetcher<String, QuotaState>(){

                @Override
                public Get makeGet(Map.Entry<String, QuotaState> entry) {
                    return QuotaUtil.makeGetForRegionServerQuotas(entry.getKey());
                }

                @Override
                public Map<String, QuotaState> fetchEntries(List<Get> gets) throws IOException {
                    return QuotaUtil.fetchRegionServerQuotas(QuotaCache.this.rsServices.getConnection(), gets);
                }
            });
        }

        private void fetchExceedThrottleQuota() {
            try {
                QuotaCache.this.exceedThrottleQuotaEnabled = QuotaUtil.isExceedThrottleQuotaEnabled(QuotaCache.this.rsServices.getConnection());
            }
            catch (IOException e) {
                LOG.warn("Unable to read if exceed throttle quota enabled from quota table", (Throwable)e);
            }
        }

        private <K, V extends QuotaState> void fetch(String type2, ConcurrentHashMap<K, V> quotasMap, Fetcher<K, V> fetcher) {
            long now = EnvironmentEdgeManager.currentTime();
            long refreshPeriod = this.getPeriod();
            long evictPeriod = refreshPeriod * 5L;
            ArrayList<Get> gets = new ArrayList<Get>();
            ArrayList<K> toRemove = new ArrayList<K>();
            for (Map.Entry<K, V> entry : quotasMap.entrySet()) {
                long lastUpdate = ((QuotaState)entry.getValue()).getLastUpdate();
                long lastQuery = ((QuotaState)entry.getValue()).getLastQuery();
                if (lastQuery > 0L && now - lastQuery >= evictPeriod) {
                    toRemove.add(entry.getKey());
                    continue;
                }
                if (!TEST_FORCE_REFRESH && now - lastUpdate < refreshPeriod) continue;
                gets.add(fetcher.makeGet(entry));
            }
            for (Map.Entry<K, V> key : toRemove) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace("evict " + type2 + " key=" + key);
                }
                quotasMap.remove(key);
            }
            if (!gets.isEmpty()) {
                try {
                    for (Map.Entry<K, V> entry : fetcher.fetchEntries(gets).entrySet()) {
                        QuotaState quotaInfo = (QuotaState)quotasMap.putIfAbsent(entry.getKey(), entry.getValue());
                        if (quotaInfo != null) {
                            quotaInfo.update((QuotaState)entry.getValue());
                        }
                        if (!LOG.isTraceEnabled()) continue;
                        LOG.trace("refresh " + type2 + " key=" + entry.getKey() + " quotas=" + quotaInfo);
                    }
                }
                catch (IOException e) {
                    LOG.warn("Unable to read " + type2 + " from quota table", (Throwable)e);
                }
            }
        }

        private void updateQuotaFactors() {
            try {
                int rsSize = QuotaCache.this.rsServices.getConnection().getAdmin().getClusterMetrics(EnumSet.of(ClusterMetrics.Option.SERVERS_NAME)).getServersName().size();
                if (rsSize != 0) {
                    QuotaCache.this.machineQuotaFactor = 1.0 / (double)rsSize;
                }
            }
            catch (IOException e) {
                LOG.warn("Get live region servers failed", (Throwable)e);
            }
            for (TableName tableName : QuotaCache.this.tableQuotaCache.keySet()) {
                double factor = 1.0;
                try {
                    long regionSize = MetaTableAccessor.getTableRegions(QuotaCache.this.rsServices.getConnection(), tableName, true).stream().filter(regionInfo -> !regionInfo.isOffline()).count();
                    if (regionSize == 0L) {
                        factor = 0.0;
                    } else {
                        int localRegionSize = QuotaCache.this.rsServices.getRegions(tableName).size();
                        factor = 1.0 * (double)localRegionSize / (double)regionSize;
                    }
                }
                catch (IOException e) {
                    LOG.warn("Get table regions failed: {}", (Object)tableName, (Object)e);
                }
                QuotaCache.this.tableMachineQuotaFactors.put(tableName, factor);
            }
        }
    }
}

