/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.utilities.sources.helpers;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Random;
import org.apache.avro.Conversions;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericFixed;
import org.apache.hudi.AvroConversionUtils;
import org.apache.hudi.avro.MercifulJsonConverterTestBase;
import org.apache.hudi.common.testutils.SchemaTestUtil;
import org.apache.hudi.utilities.exception.HoodieJsonToRowConversionException;
import org.apache.hudi.utilities.sources.helpers.MercifulJsonToRowConverter;
import org.apache.spark.sql.Dataset;
import org.apache.spark.sql.Row;
import org.apache.spark.sql.RowFactory;
import org.apache.spark.sql.SparkSession;
import org.apache.spark.sql.types.StructType;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;

class TestMercifulJsonToRowConverter
extends MercifulJsonConverterTestBase {
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private static final MercifulJsonToRowConverter CONVERTER = new MercifulJsonToRowConverter(true, "__");
    private static final String SIMPLE_AVRO_WITH_DEFAULT = "/schema/simple-test-with-default-value.avsc";
    protected static SparkSession spark;
    private static final String DECIMAL_AVRO_FILE_PATH = "/decimal-logical-type.avsc";
    private static final String DURATION_AVRO_FILE_PATH = "/duration-logical-type.avsc";
    private static final String DATE_AVRO_FILE_PATH = "/date-type.avsc";
    private static final String DATE_AVRO_INVALID_FILE_PATH = "/date-type-invalid.avsc";
    private static final String LOCAL_TIME_AVRO_FILE_PATH = "/local-timestamp-logical-type.avsc";
    private static final String TIMESTAMP_AVRO_FILE_PATH = "/timestamp-logical-type2.avsc";
    private static final String TIME_AVRO_FILE_PATH = "/time-logical-type.avsc";
    private static final String UUID_AVRO_FILE_PATH = "/uuid-logical-type.avsc";

    TestMercifulJsonToRowConverter() {
    }

    @BeforeAll
    public static void start() {
        spark = SparkSession.builder().master("local[*]").appName(TestMercifulJsonToRowConverter.class.getName()).config("spark.serializer", "org.apache.spark.serializer.KryoSerializer").getOrCreate();
    }

    @AfterAll
    public static void clear() {
        spark.close();
    }

    @Test
    void basicConversion() throws IOException {
        Schema simpleSchema = SchemaTestUtil.getSchema((String)SIMPLE_AVRO_WITH_DEFAULT);
        String name = "John Smith";
        int number = 1337;
        String color = "Blue. No yellow!";
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("name", name);
        data.put("favorite_number", number);
        data.put("favorite_color", color);
        String json = MAPPER.writeValueAsString(data);
        ArrayList<Object> values = new ArrayList<Object>(Collections.nCopies(simpleSchema.getFields().size(), null));
        values.set(0, name);
        values.set(1, number);
        values.set(3, color);
        Row recRow = RowFactory.create((Object[])values.toArray());
        Row realRow = CONVERTER.convertToRow(json, simpleSchema);
        this.validateSchemaCompatibility(Collections.singletonList(realRow), simpleSchema);
        Assertions.assertEquals((Object)recRow, (Object)realRow);
    }

    @ParameterizedTest
    @MethodSource(value={"dataNestedJsonAsString"})
    void nestedJsonAsString(String nameInput) throws IOException {
        Schema simpleSchema = SchemaTestUtil.getSimpleSchema();
        int number = 1337;
        String color = "Blue. No yellow!";
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("name", nameInput);
        data.put("favorite_number", number);
        data.put("favorite_color", color);
        String json = MAPPER.writeValueAsString(data);
        ArrayList<Object> values = new ArrayList<Object>(Collections.nCopies(simpleSchema.getFields().size(), null));
        values.set(0, nameInput);
        values.set(1, number);
        values.set(2, color);
        Row recRow = RowFactory.create((Object[])values.toArray());
        Assertions.assertEquals((Object)recRow, (Object)CONVERTER.convertToRow(json, simpleSchema));
    }

    @Test
    void decimalLogicalTypeByteTypeTest() throws IOException {
        String num = "123.45";
        BigDecimal bigDecimal = new BigDecimal(num);
        HashMap<String, String> data = new HashMap<String, String>();
        data.put("decimalField", num);
        String json = MAPPER.writeValueAsString(data);
        Schema schema = SchemaTestUtil.getSchema((String)DECIMAL_AVRO_FILE_PATH);
        Row expectRow = RowFactory.create((Object[])new Object[]{bigDecimal});
        Row realRow = CONVERTER.convertToRow(json, schema);
        this.validateSchemaCompatibility(Collections.singletonList(realRow), schema);
        Assertions.assertEquals((Object)expectRow, (Object)realRow);
    }

    @ParameterizedTest
    @MethodSource(value={"decimalBadCases"})
    void decimalLogicalTypeInvalidCaseTest(String avroFile, String strInput, Double numInput, boolean testFixedByteArray) throws IOException {
        Schema schema = SchemaTestUtil.getSchema((String)avroFile);
        HashMap<String, Object> data = new HashMap<String, Object>();
        if (strInput != null) {
            data.put("decimalField", strInput);
        } else if (numInput != null) {
            data.put("decimalField", numInput);
        } else if (testFixedByteArray) {
            int[] intArray = new int[]{0, 0, 48, 57};
            data.put("decimalField", intArray);
        }
        String json = MAPPER.writeValueAsString(data);
        Assertions.assertThrows(HoodieJsonToRowConversionException.class, () -> CONVERTER.convertToRow(json, schema));
    }

    @ParameterizedTest
    @MethodSource(value={"decimalGoodCases"})
    void decimalLogicalTypeTest(String avroFilePath, String groundTruth, String strInput, Number numInput, boolean testFixedByteArray) throws IOException {
        BigDecimal bigDecimal = new BigDecimal(groundTruth);
        HashMap<String, Object> data = new HashMap<String, Object>();
        Schema schema = SchemaTestUtil.getSchema((String)avroFilePath);
        if (strInput != null) {
            data.put("decimalField", strInput);
        } else if (numInput != null) {
            data.put("decimalField", numInput);
        } else if (testFixedByteArray) {
            Schema fieldSchema = schema.getField("decimalField").schema();
            GenericFixed fixedValue = new Conversions.DecimalConversion().toFixed(bigDecimal, fieldSchema, fieldSchema.getLogicalType());
            byte[] byteArray = fixedValue.bytes();
            int[] intArray = new int[byteArray.length];
            for (int i = 0; i < byteArray.length; ++i) {
                intArray[i] = byteArray[i] & 0xFF;
            }
            data.put("decimalField", intArray);
        }
        String json = MAPPER.writeValueAsString(data);
        Row expectRow = RowFactory.create((Object[])new Object[]{bigDecimal});
        Row realRow = CONVERTER.convertToRow(json, schema);
        this.validateSchemaCompatibility(Collections.singletonList(realRow), schema);
        Assertions.assertEquals((Object)expectRow, (Object)realRow);
    }

    @ParameterizedTest
    @MethodSource(value={"zeroScaleDecimalCases"})
    void zeroScaleDecimalConversion(String inputValue, String expected, boolean shouldConvert) {
        Schema schema = new Schema.Parser().parse("{\"namespace\": \"example.avro\",\"type\": \"record\",\"name\": \"decimalLogicalType\",\"fields\": [{\"name\": \"decimalField\", \"type\": {\"type\": \"bytes\", \"logicalType\": \"decimal\", \"precision\": 38, \"scale\": 0}}]}");
        String json = String.format("{\"decimalField\":%s}", inputValue);
        if (shouldConvert) {
            BigDecimal bigDecimal = new BigDecimal(expected);
            Row expectedRow = RowFactory.create((Object[])new Object[]{bigDecimal});
            Row actualRow = CONVERTER.convertToRow(json, schema);
            this.validateSchemaCompatibility(Collections.singletonList(actualRow), schema);
            Assertions.assertEquals((Object)expectedRow, (Object)actualRow);
        } else {
            Assertions.assertThrows(HoodieJsonToRowConversionException.class, () -> CONVERTER.convertToRow(json, schema));
        }
    }

    @ParameterizedTest
    @MethodSource(value={"durationGoodCases"})
    void durationLogicalTypeTest(int months, int days, int milliseconds) throws IOException {
        ArrayList<Integer> num = new ArrayList<Integer>();
        num.add(months);
        num.add(days);
        num.add(milliseconds);
        HashMap<String, ArrayList<Integer>> data = new HashMap<String, ArrayList<Integer>>();
        data.put("duration", num);
        String json = MAPPER.writeValueAsString(data);
        ByteBuffer buffer = ByteBuffer.allocate(12).order(ByteOrder.LITTLE_ENDIAN);
        buffer.putInt(months);
        buffer.putInt(days);
        buffer.putInt(milliseconds);
        buffer.flip();
        Schema schema = SchemaTestUtil.getSchema((String)DURATION_AVRO_FILE_PATH);
        Assertions.assertThrows(HoodieJsonToRowConversionException.class, () -> CONVERTER.convertToRow(json, schema));
    }

    @ParameterizedTest
    @MethodSource(value={"durationBadCases"})
    void durationLogicalTypeBadTest(String schemaFile, Object input) throws IOException {
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("duration", input);
        String json = MAPPER.writeValueAsString(data);
        Schema schema = SchemaTestUtil.getSchemaFromResourceFilePath((String)schemaFile);
        Assertions.assertThrows(HoodieJsonToRowConversionException.class, () -> CONVERTER.convertToRow(json, schema));
    }

    @ParameterizedTest
    @MethodSource(value={"dateProviderForRow"})
    void dateLogicalTypeTest(String groundTruthRow, Object dateInput) throws IOException {
        Schema schema = SchemaTestUtil.getSchema((String)DATE_AVRO_FILE_PATH);
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("dateField", dateInput);
        String json = MAPPER.writeValueAsString(data);
        if (groundTruthRow == null) {
            return;
        }
        Row rec = RowFactory.create((Object[])new Object[]{Date.valueOf(groundTruthRow)});
        Row realRow = CONVERTER.convertToRow(json, schema);
        this.validateSchemaCompatibility(Collections.singletonList(realRow), schema);
        Assertions.assertEquals((Object)rec.getDate(0).toString(), (Object)realRow.getDate(0).toString());
    }

    @Test
    void dateLogicalTypeTest() throws IOException {
        Schema schema = SchemaTestUtil.getSchema((String)DATE_AVRO_INVALID_FILE_PATH);
        HashMap<String, Integer> data = new HashMap<String, Integer>();
        data.put("dateField", 1);
        String json = MAPPER.writeValueAsString(data);
        Assertions.assertThrows(HoodieJsonToRowConversionException.class, () -> CONVERTER.convertToRow(json, schema));
    }

    @ParameterizedTest
    @MethodSource(value={"localTimestampGoodCaseProvider"})
    void localTimestampLogicalTypeGoodCaseTest(Long expectedMicroSecOfDay, Object timeMilli, Object timeMicro) throws IOException {
        long microSecOfDay = expectedMicroSecOfDay;
        long milliSecOfDay = expectedMicroSecOfDay / 1000L;
        Schema schema = SchemaTestUtil.getSchema((String)LOCAL_TIME_AVRO_FILE_PATH);
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("localTimestampMillisField", timeMilli);
        data.put("localTimestampMicrosField", timeMicro);
        String json = MAPPER.writeValueAsString(data);
        Row rec = RowFactory.create((Object[])new Object[]{milliSecOfDay, microSecOfDay});
        Row actualRow = CONVERTER.convertToRow(json, schema);
        this.validateSchemaCompatibility(Collections.singletonList(actualRow), schema);
        Assertions.assertEquals((Object)rec, (Object)actualRow);
    }

    @ParameterizedTest
    @MethodSource(value={"localTimestampBadCaseProvider"})
    void localTimestampLogicalTypeBadTest(String schemaFile, Object input) throws IOException {
        Schema schema = SchemaTestUtil.getSchema((String)schemaFile);
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("timestamp", input);
        String json = MAPPER.writeValueAsString(data);
        Assertions.assertThrows(HoodieJsonToRowConversionException.class, () -> CONVERTER.convertToRow(json, schema));
    }

    @ParameterizedTest
    @MethodSource(value={"timestampGoodCaseProvider"})
    void timestampLogicalTypeGoodCaseTest(Long expectedMicroSecOfDay, Object timeMilli, Object timeMicro) throws IOException {
        long microSecOfDay = expectedMicroSecOfDay;
        long milliSecOfDay = expectedMicroSecOfDay / 1000L;
        Schema schema = SchemaTestUtil.getSchema((String)TIMESTAMP_AVRO_FILE_PATH);
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("timestampMillisField", timeMilli);
        data.put("timestampMicrosField", timeMicro);
        String json = MAPPER.writeValueAsString(data);
        Row rec = RowFactory.create((Object[])new Object[]{new Timestamp(milliSecOfDay), new Timestamp(microSecOfDay / 1000L)});
        Row actualRow = CONVERTER.convertToRow(json, schema);
        this.validateSchemaCompatibility(Collections.singletonList(actualRow), schema);
        Assertions.assertEquals((Object)rec, (Object)actualRow);
    }

    @ParameterizedTest
    @MethodSource(value={"timestampBadCaseProvider"})
    void timestampLogicalTypeBadTest(Object input) throws IOException {
        Schema schema = SchemaTestUtil.getSchema((String)TIMESTAMP_AVRO_FILE_PATH);
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("timestampMillisField", input);
        data.put("timestampMicrosField", input);
        String json = MAPPER.writeValueAsString(data);
        Assertions.assertThrows(HoodieJsonToRowConversionException.class, () -> CONVERTER.convertToRow(json, schema));
    }

    @ParameterizedTest
    @MethodSource(value={"timeGoodCaseProvider"})
    void timeLogicalTypeTest(Long expectedMicroSecOfDay, Object timeMilli, Object timeMicro) throws IOException {
        long microSecOfDay = expectedMicroSecOfDay;
        int milliSecOfDay = (int)(expectedMicroSecOfDay / 1000L);
        Schema schema = SchemaTestUtil.getSchema((String)TIME_AVRO_FILE_PATH);
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("timeMicroField", timeMicro);
        data.put("timeMillisField", timeMilli);
        String json = MAPPER.writeValueAsString(data);
        Row rec = RowFactory.create((Object[])new Object[]{microSecOfDay, milliSecOfDay});
        Row realRow = CONVERTER.convertToRow(json, schema);
        this.validateSchemaCompatibility(Collections.singletonList(realRow), schema);
        Assertions.assertEquals((Object)rec.get(0).toString(), (Object)realRow.get(0).toString());
        Assertions.assertEquals((Object)rec.get(1).toString(), (Object)realRow.get(1).toString());
    }

    @ParameterizedTest
    @MethodSource(value={"timeBadCaseProvider"})
    void timeLogicalTypeBadCaseTest(Object invalidInput) throws IOException {
        String validInput = "00:00:00";
        Schema schema = SchemaTestUtil.getSchemaFromResourceFilePath((String)TIME_AVRO_FILE_PATH);
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("timeMicroField", validInput);
        data.put("timeMillisField", invalidInput);
        Assertions.assertThrows(HoodieJsonToRowConversionException.class, () -> CONVERTER.convertToRow(MAPPER.writeValueAsString((Object)data), schema));
        data.clear();
        data.put("timeMicroField", invalidInput);
        data.put("timeMillisField", validInput);
        Assertions.assertThrows(HoodieJsonToRowConversionException.class, () -> CONVERTER.convertToRow(MAPPER.writeValueAsString((Object)data), schema));
    }

    @ParameterizedTest
    @MethodSource(value={"uuidDimension"})
    void uuidLogicalTypeTest(String uuid) throws IOException {
        Schema schema = SchemaTestUtil.getSchema((String)UUID_AVRO_FILE_PATH);
        HashMap<String, String> data = new HashMap<String, String>();
        data.put("uuidField", uuid);
        String json = MAPPER.writeValueAsString(data);
        Row rec = RowFactory.create((Object[])new Object[]{uuid});
        Row real = CONVERTER.convertToRow(json, schema);
        this.validateSchemaCompatibility(Collections.singletonList(real), schema);
        Assertions.assertEquals((Object)rec, (Object)real);
    }

    @ParameterizedTest
    @MethodSource(value={"nestedRecord"})
    void nestedRecordTest(String contactInput, boolean isString) {
        String nestedSchemaStr = "{\"type\":\"record\",\"name\":\"User\",\"fields\":[{\"name\":\"name\",\"type\":\"string\"},{\"name\":\"contact\",\"type\":{\"type\":\"record\",\"name\":\"Contact\",\"fields\":[{\"name\":\"email\",\"type\":\"string\"}]}}]}";
        String json = isString ? String.format("{\"name\":\"Jane Smith\",\"contact\":{\"email\":\"%s\"}}", contactInput) : String.format("{\"name\":\"Jane Smith\",\"contact\":{\"email\":%s}}", contactInput);
        Schema nestedSchema = new Schema.Parser().parse(nestedSchemaStr);
        Row expected = RowFactory.create((Object[])new Object[]{"Jane Smith", RowFactory.create((Object[])new Object[]{contactInput})});
        Row real = CONVERTER.convertToRow(json, nestedSchema);
        this.validateSchemaCompatibility(Collections.singletonList(real), nestedSchema);
        Assertions.assertEquals((Object)expected, (Object)real);
    }

    @Test
    public void conversionWithFieldNameSanitization() throws IOException {
        String sanitizedSchemaString = "{\"namespace\": \"example.avro\", \"type\": \"record\", \"name\": \"User\", \"fields\": [{\"name\": \"__name\", \"type\": \"string\"}, {\"name\": \"favorite__number\", \"type\": \"int\"}, {\"name\": \"favorite__color__\", \"type\": \"string\"}]}";
        Schema sanitizedSchema = Schema.parse((String)sanitizedSchemaString);
        String name = "John Smith";
        int number = 1337;
        String color = "Blue. No yellow!";
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("$name", name);
        data.put("favorite-number", number);
        data.put("favorite.color!", color);
        String json = MAPPER.writeValueAsString(data);
        ArrayList<Object> values = new ArrayList<Object>(Collections.nCopies(sanitizedSchema.getFields().size(), null));
        values.set(0, name);
        values.set(1, number);
        values.set(2, color);
        Row expected = RowFactory.create((Object[])values.toArray());
        Row actual = CONVERTER.convertToRow(json, sanitizedSchema);
        this.validateSchemaCompatibility(Collections.singletonList(actual), sanitizedSchema);
        Assertions.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    void conversionWithFieldNameAliases() throws IOException {
        String schemaStringWithAliases = "{\"namespace\": \"example.avro\", \"type\": \"record\", \"name\": \"User\", \"fields\": [{\"name\": \"name\", \"type\": \"string\", \"aliases\": [\"$name\"]}, {\"name\": \"favorite_number\",  \"type\": \"int\", \"aliases\": [\"unused\", \"favorite-number\"]}, {\"name\": \"favorite_color\", \"type\": \"string\", \"aliases\": [\"favorite.color!\"]}, {\"name\": \"unmatched\", \"type\": \"string\", \"default\": \"default_value\"}]}";
        Schema sanitizedSchema = new Schema.Parser().parse(schemaStringWithAliases);
        String name = "John Smith";
        int number = 1337;
        String color = "Blue. No yellow!";
        String unmatched = "unmatched";
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("$name", name);
        data.put("favorite-number", number);
        data.put("favorite.color!", color);
        data.put("unmatched", unmatched);
        String json = MAPPER.writeValueAsString(data);
        ArrayList<Object> values = new ArrayList<Object>(Collections.nCopies(sanitizedSchema.getFields().size(), null));
        values.set(0, name);
        values.set(1, number);
        values.set(2, color);
        values.set(3, unmatched);
        Row recRow = RowFactory.create((Object[])values.toArray());
        Row realRow = CONVERTER.convertToRow(json, sanitizedSchema);
        this.validateSchemaCompatibility(Collections.singletonList(realRow), sanitizedSchema);
        Assertions.assertEquals((Object)recRow, (Object)realRow);
    }

    @ParameterizedTest
    @MethodSource(value={"encodedDecimalScalePrecisionProvider"})
    void testEncodedDecimal(int scale, int precision) throws JsonProcessingException {
        Random rand = new Random();
        BigDecimal decfield = BigDecimal.valueOf(rand.nextDouble()).setScale(scale, RoundingMode.HALF_UP).round(new MathContext(precision, RoundingMode.HALF_UP));
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("_row_key", "mykey");
        long timestamp = 214523432L;
        data.put("timestamp", timestamp);
        data.put("rider", "myrider");
        data.put("decfield", Base64.getEncoder().encodeToString(decfield.unscaledValue().toByteArray()));
        data.put("driver", "mydriver");
        data.put("lowprecision", 12.34);
        data.put("highprecision", 12.987654312312);
        data.put("fare", rand.nextDouble() * 100.0);
        data.put("_hoodie_is_deleted", false);
        String json = MAPPER.writeValueAsString(data);
        Schema tripSchema = new Schema.Parser().parse("{\"type\":\"record\",\"name\":\"tripUberRec\",\"fields\":[{\"name\":\"timestamp\",\"type\":\"long\"},{\"name\":\"_row_key\",\"type\":\"string\"},{\"name\":\"rider\",\"type\":\"string\"},{\"name\":\"decfield\",\"type\":{\"type\":\"bytes\",\"name\":\"abc\",\"logicalType\":\"decimal\",\"precision\":10,\"scale\":6}},{\"name\":\"lowprecision\",\"type\":{\"type\":\"bytes\",\"name\":\"def\",\"logicalType\":\"decimal\",\"precision\":4,\"scale\":2}},{\"name\":\"highprecision\",\"type\":{\"type\":\"bytes\",\"name\":\"ghi\",\"logicalType\":\"decimal\",\"precision\":32,\"scale\":12}},{\"name\":\"driver\",\"type\":\"string\"},{\"name\":\"fare\",\"type\":\"double\"},{\"name\": \"_hoodie_is_deleted\", \"type\": \"boolean\", \"default\": false}]}".replace("6", Integer.toString(scale)).replace("10", Integer.toString(precision)));
        Row rec = CONVERTER.convertToRow(json, tripSchema);
        this.validateSchemaCompatibility(Collections.singletonList(rec), tripSchema);
        BigDecimal actualField = rec.getDecimal(tripSchema.getField("decfield").pos());
        Assertions.assertEquals((Object)decfield, (Object)actualField);
    }

    @ParameterizedTest
    @MethodSource(value={"encodedDecimalFixedScalePrecisionProvider"})
    void testEncodedDecimalAvroSparkPostProcessorCase(int size, int scale, int precision) throws JsonProcessingException {
        Random rand = new Random();
        String postProcessSchemaString = String.format("{\"type\":\"record\",\"name\":\"tripUberRec\",\"fields\":[{\"name\":\"timestamp\",\"type\":\"long\",\"doc\":\"\"},{\"name\":\"_row_key\",\"type\":\"string\",\"doc\":\"\"},{\"name\":\"rider\",\"type\":\"string\",\"doc\":\"\"},{\"name\":\"decfield\",\"type\":{\"type\":\"fixed\",\"name\":\"fixed\",\"namespace\":\"tripUberRec.decfield\",\"size\":%d,\"logicalType\":\"decimal\",\"precision\":%d,\"scale\":%d},\"doc\":\"\"},{\"name\":\"driver\",\"type\":\"string\",\"doc\":\"\"},{\"name\":\"fare\",\"type\":\"double\",\"doc\":\"\"},{\"name\":\"_hoodie_is_deleted\",\"type\":\"boolean\",\"doc\":\"\"}]}", size, precision, scale);
        Schema postProcessSchema = new Schema.Parser().parse(postProcessSchemaString);
        BigDecimal decfield = BigDecimal.valueOf(rand.nextDouble()).setScale(scale, RoundingMode.HALF_UP).round(new MathContext(precision, RoundingMode.HALF_UP));
        HashMap<String, Object> data = new HashMap<String, Object>();
        data.put("_row_key", "mykey");
        long timestamp = 214523432L;
        data.put("timestamp", timestamp);
        data.put("rider", "myrider");
        data.put("decfield", Base64.getEncoder().encodeToString(decfield.unscaledValue().toByteArray()));
        data.put("driver", "mydriver");
        data.put("fare", rand.nextDouble() * 100.0);
        data.put("_hoodie_is_deleted", false);
        String json = MAPPER.writeValueAsString(data);
        Row rec = CONVERTER.convertToRow(json, postProcessSchema);
        BigDecimal actualField = rec.getDecimal(postProcessSchema.getField("decfield").pos());
        this.validateSchemaCompatibility(Collections.singletonList(rec), postProcessSchema);
        Assertions.assertEquals((Object)decfield, (Object)actualField);
    }

    private void validateSchemaCompatibility(List<Row> rows, Schema schema) {
        StructType rowSchema = AvroConversionUtils.convertAvroSchemaToStructType((Schema)schema);
        Dataset dataset = spark.createDataFrame(rows, rowSchema);
        Assertions.assertDoesNotThrow(() -> ((Dataset)dataset).collect(), (String)"Schema validation and dataset creation should not throw any exceptions.");
    }
}

