/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import java.time.Duration;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.iceberg.CommitCallbackTransaction;
import org.apache.iceberg.HasTableOperations;
import org.apache.iceberg.MetadataTableType;
import org.apache.iceberg.MetadataTableUtils;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SortOrder;
import org.apache.iceberg.Table;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.Transaction;
import org.apache.iceberg.catalog.Catalog;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.AlreadyExistsException;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.shaded.com.github.benmanes.caffeine.cache.Cache;
import org.apache.iceberg.shaded.com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.iceberg.shaded.com.github.benmanes.caffeine.cache.RemovalCause;
import org.apache.iceberg.shaded.com.github.benmanes.caffeine.cache.RemovalListener;
import org.apache.iceberg.shaded.com.github.benmanes.caffeine.cache.Ticker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CachingCatalog
implements Catalog {
    private static final Logger LOG = LoggerFactory.getLogger(CachingCatalog.class);
    private final Catalog catalog;
    private final boolean caseSensitive;
    protected final long expirationIntervalMillis;
    protected final Cache<TableIdentifier, Table> tableCache;

    public static Catalog wrap(Catalog catalog) {
        return CachingCatalog.wrap(catalog, -1L);
    }

    public static Catalog wrap(Catalog catalog, long expirationIntervalMillis) {
        return CachingCatalog.wrap(catalog, true, expirationIntervalMillis);
    }

    public static Catalog wrap(Catalog catalog, boolean caseSensitive, long expirationIntervalMillis) {
        return new CachingCatalog(catalog, caseSensitive, expirationIntervalMillis);
    }

    private CachingCatalog(Catalog catalog, boolean caseSensitive, long expirationIntervalMillis) {
        this(catalog, caseSensitive, expirationIntervalMillis, Ticker.systemTicker());
    }

    protected CachingCatalog(Catalog catalog, boolean caseSensitive, long expirationIntervalMillis, Ticker ticker) {
        Preconditions.checkArgument(expirationIntervalMillis != 0L, "When %s is set to 0, the catalog cache should be disabled. This indicates a bug.", (Object)"cache.expiration-interval-ms");
        this.catalog = catalog;
        this.caseSensitive = caseSensitive;
        this.expirationIntervalMillis = expirationIntervalMillis;
        this.tableCache = this.createTableCache(ticker);
    }

    private Cache<TableIdentifier, Table> createTableCache(Ticker ticker) {
        Caffeine<Object, Object> cacheBuilder = Caffeine.newBuilder().softValues();
        if (this.expirationIntervalMillis > 0L) {
            return cacheBuilder.removalListener(new MetadataTableInvalidatingRemovalListener()).executor(Runnable::run).expireAfterAccess(Duration.ofMillis(this.expirationIntervalMillis)).ticker(ticker).build();
        }
        return cacheBuilder.build();
    }

    private TableIdentifier canonicalizeIdentifier(TableIdentifier tableIdentifier) {
        if (this.caseSensitive) {
            return tableIdentifier;
        }
        return tableIdentifier.toLowerCase();
    }

    @Override
    public String name() {
        return this.catalog.name();
    }

    @Override
    public List<TableIdentifier> listTables(Namespace namespace) {
        return this.catalog.listTables(namespace);
    }

    @Override
    public Table loadTable(TableIdentifier ident) {
        TableIdentifier canonicalized = this.canonicalizeIdentifier(ident);
        Table cached = this.tableCache.getIfPresent(canonicalized);
        if (cached != null) {
            return cached;
        }
        if (MetadataTableUtils.hasMetadataTableName(canonicalized)) {
            TableIdentifier originTableIdentifier = TableIdentifier.of(canonicalized.namespace().levels());
            Table originTable = this.tableCache.get(originTableIdentifier, this.catalog::loadTable);
            if (originTable instanceof HasTableOperations) {
                TableOperations ops = ((HasTableOperations)((Object)originTable)).operations();
                MetadataTableType type = MetadataTableType.from(canonicalized.name());
                Table metadataTable = MetadataTableUtils.createMetadataTableInstance(ops, this.catalog.name(), originTableIdentifier, canonicalized, type);
                this.tableCache.put(canonicalized, metadataTable);
                return metadataTable;
            }
        }
        return this.tableCache.get(canonicalized, this.catalog::loadTable);
    }

    @Override
    public boolean dropTable(TableIdentifier ident, boolean purge) {
        boolean dropped = this.catalog.dropTable(ident, purge);
        this.invalidate(ident);
        return dropped;
    }

    @Override
    public void renameTable(TableIdentifier from, TableIdentifier to) {
        this.catalog.renameTable(from, to);
        this.invalidate(from);
    }

    @Override
    public void invalidateTable(TableIdentifier ident) {
        this.invalidate(ident);
    }

    private void invalidate(TableIdentifier ident) {
        TableIdentifier canonicalized = this.canonicalizeIdentifier(ident);
        this.tableCache.invalidate(canonicalized);
        this.tableCache.invalidateAll(this.metadataTableIdentifiers(canonicalized));
    }

    private Iterable<TableIdentifier> metadataTableIdentifiers(TableIdentifier ident) {
        ImmutableList.Builder builder = ImmutableList.builder();
        for (MetadataTableType type : MetadataTableType.values()) {
            builder.add(TableIdentifier.parse(ident + "." + type.name()));
            builder.add(TableIdentifier.parse(ident + "." + type.name().toLowerCase(Locale.ROOT)));
        }
        return builder.build();
    }

    @Override
    public Catalog.TableBuilder buildTable(TableIdentifier identifier, Schema schema) {
        return new CachingTableBuilder(identifier, schema);
    }

    private class CachingTableBuilder
    implements Catalog.TableBuilder {
        private final TableIdentifier ident;
        private final Catalog.TableBuilder innerBuilder;

        private CachingTableBuilder(TableIdentifier identifier, Schema schema) {
            this.innerBuilder = CachingCatalog.this.catalog.buildTable(identifier, schema);
            this.ident = identifier;
        }

        @Override
        public Catalog.TableBuilder withPartitionSpec(PartitionSpec spec) {
            this.innerBuilder.withPartitionSpec(spec);
            return this;
        }

        @Override
        public Catalog.TableBuilder withSortOrder(SortOrder sortOrder) {
            this.innerBuilder.withSortOrder(sortOrder);
            return this;
        }

        @Override
        public Catalog.TableBuilder withLocation(String location) {
            this.innerBuilder.withLocation(location);
            return this;
        }

        @Override
        public Catalog.TableBuilder withProperties(Map<String, String> properties) {
            this.innerBuilder.withProperties(properties);
            return this;
        }

        @Override
        public Catalog.TableBuilder withProperty(String key, String value) {
            this.innerBuilder.withProperty(key, value);
            return this;
        }

        @Override
        public Table create() {
            AtomicBoolean created = new AtomicBoolean(false);
            Table table = CachingCatalog.this.tableCache.get(CachingCatalog.this.canonicalizeIdentifier(this.ident), identifier -> {
                created.set(true);
                return this.innerBuilder.create();
            });
            if (!created.get()) {
                throw new AlreadyExistsException("Table already exists: %s", this.ident);
            }
            return table;
        }

        @Override
        public Transaction createTransaction() {
            return this.innerBuilder.createTransaction();
        }

        @Override
        public Transaction replaceTransaction() {
            return CommitCallbackTransaction.addCallback(this.innerBuilder.replaceTransaction(), () -> CachingCatalog.this.invalidate(this.ident));
        }

        @Override
        public Transaction createOrReplaceTransaction() {
            return CommitCallbackTransaction.addCallback(this.innerBuilder.createOrReplaceTransaction(), () -> CachingCatalog.this.invalidate(this.ident));
        }
    }

    class MetadataTableInvalidatingRemovalListener
    implements RemovalListener<TableIdentifier, Table> {
        MetadataTableInvalidatingRemovalListener() {
        }

        @Override
        public void onRemoval(TableIdentifier tableIdentifier, Table table, RemovalCause cause) {
            LOG.debug("Evicted {} from the table cache ({})", (Object)tableIdentifier, (Object)cause);
            if (RemovalCause.EXPIRED.equals((Object)cause) && !MetadataTableUtils.hasMetadataTableName(tableIdentifier)) {
                CachingCatalog.this.tableCache.invalidateAll(CachingCatalog.this.metadataTableIdentifiers(tableIdentifier));
            }
        }
    }
}

