/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.iceberg.FileScanTask;
import org.apache.iceberg.ManifestEntry;
import org.apache.iceberg.ManifestFile;
import org.apache.iceberg.ManifestGroup;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.util.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MicroBatches {
    private MicroBatches() {
    }

    public static MicroBatchBuilder from(Snapshot snapshot, FileIO io) {
        return new MicroBatchBuilder(snapshot, io);
    }

    public static class MicroBatchBuilder {
        private static final Logger LOG = LoggerFactory.getLogger(MicroBatchBuilder.class);
        private final Snapshot snapshot;
        private final FileIO io;
        private boolean caseSensitive;
        private Map<Integer, PartitionSpec> specsById;

        private MicroBatchBuilder(Snapshot snapshot, FileIO io) {
            this.snapshot = snapshot;
            this.io = io;
            this.caseSensitive = true;
        }

        public MicroBatchBuilder caseSensitive(boolean sensitive) {
            this.caseSensitive = sensitive;
            return this;
        }

        public MicroBatchBuilder specsById(Map<Integer, PartitionSpec> specs) {
            this.specsById = specs;
            return this;
        }

        public MicroBatch generate(long startFileIndex, long targetSizeInBytes, boolean scanAllFiles) {
            Preconditions.checkArgument(startFileIndex >= 0L, "startFileIndex is unexpectedly smaller than 0");
            Preconditions.checkArgument(targetSizeInBytes > 0L, "targetSizeInBytes should be larger than 0");
            List<ManifestFile> manifests = scanAllFiles ? this.snapshot.dataManifests(this.io) : this.snapshot.dataManifests(this.io).stream().filter(m4 -> m4.snapshotId().equals(this.snapshot.snapshotId())).collect(Collectors.toList());
            List<Pair<ManifestFile, Integer>> manifestIndexes = MicroBatchBuilder.indexManifests(manifests);
            List<Pair<ManifestFile, Integer>> skippedManifestIndexes = MicroBatchBuilder.skipManifests(manifestIndexes, startFileIndex);
            return this.generateMicroBatch(skippedManifestIndexes, startFileIndex, targetSizeInBytes, scanAllFiles);
        }

        private static List<Pair<ManifestFile, Integer>> indexManifests(List<ManifestFile> manifestFiles) {
            int currentFileIndex = 0;
            ArrayList<Pair<ManifestFile, Integer>> manifestIndexes = Lists.newArrayList();
            for (ManifestFile manifest : manifestFiles) {
                manifestIndexes.add(Pair.of(manifest, currentFileIndex));
                currentFileIndex += manifest.addedFilesCount() + manifest.existingFilesCount();
            }
            return manifestIndexes;
        }

        private static List<Pair<ManifestFile, Integer>> skipManifests(List<Pair<ManifestFile, Integer>> indexedManifests, long startFileIndex) {
            if (startFileIndex == 0L) {
                return indexedManifests;
            }
            int manifestIndex = 0;
            for (Pair<ManifestFile, Integer> manifest : indexedManifests) {
                if ((long)manifest.second().intValue() > startFileIndex) break;
                ++manifestIndex;
            }
            return indexedManifests.subList(manifestIndex - 1, indexedManifests.size());
        }

        private MicroBatch generateMicroBatch(List<Pair<ManifestFile, Integer>> indexedManifests, long startFileIndex, long targetSizeInBytes, boolean scanAllFiles) {
            if (indexedManifests.isEmpty()) {
                return new MicroBatch(this.snapshot.snapshotId(), startFileIndex, startFileIndex + 1L, 0L, Collections.emptyList(), true);
            }
            long currentSizeInBytes = 0L;
            int currentFileIndex = 0;
            boolean isLastIndex = false;
            ArrayList<FileScanTask> tasks = Lists.newArrayList();
            for (int idx = 0; idx < indexedManifests.size(); ++idx) {
                currentFileIndex = indexedManifests.get(idx).second();
                try (CloseableIterable<FileScanTask> taskIterable = this.open(indexedManifests.get(idx).first(), scanAllFiles);
                     Iterator taskIter = taskIterable.iterator();){
                    while (taskIter.hasNext()) {
                        FileScanTask task = (FileScanTask)taskIter.next();
                        if ((long)currentFileIndex >= startFileIndex) {
                            tasks.add(task);
                            currentSizeInBytes += task.length();
                        }
                        ++currentFileIndex;
                        if (currentSizeInBytes < targetSizeInBytes) continue;
                        break;
                    }
                    if (idx + 1 == indexedManifests.size() && !taskIter.hasNext()) {
                        isLastIndex = true;
                    }
                }
                catch (IOException ioe) {
                    LOG.warn("Failed to close task iterable", (Throwable)ioe);
                }
                if (currentSizeInBytes < targetSizeInBytes) continue;
                if (tasks.size() <= 1 || currentSizeInBytes <= targetSizeInBytes) break;
                FileScanTask extraTask = (FileScanTask)tasks.remove(tasks.size() - 1);
                currentSizeInBytes -= extraTask.length();
                --currentFileIndex;
                isLastIndex = false;
                break;
            }
            return new MicroBatch(this.snapshot.snapshotId(), startFileIndex, currentFileIndex, currentSizeInBytes, tasks, isLastIndex);
        }

        private CloseableIterable<FileScanTask> open(ManifestFile manifestFile, boolean scanAllFiles) {
            ManifestGroup manifestGroup = new ManifestGroup(this.io, ImmutableList.of(manifestFile)).specsById(this.specsById).caseSensitive(this.caseSensitive);
            if (!scanAllFiles) {
                manifestGroup = manifestGroup.filterManifestEntries(entry -> entry.snapshotId().longValue() == this.snapshot.snapshotId() && entry.status() == ManifestEntry.Status.ADDED).ignoreDeleted();
            }
            return manifestGroup.planFiles();
        }
    }

    public static class MicroBatch {
        private final long snapshotId;
        private final long startFileIndex;
        private final long endFileIndex;
        private final long sizeInBytes;
        private final List<FileScanTask> tasks;
        private final boolean lastIndexOfSnapshot;

        private MicroBatch(long snapshotId, long startFileIndex, long endFileIndex, long sizeInBytes, List<FileScanTask> tasks, boolean lastIndexOfSnapshot) {
            this.snapshotId = snapshotId;
            this.startFileIndex = startFileIndex;
            this.endFileIndex = endFileIndex;
            this.sizeInBytes = sizeInBytes;
            this.tasks = tasks;
            this.lastIndexOfSnapshot = lastIndexOfSnapshot;
        }

        public long snapshotId() {
            return this.snapshotId;
        }

        public long startFileIndex() {
            return this.startFileIndex;
        }

        public long endFileIndex() {
            return this.endFileIndex;
        }

        public long sizeInBytes() {
            return this.sizeInBytes;
        }

        public List<FileScanTask> tasks() {
            return this.tasks;
        }

        public boolean lastIndexOfSnapshot() {
            return this.lastIndexOfSnapshot;
        }
    }
}

