/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.nessie;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.AlreadyExistsException;
import org.apache.iceberg.exceptions.CommitFailedException;
import org.apache.iceberg.exceptions.CommitStateUnknownException;
import org.apache.iceberg.exceptions.NamespaceNotEmptyException;
import org.apache.iceberg.exceptions.NoSuchNamespaceException;
import org.apache.iceberg.exceptions.NoSuchTableException;
import org.apache.iceberg.nessie.NessieUtil;
import org.apache.iceberg.nessie.UpdateableReference;
import org.apache.iceberg.relocated.com.google.common.base.Suppliers;
import org.apache.iceberg.util.Tasks;
import org.projectnessie.client.api.CommitMultipleOperationsBuilder;
import org.projectnessie.client.api.CreateNamespaceBuilder;
import org.projectnessie.client.api.DeleteNamespaceBuilder;
import org.projectnessie.client.api.GetContentBuilder;
import org.projectnessie.client.api.GetEntriesBuilder;
import org.projectnessie.client.api.GetMultipleNamespacesBuilder;
import org.projectnessie.client.api.GetNamespaceBuilder;
import org.projectnessie.client.api.NessieApiV1;
import org.projectnessie.client.api.UpdateNamespaceBuilder;
import org.projectnessie.client.http.HttpClientException;
import org.projectnessie.error.BaseNessieClientServerException;
import org.projectnessie.error.NessieConflictException;
import org.projectnessie.error.NessieNamespaceAlreadyExistsException;
import org.projectnessie.error.NessieNamespaceNotEmptyException;
import org.projectnessie.error.NessieNamespaceNotFoundException;
import org.projectnessie.error.NessieNotFoundException;
import org.projectnessie.error.NessieReferenceNotFoundException;
import org.projectnessie.model.Branch;
import org.projectnessie.model.Content;
import org.projectnessie.model.ContentKey;
import org.projectnessie.model.EntriesResponse;
import org.projectnessie.model.GetNamespacesResponse;
import org.projectnessie.model.IcebergTable;
import org.projectnessie.model.Operation;
import org.projectnessie.model.Reference;
import org.projectnessie.model.Tag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NessieIcebergClient
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(NessieIcebergClient.class);
    private final NessieApiV1 api;
    private final Supplier<UpdateableReference> reference;
    private final Map<String, String> catalogOptions;

    public NessieIcebergClient(NessieApiV1 api, String requestedRef, String requestedHash, Map<String, String> catalogOptions) {
        this.api = api;
        this.catalogOptions = catalogOptions;
        this.reference = Suppliers.memoize(() -> this.loadReference(requestedRef, requestedHash));
    }

    public NessieApiV1 getApi() {
        return this.api;
    }

    public UpdateableReference getRef() {
        return this.reference.get();
    }

    public void refresh() throws NessieNotFoundException {
        this.getRef().refresh(this.api);
    }

    public NessieIcebergClient withReference(String requestedRef, String hash) {
        if (null == requestedRef || this.getRef().getReference().getName().equals(requestedRef) && this.getRef().getHash().equals(hash)) {
            return this;
        }
        return new NessieIcebergClient(this.getApi(), requestedRef, hash, this.catalogOptions);
    }

    private UpdateableReference loadReference(String requestedRef, String hash) {
        try {
            Reference ref;
            Reference reference = ref = requestedRef == null ? this.api.getDefaultBranch() : this.api.getReference().refName(requestedRef).get();
            if (hash != null) {
                ref = ref instanceof Branch ? Branch.of(ref.getName(), hash) : Tag.of(ref.getName(), hash);
            }
            return new UpdateableReference(ref, hash != null);
        }
        catch (NessieNotFoundException ex) {
            if (requestedRef != null) {
                throw new IllegalArgumentException(String.format("Nessie ref '%s' does not exist", requestedRef), ex);
            }
            throw new IllegalArgumentException(String.format("Nessie does not have an existing default branch. Either configure an alternative ref via '%s' or create the default branch on the server.", "nessie.ref"), ex);
        }
    }

    public List<TableIdentifier> listTables(Namespace namespace) {
        try {
            return ((GetEntriesBuilder)this.api.getEntries().reference(this.getRef().getReference())).get().getEntries().stream().filter(this.namespacePredicate(namespace)).filter(e -> Content.Type.ICEBERG_TABLE == e.getType()).map(this::toIdentifier).collect(Collectors.toList());
        }
        catch (NessieNotFoundException ex) {
            throw new NoSuchNamespaceException(ex, "Unable to list tables due to missing ref '%s'", this.getRef().getName());
        }
    }

    private Predicate<EntriesResponse.Entry> namespacePredicate(Namespace ns) {
        if (ns == null) {
            return e -> true;
        }
        List<String> namespace = Arrays.asList(ns.levels());
        return e -> {
            List<String> names = e.getName().getElements();
            if (names.size() <= namespace.size()) {
                return false;
            }
            return namespace.equals(names.subList(0, namespace.size()));
        };
    }

    private TableIdentifier toIdentifier(EntriesResponse.Entry entry) {
        List<String> elements = entry.getName().getElements();
        return TableIdentifier.of(elements.toArray(new String[elements.size()]));
    }

    public IcebergTable table(TableIdentifier tableIdentifier) {
        try {
            ContentKey key = NessieUtil.toKey(tableIdentifier);
            Content table = ((GetContentBuilder)this.api.getContent().key(key).reference(this.getRef().getReference())).get().get(key);
            return table != null ? (IcebergTable)table.unwrap(IcebergTable.class).orElse(null) : null;
        }
        catch (NessieNotFoundException e) {
            return null;
        }
    }

    public void createNamespace(Namespace namespace, Map<String, String> metadata) {
        try {
            this.getRef().checkMutable();
            ((CreateNamespaceBuilder)((CreateNamespaceBuilder)this.getApi().createNamespace().reference(this.getRef().getReference())).namespace(org.projectnessie.model.Namespace.of(namespace.levels()))).properties(metadata).create();
            this.refresh();
        }
        catch (NessieNamespaceAlreadyExistsException e) {
            throw new AlreadyExistsException(e, "Namespace already exists: %s", namespace);
        }
        catch (NessieNotFoundException e) {
            throw new RuntimeException(String.format("Cannot create Namespace '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public List<Namespace> listNamespaces(Namespace namespace) throws NoSuchNamespaceException {
        try {
            GetNamespacesResponse response = ((GetMultipleNamespacesBuilder)((GetMultipleNamespacesBuilder)this.getApi().getMultipleNamespaces().reference(this.getRef().getReference())).namespace(org.projectnessie.model.Namespace.of(namespace.levels()))).get();
            return response.getNamespaces().stream().map(ns -> Namespace.of(ns.getElements().toArray(new String[0]))).collect(Collectors.toList());
        }
        catch (NessieReferenceNotFoundException e) {
            throw new RuntimeException(String.format("Cannot list Namespaces starting from '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public boolean dropNamespace(Namespace namespace) throws NamespaceNotEmptyException {
        try {
            this.getRef().checkMutable();
            ((DeleteNamespaceBuilder)((DeleteNamespaceBuilder)this.getApi().deleteNamespace().reference(this.getRef().getReference())).namespace(org.projectnessie.model.Namespace.of(namespace.levels()))).delete();
            this.refresh();
            return true;
        }
        catch (NessieNamespaceNotFoundException e) {
            return false;
        }
        catch (NessieNotFoundException e) {
            LOG.error("Cannot drop Namespace '{}': ref '{}' is no longer valid.", new Object[]{namespace, this.getRef().getName(), e});
            return false;
        }
        catch (NessieNamespaceNotEmptyException e) {
            throw new NamespaceNotEmptyException(e, "Namespace '%s' is not empty. One or more tables exist.", namespace);
        }
    }

    public Map<String, String> loadNamespaceMetadata(Namespace namespace) throws NoSuchNamespaceException {
        try {
            return ((GetNamespaceBuilder)((GetNamespaceBuilder)this.getApi().getNamespace().reference(this.getRef().getReference())).namespace(org.projectnessie.model.Namespace.of(namespace.levels()))).get().getProperties();
        }
        catch (NessieNamespaceNotFoundException e) {
            throw new NoSuchNamespaceException(e, "Namespace does not exist: %s", namespace);
        }
        catch (NessieReferenceNotFoundException e) {
            throw new RuntimeException(String.format("Cannot load Namespace '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public boolean setProperties(Namespace namespace, Map<String, String> properties) {
        try {
            ((UpdateNamespaceBuilder)((UpdateNamespaceBuilder)this.getApi().updateProperties().reference(this.getRef().getReference())).namespace(org.projectnessie.model.Namespace.of(namespace.levels()))).updateProperties(properties).update();
            this.refresh();
            return true;
        }
        catch (NessieNamespaceNotFoundException e) {
            throw new NoSuchNamespaceException(e, "Namespace does not exist: %s", namespace);
        }
        catch (NessieNotFoundException e) {
            throw new RuntimeException(String.format("Cannot update properties on Namespace '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public boolean removeProperties(Namespace namespace, Set<String> properties) {
        try {
            ((UpdateNamespaceBuilder)((UpdateNamespaceBuilder)this.getApi().updateProperties().reference(this.getRef().getReference())).namespace(org.projectnessie.model.Namespace.of(namespace.levels()))).removeProperties(properties).update();
            this.refresh();
            return true;
        }
        catch (NessieNamespaceNotFoundException e) {
            throw new NoSuchNamespaceException(e, "Namespace does not exist: %s", namespace);
        }
        catch (NessieNotFoundException e) {
            throw new RuntimeException(String.format("Cannot remove properties from Namespace '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public void renameTable(TableIdentifier from, TableIdentifier to) {
        this.getRef().checkMutable();
        IcebergTable existingFromTable = this.table(from);
        if (existingFromTable == null) {
            throw new NoSuchTableException("Table does not exist: %s", from.name());
        }
        IcebergTable existingToTable = this.table(to);
        if (existingToTable != null) {
            throw new AlreadyExistsException("Table already exists: %s", to.name());
        }
        CommitMultipleOperationsBuilder operations = this.getApi().commitMultipleOperations().commitMeta(NessieUtil.buildCommitMetadata(String.format("Iceberg rename table from '%s' to '%s'", from, to), this.catalogOptions)).operation(Operation.Put.of(NessieUtil.toKey(to), existingFromTable, existingFromTable)).operation(Operation.Delete.of(NessieUtil.toKey(from)));
        try {
            Tasks.foreach(operations).retry(5).stopRetryOn(NessieNotFoundException.class).throwFailureWhenFinished().onFailure((o, exception) -> this.refresh()).run(ops -> {
                Branch branch = ((CommitMultipleOperationsBuilder)ops.branch(this.getRef().getAsBranch())).commit();
                this.getRef().updateReference(branch);
            }, BaseNessieClientServerException.class);
        }
        catch (NessieNotFoundException e) {
            throw new RuntimeException(String.format("Cannot rename table '%s' to '%s': ref '%s' no longer exists.", from.name(), to.name(), this.getRef().getName()), e);
        }
        catch (BaseNessieClientServerException e) {
            throw new CommitFailedException(e, "Cannot rename table '%s' to '%s': the current reference is not up to date.", from.name(), to.name());
        }
        catch (HttpClientException ex) {
            throw new CommitStateUnknownException(ex);
        }
    }

    public boolean dropTable(TableIdentifier identifier, boolean purge) {
        this.getRef().checkMutable();
        IcebergTable existingTable = this.table(identifier);
        if (existingTable == null) {
            return false;
        }
        if (purge) {
            LOG.info("Purging data for table {} was set to true but is ignored", (Object)identifier.toString());
        }
        CommitMultipleOperationsBuilder commitBuilderBase = this.getApi().commitMultipleOperations().commitMeta(NessieUtil.buildCommitMetadata(String.format("Iceberg delete table %s", identifier), this.catalogOptions)).operation(Operation.Delete.of(NessieUtil.toKey(identifier)));
        boolean threw = true;
        try {
            Tasks.foreach(commitBuilderBase).retry(5).stopRetryOn(NessieNotFoundException.class).throwFailureWhenFinished().onFailure((o, exception) -> this.refresh()).run(commitBuilder -> {
                Branch branch = ((CommitMultipleOperationsBuilder)commitBuilder.branch(this.getRef().getAsBranch())).commit();
                this.getRef().updateReference(branch);
            }, BaseNessieClientServerException.class);
            threw = false;
        }
        catch (NessieConflictException e) {
            LOG.error("Cannot drop table: failed after retry (update ref '{}' and retry)", (Object)this.getRef().getName(), (Object)e);
        }
        catch (NessieNotFoundException e) {
            LOG.error("Cannot drop table: ref '{}' is no longer valid.", (Object)this.getRef().getName(), (Object)e);
        }
        catch (BaseNessieClientServerException e) {
            LOG.error("Cannot drop table: unknown error", (Throwable)e);
        }
        return !threw;
    }

    @Override
    public void close() {
        if (null != this.api) {
            this.api.close();
        }
    }
}

