/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.flink.maintenance.operator;

import java.io.IOException;
import java.util.List;
import org.apache.flink.annotation.Internal;
import org.apache.flink.api.common.state.ListState;
import org.apache.flink.api.common.state.ListStateDescriptor;
import org.apache.flink.api.common.state.ValueState;
import org.apache.flink.api.common.state.ValueStateDescriptor;
import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.apache.flink.api.common.typeinfo.Types;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.metrics.Counter;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.state.FunctionInitializationContext;
import org.apache.flink.runtime.state.FunctionSnapshotContext;
import org.apache.flink.streaming.api.TimerService;
import org.apache.flink.streaming.api.checkpoint.CheckpointedFunction;
import org.apache.flink.streaming.api.functions.KeyedProcessFunction;
import org.apache.flink.util.Collector;
import org.apache.iceberg.flink.TableLoader;
import org.apache.iceberg.flink.maintenance.api.Trigger;
import org.apache.iceberg.flink.maintenance.api.TriggerLockFactory;
import org.apache.iceberg.flink.maintenance.operator.TableChange;
import org.apache.iceberg.flink.maintenance.operator.TableMaintenanceMetrics;
import org.apache.iceberg.flink.maintenance.operator.TriggerEvaluator;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class TriggerManager
extends KeyedProcessFunction<Boolean, TableChange, Trigger>
implements CheckpointedFunction {
    private static final Logger LOG = LoggerFactory.getLogger(TriggerManager.class);
    private final String tableName;
    private final TriggerLockFactory lockFactory;
    private final List<String> maintenanceTaskNames;
    private final List<TriggerEvaluator> evaluators;
    private final long minFireDelayMs;
    private final long lockCheckDelayMs;
    private transient Counter rateLimiterTriggeredCounter;
    private transient Counter concurrentRunThrottledCounter;
    private transient Counter nothingToTriggerCounter;
    private transient List<Counter> triggerCounters;
    private transient ValueState<Long> nextEvaluationTimeState;
    private transient ListState<TableChange> accumulatedChangesState;
    private transient ListState<Long> lastTriggerTimesState;
    private transient Long nextEvaluationTime;
    private transient List<TableChange> accumulatedChanges;
    private transient List<Long> lastTriggerTimes;
    private transient TriggerLockFactory.Lock lock;
    private transient TriggerLockFactory.Lock recoveryLock;
    private transient boolean shouldRestoreTasks = false;
    private transient boolean inited = false;
    private transient int startsFrom = 0;
    private transient boolean triggered = false;

    public TriggerManager(TableLoader tableLoader, TriggerLockFactory lockFactory, List<String> maintenanceTaskNames, List<TriggerEvaluator> evaluators, long minFireDelayMs, long lockCheckDelayMs) {
        Preconditions.checkNotNull(tableLoader, "Table loader should no be null");
        Preconditions.checkNotNull(lockFactory, "Lock factory should no be null");
        Preconditions.checkArgument(maintenanceTaskNames != null && !maintenanceTaskNames.isEmpty(), "Invalid maintenance task names: null or empty");
        Preconditions.checkArgument(evaluators != null && !evaluators.isEmpty(), "Invalid evaluators: null or empty");
        Preconditions.checkArgument(maintenanceTaskNames.size() == evaluators.size(), "Provide a name and evaluator for all of the maintenance tasks");
        Preconditions.checkArgument(minFireDelayMs > 0L, "Minimum fire delay should be at least 1.");
        Preconditions.checkArgument(lockCheckDelayMs > 0L, "Minimum lock delay rate should be at least 1 ms.");
        tableLoader.open();
        this.tableName = tableLoader.loadTable().name();
        this.lockFactory = lockFactory;
        this.maintenanceTaskNames = maintenanceTaskNames;
        this.evaluators = evaluators;
        this.minFireDelayMs = minFireDelayMs;
        this.lockCheckDelayMs = lockCheckDelayMs;
    }

    public void open(Configuration parameters) throws Exception {
        MetricGroup mainGroup = TableMaintenanceMetrics.groupFor(this.getRuntimeContext(), this.tableName);
        this.rateLimiterTriggeredCounter = mainGroup.counter("rateLimiterTriggered");
        this.concurrentRunThrottledCounter = mainGroup.counter("concurrentRunThrottled");
        this.nothingToTriggerCounter = mainGroup.counter("nothingToTrigger");
        this.triggerCounters = Lists.newArrayListWithExpectedSize(this.maintenanceTaskNames.size());
        for (int taskIndex = 0; taskIndex < this.maintenanceTaskNames.size(); ++taskIndex) {
            this.triggerCounters.add(TableMaintenanceMetrics.groupFor(mainGroup, this.maintenanceTaskNames.get(taskIndex), taskIndex).counter("triggered"));
        }
        this.nextEvaluationTimeState = this.getRuntimeContext().getState(new ValueStateDescriptor("triggerManagerNextTriggerTime", Types.LONG));
        this.accumulatedChangesState = this.getRuntimeContext().getListState(new ListStateDescriptor("triggerManagerAccumulatedChange", TypeInformation.of(TableChange.class)));
        this.lastTriggerTimesState = this.getRuntimeContext().getListState(new ListStateDescriptor("triggerManagerLastTriggerTime", Types.LONG));
    }

    public void snapshotState(FunctionSnapshotContext context) throws Exception {
        if (this.inited) {
            this.nextEvaluationTimeState.update((Object)this.nextEvaluationTime);
            this.accumulatedChangesState.update(this.accumulatedChanges);
            this.lastTriggerTimesState.update(this.lastTriggerTimes);
            LOG.info("Storing state: nextEvaluationTime {}, accumulatedChanges {}, lastTriggerTimes {}", new Object[]{this.nextEvaluationTime, this.accumulatedChanges, this.lastTriggerTimes});
        } else {
            LOG.info("Not initialized, state is not stored");
        }
    }

    public void initializeState(FunctionInitializationContext context) throws Exception {
        LOG.info("Initializing state restored: {}", (Object)context.isRestored());
        this.lockFactory.open();
        this.lock = this.lockFactory.createLock();
        this.recoveryLock = this.lockFactory.createRecoveryLock();
        if (context.isRestored()) {
            this.shouldRestoreTasks = true;
        } else {
            this.lock.unlock();
            this.recoveryLock.unlock();
        }
    }

    public void processElement(TableChange change, KeyedProcessFunction.Context ctx, Collector<Trigger> out) throws Exception {
        this.init(out, ctx.timerService());
        this.accumulatedChanges.forEach(tableChange -> tableChange.merge(change));
        long current = ctx.timerService().currentProcessingTime();
        if (this.nextEvaluationTime == null) {
            this.checkAndFire(current, ctx.timerService(), out);
        } else {
            LOG.info("Trigger manager rate limiter triggered current: {}, next: {}, accumulated changes: {}", new Object[]{current, this.nextEvaluationTime, this.accumulatedChanges});
            this.rateLimiterTriggeredCounter.inc();
        }
    }

    public void onTimer(long timestamp, KeyedProcessFunction.OnTimerContext ctx, Collector<Trigger> out) throws Exception {
        this.init(out, ctx.timerService());
        this.nextEvaluationTime = null;
        this.checkAndFire(ctx.timerService().currentProcessingTime(), ctx.timerService(), out);
    }

    public void close() throws IOException {
        this.lockFactory.close();
    }

    private void checkAndFire(long current, TimerService timerService, Collector<Trigger> out) {
        Integer taskToStart;
        if (this.shouldRestoreTasks) {
            if (this.recoveryLock.isHeld()) {
                LOG.debug("The recovery lock is still held at {}", (Object)current);
                this.schedule(timerService, current + this.lockCheckDelayMs);
                return;
            }
            LOG.info("The recovery is finished at {}", (Object)current);
            this.shouldRestoreTasks = false;
        }
        if ((taskToStart = TriggerManager.nextTrigger(this.evaluators, this.accumulatedChanges, this.lastTriggerTimes, current, this.startsFrom)) == null) {
            if (!this.triggered) {
                this.nothingToTriggerCounter.inc();
                LOG.debug("Nothing to execute at {} for collected: {}", (Object)current, this.accumulatedChanges);
            } else {
                LOG.debug("Execution check finished");
            }
            this.startsFrom = 0;
            this.triggered = false;
            return;
        }
        if (this.lock.tryLock()) {
            TableChange change = this.accumulatedChanges.get(taskToStart);
            out.collect((Object)Trigger.create(current, taskToStart));
            LOG.debug("Fired event with time: {}, collected: {} for {}", new Object[]{current, change, this.tableName});
            this.triggerCounters.get(taskToStart).inc();
            this.accumulatedChanges.set(taskToStart, TableChange.empty());
            this.lastTriggerTimes.set(taskToStart, current);
            this.schedule(timerService, current + this.minFireDelayMs);
            this.startsFrom = (taskToStart + 1) % this.evaluators.size();
            this.triggered = true;
        } else {
            LOG.info("Failed to acquire lock. Delaying task to {}", (Object)(current + this.lockCheckDelayMs));
            this.startsFrom = taskToStart;
            this.concurrentRunThrottledCounter.inc();
            this.schedule(timerService, current + this.lockCheckDelayMs);
        }
        timerService.registerProcessingTimeTimer(this.nextEvaluationTime.longValue());
    }

    private void schedule(TimerService timerService, long time) {
        this.nextEvaluationTime = time;
        timerService.registerProcessingTimeTimer(time);
    }

    private static Integer nextTrigger(List<TriggerEvaluator> evaluators, List<TableChange> changes, List<Long> lastTriggerTimes, long currentTime, int startPos) {
        int current = startPos;
        do {
            if (!evaluators.get(current).check(changes.get(current), lastTriggerTimes.get(current), currentTime)) continue;
            return current;
        } while ((current = (current + 1) % evaluators.size()) != startPos);
        return null;
    }

    private void init(Collector<Trigger> out, TimerService timerService) throws Exception {
        if (!this.inited) {
            long current = timerService.currentProcessingTime();
            this.nextEvaluationTime = (Long)this.nextEvaluationTimeState.value();
            this.accumulatedChanges = Lists.newArrayList((Iterable)this.accumulatedChangesState.get());
            this.lastTriggerTimes = Lists.newArrayList((Iterable)this.lastTriggerTimesState.get());
            if (this.accumulatedChanges.isEmpty()) {
                for (int i = 0; i < this.evaluators.size(); ++i) {
                    this.accumulatedChanges.add(TableChange.empty());
                    this.lastTriggerTimes.add(current);
                }
            }
            if (this.shouldRestoreTasks) {
                this.recoveryLock.tryLock();
                out.collect((Object)Trigger.recovery(current));
                if (this.nextEvaluationTime == null) {
                    this.schedule(timerService, current + this.minFireDelayMs);
                }
            }
            this.inited = true;
        }
    }
}

