/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.parquet;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.iceberg.FieldMetrics;
import org.apache.iceberg.Metrics;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.MetricsModes;
import org.apache.iceberg.MetricsUtil;
import org.apache.iceberg.Schema;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.expressions.Literal;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.mapping.NameMapping;
import org.apache.iceberg.parquet.ParquetConversions;
import org.apache.iceberg.parquet.ParquetIO;
import org.apache.iceberg.parquet.ParquetSchemaUtil;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.shaded.org.apache.parquet.column.ColumnDescriptor;
import org.apache.iceberg.shaded.org.apache.parquet.column.Dictionary;
import org.apache.iceberg.shaded.org.apache.parquet.column.Encoding;
import org.apache.iceberg.shaded.org.apache.parquet.column.EncodingStats;
import org.apache.iceberg.shaded.org.apache.parquet.column.page.DictionaryPage;
import org.apache.iceberg.shaded.org.apache.parquet.column.page.PageReader;
import org.apache.iceberg.shaded.org.apache.parquet.column.statistics.Statistics;
import org.apache.iceberg.shaded.org.apache.parquet.hadoop.ParquetFileReader;
import org.apache.iceberg.shaded.org.apache.parquet.hadoop.metadata.BlockMetaData;
import org.apache.iceberg.shaded.org.apache.parquet.hadoop.metadata.ColumnChunkMetaData;
import org.apache.iceberg.shaded.org.apache.parquet.hadoop.metadata.ColumnPath;
import org.apache.iceberg.shaded.org.apache.parquet.hadoop.metadata.ParquetMetadata;
import org.apache.iceberg.shaded.org.apache.parquet.io.ParquetDecodingException;
import org.apache.iceberg.shaded.org.apache.parquet.schema.MessageType;
import org.apache.iceberg.shaded.org.apache.parquet.schema.PrimitiveType;
import org.apache.iceberg.types.Conversions;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.BinaryUtil;
import org.apache.iceberg.util.UnicodeUtil;

public class ParquetUtil {
    private static final long UNIX_EPOCH_JULIAN = 2440588L;

    private ParquetUtil() {
    }

    public static Metrics fileMetrics(InputFile file, MetricsConfig metricsConfig) {
        return ParquetUtil.fileMetrics(file, metricsConfig, null);
    }

    public static Metrics fileMetrics(InputFile file, MetricsConfig metricsConfig, NameMapping nameMapping) {
        Metrics metrics;
        block8: {
            ParquetFileReader reader = ParquetFileReader.open(ParquetIO.file(file));
            try {
                metrics = ParquetUtil.footerMetrics(reader.getFooter(), Stream.empty(), metricsConfig, nameMapping);
                if (reader == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (reader != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new RuntimeIOException(e, "Failed to read footer of file: %s", file);
                }
            }
            reader.close();
        }
        return metrics;
    }

    public static Metrics footerMetrics(ParquetMetadata metadata, Stream<FieldMetrics<?>> fieldMetrics, MetricsConfig metricsConfig) {
        return ParquetUtil.footerMetrics(metadata, fieldMetrics, metricsConfig, null);
    }

    public static Metrics footerMetrics(ParquetMetadata metadata, Stream<FieldMetrics<?>> fieldMetrics, MetricsConfig metricsConfig, NameMapping nameMapping) {
        Preconditions.checkNotNull(fieldMetrics, "fieldMetrics should not be null");
        long rowCount = 0L;
        HashMap<Integer, Long> columnSizes = Maps.newHashMap();
        HashMap<Integer, Long> valueCounts = Maps.newHashMap();
        HashMap<Integer, Long> nullValueCounts = Maps.newHashMap();
        HashMap<Integer, Literal<?>> lowerBounds = Maps.newHashMap();
        HashMap<Integer, Literal<?>> upperBounds = Maps.newHashMap();
        HashSet<Integer> missingStats = Sets.newHashSet();
        MessageType parquetTypeWithIds = ParquetUtil.getParquetTypeWithIds(metadata, nameMapping);
        Schema fileSchema = ParquetSchemaUtil.convertAndPrune(parquetTypeWithIds);
        Map<Integer, FieldMetrics<?>> fieldMetricsMap = fieldMetrics.collect(Collectors.toMap(FieldMetrics::id, Function.identity()));
        List<BlockMetaData> blocks = metadata.getBlocks();
        for (BlockMetaData block : blocks) {
            rowCount += block.getRowCount();
            for (ColumnChunkMetaData column : block.getColumns()) {
                MetricsModes.MetricsMode metricsMode;
                Integer fieldId = fileSchema.aliasToId(column.getPath().toDotString());
                if (fieldId == null || (metricsMode = MetricsUtil.metricsMode(fileSchema, metricsConfig, fieldId)) == MetricsModes.None.get()) continue;
                ParquetUtil.increment(columnSizes, fieldId, column.getTotalSize());
                ParquetUtil.increment(valueCounts, fieldId, column.getValueCount());
                Statistics stats = column.getStatistics();
                if (stats != null && !stats.isEmpty()) {
                    Types.NestedField field;
                    ParquetUtil.increment(nullValueCounts, fieldId, stats.getNumNulls());
                    if (metricsMode == MetricsModes.Counts.get() || fieldMetricsMap.containsKey(fieldId) || (field = fileSchema.findField(fieldId)) == null || !stats.hasNonNullValue() || !ParquetUtil.shouldStoreBounds(column, fileSchema)) continue;
                    Literal min2 = ParquetConversions.fromParquetPrimitive(field.type(), column.getPrimitiveType(), stats.genericGetMin());
                    ParquetUtil.updateMin(lowerBounds, fieldId, field.type(), min2, metricsMode);
                    Literal max = ParquetConversions.fromParquetPrimitive(field.type(), column.getPrimitiveType(), stats.genericGetMax());
                    ParquetUtil.updateMax(upperBounds, fieldId, field.type(), max, metricsMode);
                    continue;
                }
                missingStats.add(fieldId);
            }
        }
        for (Integer fieldId : missingStats) {
            nullValueCounts.remove(fieldId);
            lowerBounds.remove(fieldId);
            upperBounds.remove(fieldId);
        }
        ParquetUtil.updateFromFieldMetrics(fieldMetricsMap, metricsConfig, fileSchema, lowerBounds, upperBounds);
        return new Metrics(rowCount, columnSizes, valueCounts, nullValueCounts, MetricsUtil.createNanValueCounts(fieldMetricsMap.values().stream(), metricsConfig, fileSchema), ParquetUtil.toBufferMap(fileSchema, lowerBounds), ParquetUtil.toBufferMap(fileSchema, upperBounds));
    }

    private static void updateFromFieldMetrics(Map<Integer, FieldMetrics<?>> idToFieldMetricsMap, MetricsConfig metricsConfig, Schema schema, Map<Integer, Literal<?>> lowerBounds, Map<Integer, Literal<?>> upperBounds) {
        idToFieldMetricsMap.entrySet().forEach(entry -> {
            int fieldId = (Integer)entry.getKey();
            FieldMetrics metrics = (FieldMetrics)entry.getValue();
            MetricsModes.MetricsMode metricsMode = MetricsUtil.metricsMode(schema, metricsConfig, fieldId);
            if (metricsMode != MetricsModes.None.get()) {
                if (!metrics.hasBounds()) {
                    lowerBounds.remove(fieldId);
                    upperBounds.remove(fieldId);
                } else if (metrics.upperBound() instanceof Float) {
                    lowerBounds.put(fieldId, Literal.of(((Float)metrics.lowerBound()).floatValue()));
                    upperBounds.put(fieldId, Literal.of(((Float)metrics.upperBound()).floatValue()));
                } else if (metrics.upperBound() instanceof Double) {
                    lowerBounds.put(fieldId, Literal.of((Double)metrics.lowerBound()));
                    upperBounds.put(fieldId, Literal.of((Double)metrics.upperBound()));
                } else {
                    throw new UnsupportedOperationException("Expected only float or double column metrics");
                }
            }
        });
    }

    private static MessageType getParquetTypeWithIds(ParquetMetadata metadata, NameMapping nameMapping) {
        MessageType type = metadata.getFileMetaData().getSchema();
        if (ParquetSchemaUtil.hasIds(type)) {
            return type;
        }
        if (nameMapping != null) {
            return ParquetSchemaUtil.applyNameMapping(type, nameMapping);
        }
        return ParquetSchemaUtil.addFallbackIds(type);
    }

    public static List<Long> getSplitOffsets(ParquetMetadata md) {
        ArrayList<Long> splitOffsets = Lists.newArrayListWithExpectedSize(md.getBlocks().size());
        for (BlockMetaData blockMetaData : md.getBlocks()) {
            splitOffsets.add(blockMetaData.getStartingPos());
        }
        Collections.sort(splitOffsets);
        return splitOffsets;
    }

    private static boolean shouldStoreBounds(ColumnChunkMetaData column, Schema schema) {
        if (column.getPrimitiveType().getPrimitiveTypeName() == PrimitiveType.PrimitiveTypeName.INT96) {
            return false;
        }
        ColumnPath columnPath = column.getPath();
        Iterator<String> pathIterator = columnPath.iterator();
        Type currentType = schema.asStruct();
        while (pathIterator.hasNext()) {
            if (currentType == null || !currentType.isStructType()) {
                return false;
            }
            String fieldName = pathIterator.next();
            currentType = currentType.asStructType().fieldType(fieldName);
        }
        return currentType != null && currentType.isPrimitiveType();
    }

    private static void increment(Map<Integer, Long> columns, int fieldId, long amount) {
        if (columns != null) {
            if (columns.containsKey(fieldId)) {
                columns.put(fieldId, columns.get(fieldId) + amount);
            } else {
                columns.put(fieldId, amount);
            }
        }
    }

    private static <T> void updateMin(Map<Integer, Literal<?>> lowerBounds, int id, Type type, Literal<T> min2, MetricsModes.MetricsMode metricsMode) {
        Literal<?> currentMin = lowerBounds.get(id);
        if (currentMin == null || min2.comparator().compare(min2.value(), currentMin.value()) < 0) {
            if (metricsMode == MetricsModes.Full.get()) {
                lowerBounds.put(id, min2);
            } else {
                MetricsModes.Truncate truncateMode = (MetricsModes.Truncate)metricsMode;
                int truncateLength = truncateMode.length();
                switch (type.typeId()) {
                    case STRING: {
                        lowerBounds.put(id, UnicodeUtil.truncateStringMin(min2, truncateLength));
                        break;
                    }
                    case FIXED: 
                    case BINARY: {
                        lowerBounds.put(id, BinaryUtil.truncateBinaryMin(min2, truncateLength));
                        break;
                    }
                    default: {
                        lowerBounds.put(id, min2);
                    }
                }
            }
        }
    }

    private static <T> void updateMax(Map<Integer, Literal<?>> upperBounds, int id, Type type, Literal<T> max, MetricsModes.MetricsMode metricsMode) {
        Literal<?> currentMax = upperBounds.get(id);
        if (currentMax == null || max.comparator().compare(max.value(), currentMax.value()) > 0) {
            if (metricsMode == MetricsModes.Full.get()) {
                upperBounds.put(id, max);
            } else {
                MetricsModes.Truncate truncateMode = (MetricsModes.Truncate)metricsMode;
                int truncateLength = truncateMode.length();
                switch (type.typeId()) {
                    case STRING: {
                        Literal<CharSequence> truncatedMaxString = UnicodeUtil.truncateStringMax(max, truncateLength);
                        if (truncatedMaxString == null) break;
                        upperBounds.put(id, truncatedMaxString);
                        break;
                    }
                    case FIXED: 
                    case BINARY: {
                        Literal<ByteBuffer> truncatedMaxBinary = BinaryUtil.truncateBinaryMax(max, truncateLength);
                        if (truncatedMaxBinary == null) break;
                        upperBounds.put(id, truncatedMaxBinary);
                        break;
                    }
                    default: {
                        upperBounds.put(id, max);
                    }
                }
            }
        }
    }

    private static Map<Integer, ByteBuffer> toBufferMap(Schema schema, Map<Integer, Literal<?>> map) {
        HashMap<Integer, ByteBuffer> bufferMap = Maps.newHashMap();
        for (Map.Entry<Integer, Literal<?>> entry : map.entrySet()) {
            bufferMap.put(entry.getKey(), Conversions.toByteBuffer(schema.findType(entry.getKey()), entry.getValue().value()));
        }
        return bufferMap;
    }

    public static boolean hasNonDictionaryPages(ColumnChunkMetaData meta) {
        EncodingStats stats = meta.getEncodingStats();
        if (stats != null) {
            return stats.hasNonDictionaryEncodedPages();
        }
        HashSet<Encoding> encodings = Sets.newHashSet(meta.getEncodings());
        if (encodings.remove((Object)Encoding.PLAIN_DICTIONARY)) {
            encodings.remove((Object)Encoding.RLE);
            encodings.remove((Object)Encoding.BIT_PACKED);
            return !encodings.isEmpty();
        }
        return true;
    }

    public static boolean hasNoBloomFilterPages(ColumnChunkMetaData meta) {
        return meta.getBloomFilterOffset() <= 0L;
    }

    public static Dictionary readDictionary(ColumnDescriptor desc, PageReader pageSource) {
        DictionaryPage dictionaryPage = pageSource.readDictionaryPage();
        if (dictionaryPage != null) {
            try {
                return dictionaryPage.getEncoding().initDictionary(desc, dictionaryPage);
            }
            catch (IOException e) {
                throw new ParquetDecodingException("could not decode the dictionary for " + desc, e);
            }
        }
        return null;
    }

    public static boolean isIntType(PrimitiveType primitiveType) {
        if (primitiveType.getOriginalType() != null) {
            switch (primitiveType.getOriginalType()) {
                case INT_8: 
                case INT_16: 
                case INT_32: 
                case DATE: {
                    return true;
                }
            }
            return false;
        }
        return primitiveType.getPrimitiveTypeName() == PrimitiveType.PrimitiveTypeName.INT32;
    }

    public static long extractTimestampInt96(ByteBuffer buffer) {
        long timeOfDayNanos = buffer.getLong();
        int julianDay = buffer.getInt();
        return TimeUnit.DAYS.toMicros((long)julianDay - 2440588L) + TimeUnit.NANOSECONDS.toMicros(timeOfDayNanos);
    }
}

