/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.io;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import org.apache.iceberg.exceptions.NotFoundException;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.io.ByteBufferInputStream;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.io.IOUtil;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.io.SeekableInputStream;
import org.apache.iceberg.relocated.com.google.common.base.MoreObjects;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.shaded.com.github.benmanes.caffeine.cache.Cache;
import org.apache.iceberg.shaded.com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.iceberg.shaded.com.github.benmanes.caffeine.cache.stats.CacheStats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContentCache {
    private static final Logger LOG = LoggerFactory.getLogger(ContentCache.class);
    private static final int BUFFER_CHUNK_SIZE = 0x400000;
    private final long expireAfterAccessMs;
    private final long maxTotalBytes;
    private final long maxContentLength;
    private final Cache<String, CacheEntry> cache;

    public ContentCache(long expireAfterAccessMs, long maxTotalBytes, long maxContentLength) {
        ValidationException.check(expireAfterAccessMs >= 0L, "expireAfterAccessMs is less than 0", new Object[0]);
        ValidationException.check(maxTotalBytes > 0L, "maxTotalBytes is equal or less than 0", new Object[0]);
        ValidationException.check(maxContentLength > 0L, "maxContentLength is equal or less than 0", new Object[0]);
        this.expireAfterAccessMs = expireAfterAccessMs;
        this.maxTotalBytes = maxTotalBytes;
        this.maxContentLength = maxContentLength;
        Caffeine<Object, Object> builder = Caffeine.newBuilder();
        if (expireAfterAccessMs > 0L) {
            builder = builder.expireAfterAccess(Duration.ofMillis(expireAfterAccessMs));
        }
        this.cache = builder.maximumWeight(maxTotalBytes).weigher((key, value) -> (int)Math.min(((CacheEntry)value).length, Integer.MAX_VALUE)).softValues().removalListener((location, cacheEntry, cause) -> LOG.debug("Evicted {} from ContentCache ({})", location, (Object)cause)).recordStats().build();
    }

    public long expireAfterAccess() {
        return this.expireAfterAccessMs;
    }

    public long maxContentLength() {
        return this.maxContentLength;
    }

    public long maxTotalBytes() {
        return this.maxTotalBytes;
    }

    public CacheStats stats() {
        return this.cache.stats();
    }

    public CacheEntry get(String key, Function<String, CacheEntry> mappingFunction) {
        return this.cache.get(key, mappingFunction);
    }

    public CacheEntry getIfPresent(String location) {
        return this.cache.getIfPresent(location);
    }

    public InputFile tryCache(FileIO io, String location, long length) {
        if (length <= this.maxContentLength) {
            return new CachingInputFile(this, io, location, length);
        }
        return io.newInputFile(location, length);
    }

    public void invalidate(String key) {
        this.cache.invalidate(key);
    }

    public void invalidateAll() {
        this.cache.invalidateAll();
    }

    public void cleanUp() {
        this.cache.cleanUp();
    }

    public long estimatedCacheSize() {
        return this.cache.estimatedSize();
    }

    public String toString() {
        return MoreObjects.toStringHelper(this).add("expireAfterAccessMs", this.expireAfterAccessMs).add("maxContentLength", this.maxContentLength).add("maxTotalBytes", this.maxTotalBytes).add("cacheStats", this.cache.stats()).toString();
    }

    private static class CachingInputFile
    implements InputFile {
        private final ContentCache contentCache;
        private final FileIO io;
        private final String location;
        private final long length;
        private InputFile fallbackInputFile = null;

        private CachingInputFile(ContentCache cache, FileIO io, String location, long length) {
            this.contentCache = cache;
            this.io = io;
            this.location = location;
            this.length = length;
        }

        private InputFile wrappedInputFile() {
            if (this.fallbackInputFile == null) {
                this.fallbackInputFile = this.io.newInputFile(this.location, this.length);
            }
            return this.fallbackInputFile;
        }

        @Override
        public long getLength() {
            CacheEntry buf = this.contentCache.getIfPresent(this.location);
            if (buf != null) {
                return buf.length;
            }
            if (this.fallbackInputFile != null) {
                return this.fallbackInputFile.getLength();
            }
            return this.length;
        }

        @Override
        public SeekableInputStream newStream() {
            try {
                if (this.getLength() <= this.contentCache.maxContentLength()) {
                    return this.cachedStream();
                }
                return this.wrappedInputFile().newStream();
            }
            catch (FileNotFoundException e) {
                throw new NotFoundException(e, "Failed to open input stream for file %s: %s", this.location, e.toString());
            }
            catch (IOException e) {
                throw new UncheckedIOException(String.format("Failed to open input stream for file %s: %s", this.location, e), e);
            }
        }

        @Override
        public String location() {
            return this.location;
        }

        @Override
        public boolean exists() {
            CacheEntry buf = this.contentCache.getIfPresent(this.location);
            return buf != null || this.wrappedInputFile().exists();
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        private CacheEntry cacheEntry() {
            long start = System.currentTimeMillis();
            try (SeekableInputStream stream = this.wrappedInputFile().newStream();){
                long fileLength;
                int bytesRead;
                ArrayList<ByteBuffer> buffers = Lists.newArrayList();
                for (long totalBytesToRead = fileLength = this.getLength(); totalBytesToRead > 0L; totalBytesToRead -= (long)bytesRead) {
                    int bytesToRead = (int)Math.min(0x400000L, totalBytesToRead);
                    byte[] buf = new byte[bytesToRead];
                    bytesRead = IOUtil.readRemaining(stream, buf, 0, bytesToRead);
                    if (bytesRead < bytesToRead) {
                        throw new IOException(String.format("Expected to read %d bytes, but only %d bytes read.", fileLength, fileLength - totalBytesToRead));
                    }
                    buffers.add(ByteBuffer.wrap(buf));
                }
                CacheEntry newEntry = new CacheEntry(fileLength, buffers);
                LOG.debug("cacheEntry took {} ms for {}", (Object)(System.currentTimeMillis() - start), (Object)this.location);
                CacheEntry cacheEntry = newEntry;
                return cacheEntry;
            }
            catch (IOException ex) {
                throw new UncheckedIOException(ex);
            }
        }

        private SeekableInputStream cachedStream() throws IOException {
            try {
                CacheEntry entry = this.contentCache.get(this.location, k -> this.cacheEntry());
                Preconditions.checkNotNull(entry, "CacheEntry should not be null when there is no RuntimeException occurs");
                LOG.debug("Cache stats: {}", (Object)this.contentCache.stats());
                return ByteBufferInputStream.wrap(entry.buffers);
            }
            catch (UncheckedIOException ex) {
                throw ex.getCause();
            }
            catch (RuntimeException ex) {
                throw new IOException("Caught an error while reading from cache", ex);
            }
        }
    }

    private static class CacheEntry {
        private final long length;
        private final List<ByteBuffer> buffers;

        private CacheEntry(long length, List<ByteBuffer> buffers) {
            this.length = length;
            this.buffers = buffers;
        }
    }
}

