/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.function.Function;
import org.apache.iceberg.ContentFile;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.HasTableOperations;
import org.apache.iceberg.ManifestEntry;
import org.apache.iceberg.ManifestFile;
import org.apache.iceberg.ManifestGroup;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.Table;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.TableScan;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.expressions.And;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.Literal;
import org.apache.iceberg.expressions.NamedReference;
import org.apache.iceberg.expressions.UnboundPredicate;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableSortedMap;
import org.apache.iceberg.relocated.com.google.common.collect.Iterables;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.types.Comparators;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.DateTimeUtil;
import org.apache.iceberg.util.Pair;

public class ScanSummary {
    private static final ImmutableList<String> SCAN_SUMMARY_COLUMNS = ImmutableList.of("partition", "record_count", "file_size_in_bytes");

    private ScanSummary() {
    }

    public static Builder of(TableScan scan) {
        return new Builder(scan);
    }

    static Expression joinFilters(List<Expression> expressions) {
        Expression result = Expressions.alwaysTrue();
        for (Expression expression : expressions) {
            result = Expressions.and(result, expression);
        }
        return result;
    }

    static long toMillis(long timestamp) {
        if (timestamp < 10000000000L) {
            return timestamp * 1000L;
        }
        if (timestamp < 10000000000000L) {
            return timestamp;
        }
        return timestamp / 1000L;
    }

    static Pair<Long, Long> timestampRange(List<UnboundPredicate<Long>> timeFilters) {
        long minTimestamp = Long.MIN_VALUE;
        long maxTimestamp = Long.MAX_VALUE;
        block7: for (UnboundPredicate<Long> pred : timeFilters) {
            long value = pred.literal().value();
            switch (pred.op()) {
                case LT: {
                    if (value - 1L >= maxTimestamp) continue block7;
                    maxTimestamp = value - 1L;
                    continue block7;
                }
                case LT_EQ: {
                    if (value >= maxTimestamp) continue block7;
                    maxTimestamp = value;
                    continue block7;
                }
                case GT: {
                    if (value + 1L <= minTimestamp) continue block7;
                    minTimestamp = value + 1L;
                    continue block7;
                }
                case GT_EQ: {
                    if (value <= minTimestamp) continue block7;
                    minTimestamp = value;
                    continue block7;
                }
                case EQ: {
                    if (value < maxTimestamp) {
                        maxTimestamp = value;
                    }
                    if (value <= minTimestamp) continue block7;
                    minTimestamp = value;
                    continue block7;
                }
            }
            throw new UnsupportedOperationException("Cannot filter timestamps using predicate: " + pred);
        }
        if (maxTimestamp < minTimestamp) {
            throw new IllegalArgumentException("No timestamps can match filters: " + Joiner.on(", ").join(timeFilters));
        }
        return Pair.of(minTimestamp, maxTimestamp);
    }

    private static class TopN<K, V> {
        private final int maxSize;
        private final boolean throwIfLimited;
        private final NavigableMap<K, V> map;
        private final Comparator<? super K> keyComparator;
        private K cut = null;

        TopN(int maxSize, boolean throwIfLimited, Comparator<? super K> keyComparator) {
            this.maxSize = maxSize;
            this.throwIfLimited = throwIfLimited;
            this.map = Maps.newTreeMap(keyComparator);
            this.keyComparator = keyComparator;
        }

        public void update(K key, Function<V, V> updateFunc) {
            if (this.cut != null && this.keyComparator.compare(this.cut, key) <= 0) {
                return;
            }
            this.map.put(key, updateFunc.apply(this.map.get(key)));
            while (this.map.size() > this.maxSize) {
                if (this.throwIfLimited) {
                    throw new IllegalStateException(String.format(Locale.ROOT, "Too many matching keys: more than %d", this.maxSize));
                }
                this.cut = this.map.lastKey();
                this.map.remove(this.cut);
            }
        }

        public Map<K, V> get() {
            return ImmutableSortedMap.copyOfSorted(this.map);
        }
    }

    public static class PartitionMetrics {
        private int fileCount = 0;
        private long recordCount = 0L;
        private long totalSize = 0L;
        private Long dataTimestampMillis = null;

        public int fileCount() {
            return this.fileCount;
        }

        public long recordCount() {
            return this.recordCount;
        }

        public long totalSize() {
            return this.totalSize;
        }

        public Long dataTimestampMillis() {
            return this.dataTimestampMillis;
        }

        PartitionMetrics updateFromCounts(int numFiles, long filesRecordCount, long filesSize, Long timestampMillis) {
            this.fileCount += numFiles;
            this.recordCount += filesRecordCount;
            this.totalSize += filesSize;
            if (timestampMillis != null && (this.dataTimestampMillis == null || this.dataTimestampMillis < timestampMillis)) {
                this.dataTimestampMillis = timestampMillis;
            }
            return this;
        }

        private PartitionMetrics updateFromFile(ContentFile<?> file, Long timestampMillis) {
            ++this.fileCount;
            this.recordCount += file.recordCount();
            this.totalSize += file.fileSizeInBytes();
            if (timestampMillis != null && (this.dataTimestampMillis == null || this.dataTimestampMillis < timestampMillis)) {
                this.dataTimestampMillis = timestampMillis;
            }
            return this;
        }

        public String toString() {
            String dataTimestamp = this.dataTimestampMillis != null ? DateTimeUtil.formatTimestampMillis(this.dataTimestampMillis) : null;
            return "PartitionMetrics(fileCount=" + this.fileCount + ", recordCount=" + this.recordCount + ", totalSize=" + this.totalSize + ", dataTimestamp=" + dataTimestamp + ")";
        }
    }

    public static class Builder {
        private static final Set<String> TIMESTAMP_NAMES = Sets.newHashSet("dateCreated", "lastUpdated");
        private final TableScan scan;
        private final Table table;
        private final TableOperations ops;
        private final Map<Long, Long> snapshotTimestamps;
        private int limit = Integer.MAX_VALUE;
        private boolean throwIfLimited = false;
        private final List<UnboundPredicate<Long>> timeFilters = Lists.newArrayList();

        public Builder(TableScan scan) {
            this.scan = scan;
            this.table = scan.table();
            this.ops = ((HasTableOperations)((Object)this.table)).operations();
            ImmutableMap.Builder<Long, Long> builder = ImmutableMap.builder();
            for (Snapshot snap : this.table.snapshots()) {
                builder.put(snap.snapshotId(), snap.timestampMillis());
            }
            this.snapshotTimestamps = builder.build();
        }

        private void addTimestampFilter(UnboundPredicate<Long> filter) {
            this.throwIfLimited();
            this.timeFilters.add(filter);
        }

        public Builder after(String timestamp) {
            Literal tsLiteral = Literal.of(timestamp).to(Types.TimestampType.withoutZone());
            return this.after((Long)tsLiteral.value() / 1000L);
        }

        public Builder after(long timestampMillis) {
            this.addTimestampFilter(Expressions.greaterThanOrEqual("timestamp_ms", Long.valueOf(timestampMillis)));
            return this;
        }

        public Builder before(String timestamp) {
            Literal tsLiteral = Literal.of(timestamp).to(Types.TimestampType.withoutZone());
            return this.before((Long)tsLiteral.value() / 1000L);
        }

        public Builder before(long timestampMillis) {
            this.addTimestampFilter(Expressions.lessThanOrEqual("timestamp_ms", Long.valueOf(timestampMillis)));
            return this;
        }

        public Builder throwIfLimited() {
            this.throwIfLimited = true;
            return this;
        }

        public Builder limit(int numPartitions) {
            this.limit = numPartitions;
            return this;
        }

        private void removeTimeFilters(List<Expression> expressions, Expression expression) {
            UnboundPredicate pred;
            if (expression.op() == Expression.Operation.AND) {
                And and = (And)expression;
                this.removeTimeFilters(expressions, and.left());
                this.removeTimeFilters(expressions, and.right());
                return;
            }
            if (expression instanceof UnboundPredicate && (pred = (UnboundPredicate)expression).term() instanceof NamedReference) {
                NamedReference ref = (NamedReference)pred.term();
                Literal lit = pred.literal();
                if (TIMESTAMP_NAMES.contains(ref.name())) {
                    Literal tsLiteral = lit.to(Types.TimestampType.withoutZone());
                    long millis = ScanSummary.toMillis((Long)tsLiteral.value());
                    this.addTimestampFilter(Expressions.predicate(pred.op(), "timestamp_ms", millis));
                    return;
                }
            }
            expressions.add(expression);
        }

        public Map<String, PartitionMetrics> build() {
            if (this.table.currentSnapshot() == null) {
                return ImmutableMap.of();
            }
            ArrayList<Expression> filters = Lists.newArrayList();
            this.removeTimeFilters(filters, Expressions.rewriteNot(this.scan.filter()));
            Expression rowFilter = ScanSummary.joinFilters(filters);
            Iterable<ManifestFile> manifests = this.table.currentSnapshot().dataManifests(this.ops.io());
            boolean filterByTimestamp = !this.timeFilters.isEmpty();
            HashSet<Long> snapshotsInTimeRange = Sets.newHashSet();
            if (filterByTimestamp) {
                Pair<Long, Long> range = ScanSummary.timestampRange(this.timeFilters);
                long minTimestamp = range.first();
                long maxTimestamp = range.second();
                Snapshot oldestSnapshot = this.table.currentSnapshot();
                for (Map.Entry<Long, Long> entry : this.snapshotTimestamps.entrySet()) {
                    long snapshotId = entry.getKey();
                    long timestamp = entry.getValue();
                    if (timestamp < oldestSnapshot.timestampMillis()) {
                        oldestSnapshot = this.ops.current().snapshot(snapshotId);
                    }
                    if (timestamp < minTimestamp || timestamp > maxTimestamp) continue;
                    snapshotsInTimeRange.add(snapshotId);
                }
                if (snapshotsInTimeRange.contains(oldestSnapshot.snapshotId()) && minTimestamp < oldestSnapshot.timestampMillis()) {
                    throw new IllegalArgumentException("Cannot satisfy time filters: time range may include expired snapshots");
                }
                manifests = Iterables.filter(manifests, manifest -> {
                    if (manifest.snapshotId() == null) {
                        return true;
                    }
                    Long timestamp = this.snapshotTimestamps.get(manifest.snapshotId());
                    return timestamp != null && timestamp >= minTimestamp;
                });
            }
            return this.computeTopPartitionMetrics(rowFilter, manifests, filterByTimestamp, snapshotsInTimeRange);
        }

        private Map<String, PartitionMetrics> computeTopPartitionMetrics(Expression rowFilter, Iterable<ManifestFile> manifests, boolean filterByTimestamp, Set<Long> snapshotsInTimeRange) {
            TopN<CharSequence, PartitionMetrics> topN = new TopN<CharSequence, PartitionMetrics>(this.limit, this.throwIfLimited, Comparators.charSequences());
            try (CloseableIterable<ManifestEntry<DataFile>> entries = new ManifestGroup(this.ops.io(), manifests).specsById(this.ops.current().specsById()).filterData(rowFilter).ignoreDeleted().select(SCAN_SUMMARY_COLUMNS).entries();){
                PartitionSpec spec = this.table.spec();
                for (ManifestEntry manifestEntry : entries) {
                    Long timestamp = this.snapshotTimestamps.get(manifestEntry.snapshotId());
                    if (filterByTimestamp && !snapshotsInTimeRange.contains(manifestEntry.snapshotId())) continue;
                    String partition = spec.partitionToPath(manifestEntry.file().partition());
                    topN.update(partition, metrics -> (metrics == null ? new PartitionMetrics() : metrics).updateFromFile((ContentFile<?>)entry.file(), timestamp));
                }
            }
            catch (IOException e) {
                throw new RuntimeIOException(e);
            }
            return topN.get();
        }
    }
}

