/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.spark.extensions;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.iceberg.AssertHelpers;
import org.apache.iceberg.SnapshotRef;
import org.apache.iceberg.Table;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.spark.SparkCatalogConfig;
import org.apache.iceberg.spark.extensions.SparkExtensionsTestBase;
import org.apache.iceberg.spark.source.SimpleRecord;
import org.apache.spark.sql.Dataset;
import org.apache.spark.sql.catalyst.analysis.NoSuchTableException;
import org.apache.spark.sql.catalyst.parser.extensions.IcebergParseException;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runners.Parameterized;

public class TestBranchDDL
extends SparkExtensionsTestBase {
    private static final String[] TIME_UNITS = new String[]{"DAYS", "HOURS", "MINUTES"};

    @Before
    public void before() {
        this.sql("CREATE TABLE %s (id INT, data STRING) USING iceberg", new Object[]{this.tableName});
    }

    @After
    public void removeTable() {
        this.sql("DROP TABLE IF EXISTS %s", new Object[]{this.tableName});
    }

    @Parameterized.Parameters(name="catalogName = {0}, implementation = {1}, config = {2}")
    public static Object[][] parameters() {
        return new Object[][]{{SparkCatalogConfig.SPARK.catalogName(), SparkCatalogConfig.SPARK.implementation(), SparkCatalogConfig.SPARK.properties()}};
    }

    public TestBranchDDL(String catalog, String implementation, Map<String, String> properties) {
        super(catalog, implementation, properties);
    }

    @Test
    public void testCreateBranch() throws NoSuchTableException {
        Table table = this.insertRows();
        long snapshotId = table.currentSnapshot().snapshotId();
        Integer minSnapshotsToKeep = 2;
        long maxSnapshotAge = 2L;
        long maxRefAge = 10L;
        for (String timeUnit : TIME_UNITS) {
            String branchName = "b1" + timeUnit;
            this.sql("ALTER TABLE %s CREATE BRANCH %s AS OF VERSION %d RETAIN %d %s WITH SNAPSHOT RETENTION %d SNAPSHOTS %d %s", new Object[]{this.tableName, branchName, snapshotId, maxRefAge, timeUnit, minSnapshotsToKeep, maxSnapshotAge, timeUnit});
            table.refresh();
            SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
            Assert.assertEquals((long)table.currentSnapshot().snapshotId(), (long)ref.snapshotId());
            Assert.assertEquals((Object)minSnapshotsToKeep, (Object)ref.minSnapshotsToKeep());
            Assert.assertEquals((long)TimeUnit.valueOf(timeUnit).toMillis(maxSnapshotAge), (long)ref.maxSnapshotAgeMs());
            Assert.assertEquals((long)TimeUnit.valueOf(timeUnit).toMillis(maxRefAge), (long)ref.maxRefAgeMs());
        }
    }

    @Test
    public void testCreateBranchOnEmptyTable() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.sql("ALTER TABLE %s CREATE BRANCH %s", new Object[]{this.tableName, "b1"})).isInstanceOf(IllegalArgumentException.class)).hasMessageContaining("Cannot complete create or replace branch operation on %s, main has no snapshot", new Object[]{this.tableName});
    }

    @Test
    public void testCreateBranchUseDefaultConfig() throws NoSuchTableException {
        Table table = this.insertRows();
        String branchName = "b1";
        this.sql("ALTER TABLE %s CREATE BRANCH %s", new Object[]{this.tableName, branchName});
        table.refresh();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertEquals((long)table.currentSnapshot().snapshotId(), (long)ref.snapshotId());
        Assert.assertNull((Object)ref.minSnapshotsToKeep());
        Assert.assertNull((Object)ref.maxSnapshotAgeMs());
        Assert.assertNull((Object)ref.maxRefAgeMs());
    }

    @Test
    public void testCreateBranchUseCustomMinSnapshotsToKeep() throws NoSuchTableException {
        Integer minSnapshotsToKeep = 2;
        Table table = this.insertRows();
        String branchName = "b1";
        this.sql("ALTER TABLE %s CREATE BRANCH %s WITH SNAPSHOT RETENTION %d SNAPSHOTS", new Object[]{this.tableName, branchName, minSnapshotsToKeep});
        table.refresh();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertEquals((long)table.currentSnapshot().snapshotId(), (long)ref.snapshotId());
        Assert.assertEquals((Object)minSnapshotsToKeep, (Object)ref.minSnapshotsToKeep());
        Assert.assertNull((Object)ref.maxSnapshotAgeMs());
        Assert.assertNull((Object)ref.maxRefAgeMs());
    }

    @Test
    public void testCreateBranchUseCustomMaxSnapshotAge() throws NoSuchTableException {
        long maxSnapshotAge = 2L;
        Table table = this.insertRows();
        String branchName = "b1";
        this.sql("ALTER TABLE %s CREATE BRANCH %s WITH SNAPSHOT RETENTION %d DAYS", new Object[]{this.tableName, branchName, maxSnapshotAge});
        table.refresh();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertEquals((long)table.currentSnapshot().snapshotId(), (long)ref.snapshotId());
        Assert.assertNull((Object)ref.minSnapshotsToKeep());
        Assert.assertEquals((long)TimeUnit.DAYS.toMillis(maxSnapshotAge), (long)ref.maxSnapshotAgeMs());
        Assert.assertNull((Object)ref.maxRefAgeMs());
    }

    @Test
    public void testCreateBranchIfNotExists() throws NoSuchTableException {
        long maxSnapshotAge = 2L;
        Table table = this.insertRows();
        String branchName = "b1";
        this.sql("ALTER TABLE %s CREATE BRANCH %s WITH SNAPSHOT RETENTION %d DAYS", new Object[]{this.tableName, branchName, maxSnapshotAge});
        AssertHelpers.assertThrows((String)"Cannot create an existing branch", IllegalArgumentException.class, (String)"Ref b1 already exists", () -> this.sql("ALTER TABLE %s CREATE BRANCH %s", new Object[]{this.tableName, branchName}));
        this.sql("ALTER TABLE %s CREATE BRANCH IF NOT EXISTS %s", new Object[]{this.tableName, branchName});
        table.refresh();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertEquals((long)table.currentSnapshot().snapshotId(), (long)ref.snapshotId());
        Assert.assertNull((Object)ref.minSnapshotsToKeep());
        Assert.assertEquals((long)TimeUnit.DAYS.toMillis(maxSnapshotAge), (long)ref.maxSnapshotAgeMs());
        Assert.assertNull((Object)ref.maxRefAgeMs());
    }

    @Test
    public void testCreateBranchUseCustomMinSnapshotsToKeepAndMaxSnapshotAge() throws NoSuchTableException {
        Integer minSnapshotsToKeep = 2;
        long maxSnapshotAge = 2L;
        Table table = this.insertRows();
        String branchName = "b1";
        this.sql("ALTER TABLE %s CREATE BRANCH %s WITH SNAPSHOT RETENTION %d SNAPSHOTS %d DAYS", new Object[]{this.tableName, branchName, minSnapshotsToKeep, maxSnapshotAge});
        table.refresh();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertEquals((long)table.currentSnapshot().snapshotId(), (long)ref.snapshotId());
        Assert.assertEquals((Object)minSnapshotsToKeep, (Object)ref.minSnapshotsToKeep());
        Assert.assertEquals((long)TimeUnit.DAYS.toMillis(maxSnapshotAge), (long)ref.maxSnapshotAgeMs());
        Assert.assertNull((Object)ref.maxRefAgeMs());
        AssertHelpers.assertThrows((String)"Illegal statement", IcebergParseException.class, (String)"no viable alternative at input 'WITH SNAPSHOT RETENTION'", () -> this.sql("ALTER TABLE %s CREATE BRANCH %s WITH SNAPSHOT RETENTION", new Object[]{this.tableName, branchName}));
    }

    @Test
    public void testCreateBranchUseCustomMaxRefAge() throws NoSuchTableException {
        long maxRefAge = 10L;
        Table table = this.insertRows();
        String branchName = "b1";
        this.sql("ALTER TABLE %s CREATE BRANCH %s RETAIN %d DAYS", new Object[]{this.tableName, branchName, maxRefAge});
        table.refresh();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertEquals((long)table.currentSnapshot().snapshotId(), (long)ref.snapshotId());
        Assert.assertNull((Object)ref.minSnapshotsToKeep());
        Assert.assertNull((Object)ref.maxSnapshotAgeMs());
        Assert.assertEquals((long)TimeUnit.DAYS.toMillis(maxRefAge), (long)ref.maxRefAgeMs());
        AssertHelpers.assertThrows((String)"Illegal statement", IcebergParseException.class, (String)"mismatched input", () -> this.sql("ALTER TABLE %s CREATE BRANCH %s RETAIN", new Object[]{this.tableName, branchName}));
        AssertHelpers.assertThrows((String)"Illegal statement", IcebergParseException.class, (String)"mismatched input", () -> this.sql("ALTER TABLE %s CREATE BRANCH %s RETAIN %s DAYS", new Object[]{this.tableName, branchName, "abc"}));
        AssertHelpers.assertThrows((String)"Illegal statement", IcebergParseException.class, (String)"mismatched input 'SECONDS' expecting {'DAYS', 'HOURS', 'MINUTES'}", () -> this.sql("ALTER TABLE %s CREATE BRANCH %s RETAIN %d SECONDS", new Object[]{this.tableName, branchName, maxRefAge}));
    }

    @Test
    public void testDropBranch() throws NoSuchTableException {
        this.insertRows();
        Table table = this.validationCatalog.loadTable(this.tableIdent);
        String branchName = "b1";
        table.manageSnapshots().createBranch(branchName, table.currentSnapshot().snapshotId()).commit();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertEquals((long)table.currentSnapshot().snapshotId(), (long)ref.snapshotId());
        this.sql("ALTER TABLE %s DROP BRANCH %s", new Object[]{this.tableName, branchName});
        table.refresh();
        ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertNull((Object)ref);
    }

    @Test
    public void testDropBranchDoesNotExist() {
        AssertHelpers.assertThrows((String)"Cannot perform drop branch on branch which does not exist", IllegalArgumentException.class, (String)"Branch does not exist: nonExistingBranch", () -> this.sql("ALTER TABLE %s DROP BRANCH %s", new Object[]{this.tableName, "nonExistingBranch"}));
    }

    @Test
    public void testDropBranchFailsForTag() throws NoSuchTableException {
        String tagName = "b1";
        Table table = this.insertRows();
        table.manageSnapshots().createTag(tagName, table.currentSnapshot().snapshotId()).commit();
        AssertHelpers.assertThrows((String)"Cannot perform drop branch on tag", IllegalArgumentException.class, (String)"Ref b1 is a tag not a branch", () -> this.sql("ALTER TABLE %s DROP BRANCH %s", new Object[]{this.tableName, tagName}));
    }

    @Test
    public void testDropBranchNonConformingName() {
        AssertHelpers.assertThrows((String)"Non-conforming branch name", IcebergParseException.class, (String)"mismatched input '123'", () -> this.sql("ALTER TABLE %s DROP BRANCH %s", new Object[]{this.tableName, "123"}));
    }

    @Test
    public void testDropMainBranchFails() {
        AssertHelpers.assertThrows((String)"Cannot drop the main branch", IllegalArgumentException.class, (String)"Cannot remove main branch", () -> this.sql("ALTER TABLE %s DROP BRANCH main", new Object[]{this.tableName}));
    }

    @Test
    public void testDropBranchIfExists() {
        String branchName = "nonExistingBranch";
        Table table = this.validationCatalog.loadTable(this.tableIdent);
        Assert.assertNull(table.refs().get(branchName));
        this.sql("ALTER TABLE %s DROP BRANCH IF EXISTS %s", new Object[]{this.tableName, branchName});
        table.refresh();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertNull((Object)ref);
    }

    @Test
    public void testReplaceBranchFailsForTag() throws NoSuchTableException {
        String tagName = "tag1";
        ImmutableList records = ImmutableList.of((Object)new SimpleRecord(Integer.valueOf(1), "a"), (Object)new SimpleRecord(Integer.valueOf(2), "b"));
        Dataset df = spark.createDataFrame((List)records, SimpleRecord.class);
        df.writeTo(this.tableName).append();
        Table table = this.validationCatalog.loadTable(this.tableIdent);
        long first = table.currentSnapshot().snapshotId();
        table.manageSnapshots().createTag(tagName, first).commit();
        df.writeTo(this.tableName).append();
        long second = table.currentSnapshot().snapshotId();
        AssertHelpers.assertThrows((String)"Cannot perform replace branch on tags", IllegalArgumentException.class, (String)"Ref tag1 is a tag not a branch", () -> this.sql("ALTER TABLE %s REPLACE BRANCH %s AS OF VERSION %d", new Object[]{this.tableName, tagName, second}));
    }

    @Test
    public void testReplaceBranch() throws NoSuchTableException {
        ImmutableList records = ImmutableList.of((Object)new SimpleRecord(Integer.valueOf(1), "a"), (Object)new SimpleRecord(Integer.valueOf(2), "b"));
        Dataset df = spark.createDataFrame((List)records, SimpleRecord.class);
        df.writeTo(this.tableName).append();
        Table table = this.validationCatalog.loadTable(this.tableIdent);
        long first = table.currentSnapshot().snapshotId();
        String branchName = "b1";
        long expectedMaxRefAgeMs = 1000L;
        int expectedMinSnapshotsToKeep = 2;
        long expectedMaxSnapshotAgeMs = 1000L;
        table.manageSnapshots().createBranch(branchName, first).setMaxRefAgeMs(branchName, expectedMaxRefAgeMs).setMinSnapshotsToKeep(branchName, expectedMinSnapshotsToKeep).setMaxSnapshotAgeMs(branchName, expectedMaxSnapshotAgeMs).commit();
        df.writeTo(this.tableName).append();
        long second = table.currentSnapshot().snapshotId();
        this.sql("ALTER TABLE %s REPLACE BRANCH %s AS OF VERSION %d", new Object[]{this.tableName, branchName, second});
        table.refresh();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertNotNull((Object)ref);
        Assert.assertEquals((long)second, (long)ref.snapshotId());
        Assert.assertEquals((long)expectedMinSnapshotsToKeep, (long)ref.minSnapshotsToKeep().intValue());
        Assert.assertEquals((long)expectedMaxSnapshotAgeMs, (long)ref.maxSnapshotAgeMs());
        Assert.assertEquals((long)expectedMaxRefAgeMs, (long)ref.maxRefAgeMs());
    }

    @Test
    public void testReplaceBranchDoesNotExist() throws NoSuchTableException {
        ImmutableList records = ImmutableList.of((Object)new SimpleRecord(Integer.valueOf(1), "a"), (Object)new SimpleRecord(Integer.valueOf(2), "b"));
        Dataset df = spark.createDataFrame((List)records, SimpleRecord.class);
        df.writeTo(this.tableName).append();
        Table table = this.validationCatalog.loadTable(this.tableIdent);
        AssertHelpers.assertThrows((String)"Cannot perform replace branch on branch which does not exist", IllegalArgumentException.class, (String)"Branch does not exist", () -> this.sql("ALTER TABLE %s REPLACE BRANCH %s AS OF VERSION %d", new Object[]{this.tableName, "someBranch", table.currentSnapshot().snapshotId()}));
    }

    @Test
    public void testReplaceBranchWithRetain() throws NoSuchTableException {
        ImmutableList records = ImmutableList.of((Object)new SimpleRecord(Integer.valueOf(1), "a"), (Object)new SimpleRecord(Integer.valueOf(2), "b"));
        Dataset df = spark.createDataFrame((List)records, SimpleRecord.class);
        df.writeTo(this.tableName).append();
        Table table = this.validationCatalog.loadTable(this.tableIdent);
        long first = table.currentSnapshot().snapshotId();
        String branchName = "b1";
        table.manageSnapshots().createBranch(branchName, first).commit();
        SnapshotRef b1 = (SnapshotRef)table.refs().get(branchName);
        Integer minSnapshotsToKeep = b1.minSnapshotsToKeep();
        Long maxSnapshotAgeMs = b1.maxSnapshotAgeMs();
        df.writeTo(this.tableName).append();
        long second = table.currentSnapshot().snapshotId();
        long maxRefAge = 10L;
        for (String timeUnit : TIME_UNITS) {
            this.sql("ALTER TABLE %s REPLACE BRANCH %s AS OF VERSION %d RETAIN %d %s", new Object[]{this.tableName, branchName, second, maxRefAge, timeUnit});
            table.refresh();
            SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
            Assert.assertNotNull((Object)ref);
            Assert.assertEquals((long)second, (long)ref.snapshotId());
            Assert.assertEquals((Object)minSnapshotsToKeep, (Object)ref.minSnapshotsToKeep());
            Assert.assertEquals((Object)maxSnapshotAgeMs, (Object)ref.maxSnapshotAgeMs());
            Assert.assertEquals((long)TimeUnit.valueOf(timeUnit).toMillis(maxRefAge), (long)ref.maxRefAgeMs());
        }
    }

    @Test
    public void testReplaceBranchWithSnapshotRetention() throws NoSuchTableException {
        ImmutableList records = ImmutableList.of((Object)new SimpleRecord(Integer.valueOf(1), "a"), (Object)new SimpleRecord(Integer.valueOf(2), "b"));
        Dataset df = spark.createDataFrame((List)records, SimpleRecord.class);
        df.writeTo(this.tableName).append();
        String branchName = "b1";
        Table table = this.validationCatalog.loadTable(this.tableIdent);
        long first = table.currentSnapshot().snapshotId();
        table.manageSnapshots().createBranch(branchName, first).commit();
        df.writeTo(this.tableName).append();
        long second = table.currentSnapshot().snapshotId();
        Integer minSnapshotsToKeep = 2;
        long maxSnapshotAge = 2L;
        Long maxRefAgeMs = ((SnapshotRef)table.refs().get(branchName)).maxRefAgeMs();
        for (String timeUnit : TIME_UNITS) {
            this.sql("ALTER TABLE %s REPLACE BRANCH %s AS OF VERSION %d WITH SNAPSHOT RETENTION %d SNAPSHOTS %d %s", new Object[]{this.tableName, branchName, second, minSnapshotsToKeep, maxSnapshotAge, timeUnit});
            table.refresh();
            SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
            Assert.assertNotNull((Object)ref);
            Assert.assertEquals((long)second, (long)ref.snapshotId());
            Assert.assertEquals((Object)minSnapshotsToKeep, (Object)ref.minSnapshotsToKeep());
            Assert.assertEquals((long)TimeUnit.valueOf(timeUnit).toMillis(maxSnapshotAge), (long)ref.maxSnapshotAgeMs());
            Assert.assertEquals((Object)maxRefAgeMs, (Object)ref.maxRefAgeMs());
        }
    }

    @Test
    public void testReplaceBranchWithRetainAndSnapshotRetention() throws NoSuchTableException {
        ImmutableList records = ImmutableList.of((Object)new SimpleRecord(Integer.valueOf(1), "a"), (Object)new SimpleRecord(Integer.valueOf(2), "b"));
        Dataset df = spark.createDataFrame((List)records, SimpleRecord.class);
        df.writeTo(this.tableName).append();
        Table table = this.validationCatalog.loadTable(this.tableIdent);
        long first = table.currentSnapshot().snapshotId();
        String branchName = "b1";
        table.manageSnapshots().createBranch(branchName, first).commit();
        df.writeTo(this.tableName).append();
        long second = table.currentSnapshot().snapshotId();
        Integer minSnapshotsToKeep = 2;
        long maxSnapshotAge = 2L;
        long maxRefAge = 10L;
        for (String timeUnit : TIME_UNITS) {
            this.sql("ALTER TABLE %s REPLACE BRANCH %s AS OF VERSION %d RETAIN %d %s WITH SNAPSHOT RETENTION %d SNAPSHOTS %d %s", new Object[]{this.tableName, branchName, second, maxRefAge, timeUnit, minSnapshotsToKeep, maxSnapshotAge, timeUnit});
            table.refresh();
            SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
            Assert.assertNotNull((Object)ref);
            Assert.assertEquals((long)second, (long)ref.snapshotId());
            Assert.assertEquals((Object)minSnapshotsToKeep, (Object)ref.minSnapshotsToKeep());
            Assert.assertEquals((long)TimeUnit.valueOf(timeUnit).toMillis(maxSnapshotAge), (long)ref.maxSnapshotAgeMs());
            Assert.assertEquals((long)TimeUnit.valueOf(timeUnit).toMillis(maxRefAge), (long)ref.maxRefAgeMs());
        }
    }

    @Test
    public void testCreateOrReplace() throws NoSuchTableException {
        ImmutableList records = ImmutableList.of((Object)new SimpleRecord(Integer.valueOf(1), "a"), (Object)new SimpleRecord(Integer.valueOf(2), "b"));
        Dataset df = spark.createDataFrame((List)records, SimpleRecord.class);
        df.writeTo(this.tableName).append();
        Table table = this.validationCatalog.loadTable(this.tableIdent);
        long first = table.currentSnapshot().snapshotId();
        String branchName = "b1";
        df.writeTo(this.tableName).append();
        long second = table.currentSnapshot().snapshotId();
        table.manageSnapshots().createBranch(branchName, second).commit();
        this.sql("ALTER TABLE %s CREATE OR REPLACE BRANCH %s AS OF VERSION %d", new Object[]{this.tableName, branchName, first});
        table.refresh();
        SnapshotRef ref = (SnapshotRef)table.refs().get(branchName);
        Assert.assertNotNull((Object)ref);
        Assert.assertEquals((long)first, (long)ref.snapshotId());
    }

    private Table insertRows() throws NoSuchTableException {
        ImmutableList records = ImmutableList.of((Object)new SimpleRecord(Integer.valueOf(1), "a"), (Object)new SimpleRecord(Integer.valueOf(2), "b"));
        Dataset df = spark.createDataFrame((List)records, SimpleRecord.class);
        df.writeTo(this.tableName).append();
        return this.validationCatalog.loadTable(this.tableIdent);
    }
}

