/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.spark.source;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import org.apache.iceberg.AppendFiles;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.DataFiles;
import org.apache.iceberg.FileFormat;
import org.apache.iceberg.Metrics;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.ReplacePartitions;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SnapshotUpdate;
import org.apache.iceberg.Table;
import org.apache.iceberg.avro.Avro;
import org.apache.iceberg.encryption.EncryptedOutputFile;
import org.apache.iceberg.encryption.EncryptionManager;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.io.FileAppender;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.io.LocationProvider;
import org.apache.iceberg.io.OutputFile;
import org.apache.iceberg.parquet.Parquet;
import org.apache.iceberg.shaded.com.google.common.collect.ImmutableList;
import org.apache.iceberg.shaded.com.google.common.collect.Iterables;
import org.apache.iceberg.shaded.com.google.common.collect.Lists;
import org.apache.iceberg.shaded.com.google.common.collect.Sets;
import org.apache.iceberg.spark.data.SparkAvroWriter;
import org.apache.iceberg.spark.data.SparkParquetWriters;
import org.apache.iceberg.spark.source.PartitionKey;
import org.apache.iceberg.util.PropertyUtil;
import org.apache.iceberg.util.Tasks;
import org.apache.spark.sql.catalyst.InternalRow;
import org.apache.spark.sql.sources.v2.DataSourceOptions;
import org.apache.spark.sql.sources.v2.writer.DataSourceWriter;
import org.apache.spark.sql.sources.v2.writer.DataWriter;
import org.apache.spark.sql.sources.v2.writer.DataWriterFactory;
import org.apache.spark.sql.sources.v2.writer.WriterCommitMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Writer
implements DataSourceWriter {
    private static final Logger LOG = LoggerFactory.getLogger(Writer.class);
    private final Table table;
    private final FileFormat format;
    private final FileIO fileIo;
    private final EncryptionManager encryptionManager;
    private final boolean replacePartitions;
    private final String applicationId;
    private final String wapId;
    private final long targetFileSize;

    Writer(Table table, DataSourceOptions options, boolean replacePartitions, String applicationId) {
        this(table, options, replacePartitions, applicationId, null);
    }

    Writer(Table table, DataSourceOptions options, boolean replacePartitions, String applicationId, String wapId) {
        this.table = table;
        this.format = this.getFileFormat(table.properties(), options);
        this.fileIo = table.io();
        this.encryptionManager = table.encryption();
        this.replacePartitions = replacePartitions;
        this.applicationId = applicationId;
        this.wapId = wapId;
        long tableTargetFileSize = PropertyUtil.propertyAsLong(table.properties(), "write.target-file-size-bytes", Long.MAX_VALUE);
        this.targetFileSize = options.getLong("target-file-size-bytes", tableTargetFileSize);
    }

    private FileFormat getFileFormat(Map<String, String> tableProperties, DataSourceOptions options) {
        Optional formatOption = options.get("write-format");
        String formatString = formatOption.orElse(tableProperties.getOrDefault("write.format.default", "parquet"));
        return FileFormat.valueOf(formatString.toUpperCase(Locale.ENGLISH));
    }

    private boolean isWapTable() {
        return Boolean.parseBoolean(this.table.properties().getOrDefault("write.wap.enabled", "false"));
    }

    public DataWriterFactory<InternalRow> createWriterFactory() {
        return new WriterFactory(this.table.spec(), this.format, this.table.locationProvider(), this.table.properties(), this.fileIo, this.encryptionManager, this.targetFileSize);
    }

    public void commit(WriterCommitMessage[] messages) {
        if (this.replacePartitions) {
            this.replacePartitions(messages);
        } else {
            this.append(messages);
        }
    }

    protected void commitOperation(SnapshotUpdate<?> operation, int numFiles, String description) {
        LOG.info("Committing {} with {} files to table {}", new Object[]{description, numFiles, this.table});
        if (this.applicationId != null) {
            operation.set("spark.app.id", this.applicationId);
        }
        if (this.isWapTable() && this.wapId != null) {
            operation.set("wap.id", this.wapId);
            operation.stageOnly();
        }
        long start = System.currentTimeMillis();
        operation.commit();
        long duration = System.currentTimeMillis() - start;
        LOG.info("Committed in {} ms", (Object)duration);
    }

    private void append(WriterCommitMessage[] messages) {
        AppendFiles append = this.table.newAppend();
        int numFiles = 0;
        for (DataFile file : this.files(messages)) {
            ++numFiles;
            append.appendFile(file);
        }
        this.commitOperation(append, numFiles, "append");
    }

    private void replacePartitions(WriterCommitMessage[] messages) {
        ReplacePartitions dynamicOverwrite = this.table.newReplacePartitions();
        int numFiles = 0;
        for (DataFile file : this.files(messages)) {
            ++numFiles;
            dynamicOverwrite.addFile(file);
        }
        this.commitOperation(dynamicOverwrite, numFiles, "dynamic partition overwrite");
    }

    public void abort(WriterCommitMessage[] messages) {
        Tasks.foreach(this.files(messages)).retry(this.propertyAsInt("commit.retry.num-retries", 4)).exponentialBackoff(this.propertyAsInt("commit.retry.min-wait-ms", 100), this.propertyAsInt("commit.retry.max-wait-ms", 60000), this.propertyAsInt("commit.retry.total-timeout-ms", 1800000), 2.0).throwFailureWhenFinished().run(file -> this.fileIo.deleteFile(file.path().toString()));
    }

    protected Table table() {
        return this.table;
    }

    protected Iterable<DataFile> files(WriterCommitMessage[] messages) {
        if (messages.length > 0) {
            return Iterables.concat(Iterables.transform(Arrays.asList(messages), message -> message != null ? ImmutableList.copyOf(((TaskCommit)message).files()) : ImmutableList.of()));
        }
        return ImmutableList.of();
    }

    private int propertyAsInt(String property, int defaultValue) {
        Map<String, String> properties = this.table.properties();
        String value = properties.get(property);
        if (value != null) {
            return Integer.parseInt(properties.get(property));
        }
        return defaultValue;
    }

    public String toString() {
        return String.format("IcebergWrite(table=%s, format=%s)", new Object[]{this.table, this.format});
    }

    private static class PartitionedWriter
    extends BaseWriter {
        private final PartitionKey key;
        private final Set<PartitionKey> completedPartitions = Sets.newHashSet();

        PartitionedWriter(PartitionSpec spec, FileFormat format, AppenderFactory<InternalRow> appenderFactory, WriterFactory.OutputFileFactory fileFactory, FileIO fileIo, long targetFileSize) {
            super(spec, format, appenderFactory, fileFactory, fileIo, targetFileSize);
            this.key = new PartitionKey(spec);
        }

        @Override
        public void write(InternalRow row) throws IOException {
            this.key.partition(row);
            PartitionKey currentKey = this.getCurrentKey();
            if (!this.key.equals(currentKey)) {
                this.closeCurrent();
                this.completedPartitions.add(currentKey);
                if (this.completedPartitions.contains(this.key)) {
                    PartitionKey existingKey = Iterables.find(this.completedPartitions, this.key::equals, null);
                    LOG.warn("Duplicate key: {} == {}", (Object)existingKey, (Object)this.key);
                    throw new IllegalStateException("Already closed files for partition: " + this.key.toPath());
                }
                this.setCurrentKey(this.key.copy());
                this.openCurrent();
            }
            this.writeInternal(row);
        }
    }

    private static class UnpartitionedWriter
    extends BaseWriter {
        private static final int ROWS_DIVISOR = 1000;

        UnpartitionedWriter(PartitionSpec spec, FileFormat format, AppenderFactory<InternalRow> appenderFactory, WriterFactory.OutputFileFactory fileFactory, FileIO fileIo, long targetFileSize) {
            super(spec, format, appenderFactory, fileFactory, fileIo, targetFileSize);
            this.openCurrent();
        }

        @Override
        public void write(InternalRow row) throws IOException {
            this.writeInternal(row);
        }
    }

    private static abstract class BaseWriter
    implements DataWriter<InternalRow> {
        protected static final int ROWS_DIVISOR = 1000;
        private final List<DataFile> completedFiles = Lists.newArrayList();
        private final PartitionSpec spec;
        private final FileFormat format;
        private final AppenderFactory<InternalRow> appenderFactory;
        private final WriterFactory.OutputFileFactory fileFactory;
        private final FileIO fileIo;
        private final long targetFileSize;
        private PartitionKey currentKey = null;
        private FileAppender<InternalRow> currentAppender = null;
        private EncryptedOutputFile currentFile = null;
        private long currentRows = 0L;

        BaseWriter(PartitionSpec spec, FileFormat format, AppenderFactory<InternalRow> appenderFactory, WriterFactory.OutputFileFactory fileFactory, FileIO fileIo, long targetFileSize) {
            this.spec = spec;
            this.format = format;
            this.appenderFactory = appenderFactory;
            this.fileFactory = fileFactory;
            this.fileIo = fileIo;
            this.targetFileSize = targetFileSize;
        }

        public abstract void write(InternalRow var1) throws IOException;

        public void writeInternal(InternalRow row) throws IOException {
            if (this.currentRows % 1000L == 0L && this.currentAppender.length() >= this.targetFileSize) {
                this.closeCurrent();
                this.openCurrent();
            }
            this.currentAppender.add(row);
            ++this.currentRows;
        }

        public WriterCommitMessage commit() throws IOException {
            this.closeCurrent();
            return new TaskCommit(this.completedFiles);
        }

        public void abort() throws IOException {
            this.closeCurrent();
            Tasks.foreach(this.completedFiles).throwFailureWhenFinished().noRetry().run(file -> this.fileIo.deleteFile(file.path().toString()));
        }

        protected void openCurrent() {
            this.currentFile = this.spec.fields().size() == 0 ? this.fileFactory.newOutputFile() : this.fileFactory.newOutputFile(this.currentKey);
            this.currentAppender = this.appenderFactory.newAppender(this.currentFile.encryptingOutputFile(), this.format);
            this.currentRows = 0L;
        }

        protected void closeCurrent() throws IOException {
            if (this.currentAppender != null) {
                this.currentAppender.close();
                Metrics metrics = this.currentAppender.metrics();
                List<Long> splitOffsets = this.currentAppender.splitOffsets();
                this.currentAppender = null;
                if (metrics.recordCount() == 0L) {
                    this.fileIo.deleteFile(this.currentFile.encryptingOutputFile());
                } else {
                    DataFile dataFile = DataFiles.builder(this.spec).withEncryptedOutputFile(this.currentFile).withPartition(this.spec.fields().size() == 0 ? null : this.currentKey).withMetrics(metrics).withSplitOffsets(splitOffsets).build();
                    this.completedFiles.add(dataFile);
                }
                this.currentFile = null;
            }
        }

        protected PartitionKey getCurrentKey() {
            return this.currentKey;
        }

        protected void setCurrentKey(PartitionKey currentKey) {
            this.currentKey = currentKey;
        }
    }

    private static interface AppenderFactory<T> {
        public FileAppender<T> newAppender(OutputFile var1, FileFormat var2);
    }

    private static class WriterFactory
    implements DataWriterFactory<InternalRow> {
        private final PartitionSpec spec;
        private final FileFormat format;
        private final LocationProvider locations;
        private final Map<String, String> properties;
        private final FileIO fileIo;
        private final EncryptionManager encryptionManager;
        private final long targetFileSize;

        WriterFactory(PartitionSpec spec, FileFormat format, LocationProvider locations, Map<String, String> properties, FileIO fileIo, EncryptionManager encryptionManager, long targetFileSize) {
            this.spec = spec;
            this.format = format;
            this.locations = locations;
            this.properties = properties;
            this.fileIo = fileIo;
            this.encryptionManager = encryptionManager;
            this.targetFileSize = targetFileSize;
        }

        public DataWriter<InternalRow> createDataWriter(int partitionId, long taskId, long epochId) {
            OutputFileFactory fileFactory = new OutputFileFactory(partitionId, taskId, epochId);
            SparkAppenderFactory appenderFactory = new SparkAppenderFactory();
            if (this.spec.fields().isEmpty()) {
                return new UnpartitionedWriter(this.spec, this.format, appenderFactory, fileFactory, this.fileIo, this.targetFileSize);
            }
            return new PartitionedWriter(this.spec, this.format, appenderFactory, fileFactory, this.fileIo, this.targetFileSize);
        }

        private class OutputFileFactory {
            private final int partitionId;
            private final long taskId;
            private final long epochId;
            private final String uuid = UUID.randomUUID().toString();
            private int fileCount;

            OutputFileFactory(int partitionId, long taskId, long epochId) {
                this.partitionId = partitionId;
                this.taskId = taskId;
                this.epochId = epochId;
                this.fileCount = 0;
            }

            private String generateFilename() {
                return WriterFactory.this.format.addExtension(String.format("%05d-%d-%s-%05d", this.partitionId, this.taskId, this.uuid, this.fileCount++));
            }

            public EncryptedOutputFile newOutputFile() {
                OutputFile file = WriterFactory.this.fileIo.newOutputFile(WriterFactory.this.locations.newDataLocation(this.generateFilename()));
                return WriterFactory.this.encryptionManager.encrypt(file);
            }

            public EncryptedOutputFile newOutputFile(PartitionKey key) {
                OutputFile rawOutputFile = WriterFactory.this.fileIo.newOutputFile(WriterFactory.this.locations.newDataLocation(WriterFactory.this.spec, key, this.generateFilename()));
                return WriterFactory.this.encryptionManager.encrypt(rawOutputFile);
            }
        }

        private class SparkAppenderFactory
        implements AppenderFactory<InternalRow> {
            private SparkAppenderFactory() {
            }

            @Override
            public FileAppender<InternalRow> newAppender(OutputFile file, FileFormat fileFormat) {
                Schema schema = WriterFactory.this.spec.schema();
                MetricsConfig metricsConfig = MetricsConfig.fromProperties(WriterFactory.this.properties);
                try {
                    switch (fileFormat) {
                        case PARQUET: {
                            return Parquet.write(file).createWriterFunc(msgType -> SparkParquetWriters.buildWriter(schema, msgType)).setAll(WriterFactory.this.properties).metricsConfig(metricsConfig).schema(schema).overwrite().build();
                        }
                        case AVRO: {
                            return Avro.write(file).createWriterFunc(ignored -> new SparkAvroWriter(schema)).setAll(WriterFactory.this.properties).schema(schema).overwrite().build();
                        }
                    }
                    throw new UnsupportedOperationException("Cannot write unknown format: " + (Object)((Object)fileFormat));
                }
                catch (IOException e) {
                    throw new RuntimeIOException(e);
                }
            }
        }
    }

    private static class TaskCommit
    implements WriterCommitMessage {
        private final DataFile[] files;

        TaskCommit() {
            this.files = new DataFile[0];
        }

        TaskCommit(DataFile file) {
            this.files = new DataFile[]{file};
        }

        TaskCommit(List<DataFile> files) {
            this.files = files.toArray(new DataFile[files.size()]);
        }

        DataFile[] files() {
            return this.files;
        }
    }
}

