/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.transforms;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Set;
import java.util.UUID;
import org.apache.iceberg.expressions.BoundPredicate;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.UnboundPredicate;
import org.apache.iceberg.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.iceberg.shaded.com.google.common.base.Objects;
import org.apache.iceberg.shaded.com.google.common.collect.Sets;
import org.apache.iceberg.shaded.com.google.common.hash.HashFunction;
import org.apache.iceberg.shaded.com.google.common.hash.Hashing;
import org.apache.iceberg.transforms.Transform;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;

abstract class Bucket<T>
implements Transform<T, Integer> {
    private static final HashFunction MURMUR3 = Hashing.murmur3_32();
    private final int numBuckets;

    static <T> Bucket<T> get(Type type, int numBuckets) {
        switch (type.typeId()) {
            case DATE: 
            case INTEGER: {
                return new BucketInteger(numBuckets);
            }
            case TIME: 
            case TIMESTAMP: 
            case LONG: {
                return new BucketLong(numBuckets);
            }
            case DECIMAL: {
                return new BucketDecimal(numBuckets);
            }
            case STRING: {
                return new BucketString(numBuckets);
            }
            case FIXED: 
            case BINARY: {
                return new BucketByteBuffer(numBuckets);
            }
            case UUID: {
                return new BucketUUID(numBuckets);
            }
        }
        throw new IllegalArgumentException("Cannot bucket by type: " + type);
    }

    private Bucket(int numBuckets) {
        this.numBuckets = numBuckets;
    }

    public Integer numBuckets() {
        return this.numBuckets;
    }

    @VisibleForTesting
    abstract int hash(T var1);

    @Override
    public Integer apply(T value) {
        if (value == null) {
            return null;
        }
        return (this.hash(value) & Integer.MAX_VALUE) % this.numBuckets;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Bucket bucket = (Bucket)o;
        return this.numBuckets == bucket.numBuckets;
    }

    public int hashCode() {
        return Objects.hashCode(this.numBuckets);
    }

    public String toString() {
        return "bucket[" + this.numBuckets + "]";
    }

    @Override
    public UnboundPredicate<Integer> project(String name, BoundPredicate<T> predicate) {
        switch (predicate.op()) {
            case EQ: {
                return Expressions.predicate(predicate.op(), name, this.apply(predicate.literal().value()));
            }
        }
        return null;
    }

    @Override
    public UnboundPredicate<Integer> projectStrict(String name, BoundPredicate<T> predicate) {
        switch (predicate.op()) {
            case NOT_EQ: {
                return Expressions.predicate(predicate.op(), name, this.apply(predicate.literal().value()));
            }
        }
        return null;
    }

    @Override
    public Type getResultType(Type sourceType) {
        return Types.IntegerType.get();
    }

    private static class BucketDecimal
    extends Bucket<BigDecimal> {
        private BucketDecimal(int numBuckets) {
            super(numBuckets);
        }

        @Override
        public int hash(BigDecimal value) {
            return MURMUR3.hashBytes(value.unscaledValue().toByteArray()).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.DECIMAL;
        }
    }

    private static class BucketUUID
    extends Bucket<UUID> {
        private BucketUUID(int numBuckets) {
            super(numBuckets);
        }

        @Override
        public int hash(UUID value) {
            return MURMUR3.newHasher(16).putLong(Long.reverseBytes(value.getMostSignificantBits())).putLong(Long.reverseBytes(value.getLeastSignificantBits())).hash().asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.UUID;
        }
    }

    private static class BucketByteBuffer
    extends Bucket<ByteBuffer> {
        private static final Set<Type.TypeID> SUPPORTED_TYPES = Sets.newHashSet(Type.TypeID.BINARY, Type.TypeID.FIXED);

        private BucketByteBuffer(int numBuckets) {
            super(numBuckets);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public int hash(ByteBuffer value) {
            if (value.hasArray()) {
                return MURMUR3.hashBytes(value.array(), value.arrayOffset() + value.position(), value.arrayOffset() + value.remaining()).asInt();
            }
            int position = value.position();
            byte[] copy = new byte[value.remaining()];
            try {
                value.get(copy);
            }
            finally {
                value.position(position);
            }
            return MURMUR3.hashBytes(copy).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return SUPPORTED_TYPES.contains((Object)type.typeId());
        }
    }

    private static class BucketBytes
    extends Bucket<byte[]> {
        private static final Set<Type.TypeID> SUPPORTED_TYPES = Sets.newHashSet(Type.TypeID.BINARY, Type.TypeID.FIXED);

        private BucketBytes(int numBuckets) {
            super(numBuckets);
        }

        @Override
        public int hash(byte[] value) {
            return MURMUR3.hashBytes(value).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return SUPPORTED_TYPES.contains((Object)type.typeId());
        }
    }

    private static class BucketString
    extends Bucket<CharSequence> {
        private BucketString(int numBuckets) {
            super(numBuckets);
        }

        @Override
        public int hash(CharSequence value) {
            return MURMUR3.hashString(value, StandardCharsets.UTF_8).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.STRING;
        }
    }

    static class BucketDouble
    extends Bucket<Double> {
        BucketDouble(int numBuckets) {
            super(numBuckets);
        }

        @Override
        public int hash(Double value) {
            return MURMUR3.hashLong(Double.doubleToLongBits(value)).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.DOUBLE;
        }
    }

    static class BucketFloat
    extends Bucket<Float> {
        BucketFloat(int numBuckets) {
            super(numBuckets);
        }

        @Override
        public int hash(Float value) {
            return MURMUR3.hashLong(Double.doubleToLongBits(value.floatValue())).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.FLOAT;
        }
    }

    private static class BucketLong
    extends Bucket<Long> {
        private BucketLong(int numBuckets) {
            super(numBuckets);
        }

        @Override
        public int hash(Long value) {
            return MURMUR3.hashLong(value).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.LONG || type.typeId() == Type.TypeID.TIME || type.typeId() == Type.TypeID.TIMESTAMP;
        }
    }

    private static class BucketInteger
    extends Bucket<Integer> {
        private BucketInteger(int numBuckets) {
            super(numBuckets);
        }

        @Override
        public int hash(Integer value) {
            return MURMUR3.hashLong(value.longValue()).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.INTEGER || type.typeId() == Type.TypeID.DATE;
        }
    }
}

