/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.examples.streaming;

import org.apache.ignite.*;
import org.apache.ignite.examples.*;
import org.apache.ignite.lang.*;
import org.apache.ignite.streamer.*;
import org.apache.ignite.streamer.index.*;
import org.jetbrains.annotations.*;

import java.util.*;
import java.util.concurrent.*;

/**
 * This streamer example is inspired by <a href="https://foursquare.com/">Foursquare</a>
 * project. It shows the usage of window indexes, and, particularly,
 * unique index, which allows to skip some duplicated events based on a certain
 * criteria.
 * <p>
 * In this example we have a number of places with 2D locations, and a number
 * of users who perform check-ins in random locations from time to time.
 * A check-in is a streamer event, which is processed through the pipeline
 * of 2 stages and added sequentially to two windows, both of which hold entries
 * within past 10 seconds (the rest of the entries are evicted).
 * <p>
 * First stage simply ensures the user does not check-in twice within the
 * 10 second time interval. This check is done by using a unique window hash index
 * with user name as a key. In case of a duplicate, an error is handled and
 * reported.
 * <p>
 * Second stage checks if a user has checked-in in one of the tracked places.
 * If that's the case, then an info entry is added to a second window. Again,
 * it is only valid for 10 seconds, and is evicted afterwards.
 * <p>
 * There is a separate timer task, which polls a second window index and displays
 * the users that have checked-in in the known places within the last 10 seconds.
 * <p>
 * Remote nodes should always be started with special configuration file:
 * {@code 'ignite.{sh|bat} examples/config/example-streamer.xml'}.
 * When starting nodes this way JAR file containing the examples code
 * should be placed to {@code IGNITE_HOME/libs} folder. You can build
 * {@code ignite-examples.jar} by running {@code mvn package} in
 * {@code IGNITE_HOME/examples} folder. After that {@code ignite-examples.jar}
 * will be generated by Maven in {@code IGNITE_HOME/examples/target} folder.
 * <p>
 * Alternatively you can run {@link StreamingNodeStartup} in another JVM which will start node
 * with {@code examples/config/example-streamer.xml} configuration.
 */
public class StreamingCheckInExample {
    /** Streamer name. */
    private static final String STREAMER_NAME = "check-in";

    /**
     * Nearby distance. Locations with distance less than or equal
     * to this one are considered to be nearby.
     */
    private static final double NEARBY_DISTANCE = 5.0d;

    /** Random number generator. */
    private static final Random RAND = new Random();

    /** Total number of events to generate. */
    private static final int CNT = 60;

    /** User names. */
    private static final String[] USER_NAMES = {
        "Alice", "Bob", "Ann", "Joe", "Mary", "Peter", "Lisa", "Tom", "Kate", "Sam"
    };

    /** Places, for which to track user check-ins. */
    private static final Place[] TRACKED_PLACES = {
        new Place("Theatre", new Location(1.234, 2.567)),
        new Place("Bowling", new Location(10.111, 5.213)),
        new Place("Bar", new Location(15.199, 16.781)),
        new Place("Cinema", new Location(3.77, 20.239))
    };

    /** Max X coordinate. */
    private static final int MAX_X = 30;

    /** Max Y coordinate. */
    private static final int MAX_Y = 30;

    /**
     * Executes example.
     *
     * @param args Command line arguments, none required.
     * @throws IgniteException If example execution failed.
     */
    public static void main(String[] args) throws IgniteException {
        Timer timer = new Timer("check-in-query-worker");

        // Start ignite.
        final Ignite ignite = Ignition.start("examples/config/example-streamer.xml");

        System.out.println();
        System.out.println(">>> Streaming check-in example started.");

        try {
            // Get the streamer.
            IgniteStreamer streamer = ignite.streamer(STREAMER_NAME);

            assert streamer != null;

            // Add a failure listener.
            streamer.addStreamerFailureListener(new StreamerFailureListener() {
                @Override public void onFailure(String stageName, Collection<Object> evts, Throwable err) {
                    System.err.println("Failure [stage=" + stageName + ", evts=" + evts + ", err=" + err.getMessage());
                }
            });

            // Periodically display users, who have checked-in in known places.
            scheduleQuery(streamer, timer);

            // Stream the check-in events.
            streamData(streamer);

            timer.cancel();

            // Reset all streamers on all nodes to make sure that
            // consecutive executions start from scratch.
            ignite.compute().broadcast(new IgniteRunnable() {
                @Override public void run() {
                    if (!ExamplesUtils.hasStreamer(ignite, STREAMER_NAME))
                        System.err.println("Default streamer not found (is example-streamer.xml " +
                            "configuration used on all nodes?)");
                    else {
                        IgniteStreamer streamer = ignite.streamer(STREAMER_NAME);

                        System.out.println("Clearing streamer data.");

                        streamer.reset();
                    }
                }
            });
        }
        finally {
            Ignition.stop(true);
        }
    }

    /**
     * Schedules the query to periodically output the users, who have
     * checked-in in tracked places.
     *
     * @param streamer Streamer.
     * @param timer Timer.
     */
    private static void scheduleQuery(final IgniteStreamer streamer, Timer timer) {
        TimerTask task = new TimerTask() {
            @Override public void run() {
                try {
                    // Send reduce query to all streamers running on local and remote noes.
                    Map<String, Place> userPlaces = streamer.context().reduce(
                        // This closure will execute on remote nodes.
                        new IgniteClosure<StreamerContext, Map<String, Place>>() {
                            @Override public Map<String, Place> apply(
                                StreamerContext ctx) {
                                StreamerWindow<LocationInfo> win =
                                    ctx.window(DetectPlacesStage.class.getSimpleName());

                                assert win != null;

                                StreamerIndex<LocationInfo, String, Place> idxView = win.index();

                                Collection<StreamerIndexEntry<LocationInfo, String, Place>> entries =
                                    idxView.entries(0);

                                Map<String, Place> ret = new HashMap<>(entries.size(), 1.0f);

                                for (StreamerIndexEntry<LocationInfo, String, Place> e : entries)
                                    ret.put(e.key(), e.value());

                                return ret;
                            }
                        },
                        new IgniteReducer<Map<String, Place>, Map<String, Place>>() {
                            private Map<String, Place> map;

                            @Override public boolean collect(@Nullable Map<String, Place> m) {
                                if (m == null)
                                    return false;

                                if (map != null)
                                    map.putAll(m);
                                else
                                    map = m;

                                return true;
                            }

                            @Override public Map<String, Place> reduce() {
                                return map;
                            }
                        }
                    );

                    StringBuilder sb = new StringBuilder("----------------\n");

                    for (Map.Entry<String, Place> userPlace : userPlaces.entrySet())
                        sb.append(String.format("%s is at the %s (%s)\n", userPlace.getKey(),
                            userPlace.getValue().name(), userPlace.getValue().location()));

                    sb.append("----------------\n");

                    System.out.print(sb.toString());
                }
                catch (IgniteException e) {
                    e.printStackTrace();
                }
            }
        };

        // Run task every 3 seconds.
        timer.schedule(task, 3000, 3000);
    }

    /**
     * Streams check-in events into the system.
     *
     * @param streamer Streamer.
     * @throws IgniteException If failed.
     */
    @SuppressWarnings("BusyWait")
    private static void streamData(IgniteStreamer streamer) throws IgniteException {
        try {
            for (int i = 0; i < CNT; i++) {
                CheckInEvent evt = new CheckInEvent(
                    USER_NAMES[ThreadLocalRandom.current().nextInt(USER_NAMES.length)],
                    new Location(
                        RAND.nextDouble() + RAND.nextInt(MAX_X - 1),
                        RAND.nextDouble() + RAND.nextInt(MAX_Y))
                );

                System.out.println(">>> Generating event: " + evt);

                streamer.addEvent(evt);

                Thread.sleep(1000);
            }
        }
        catch (InterruptedException ignored) {
            // No-op.
        }
    }

    /**
     * Entity class that represents a 2D location.
     */
    private static class Location {
        /** Check-in location on X axis (longitude). */
        private final double x;

        /** Check-in location on Y axis (latitude). */
        private final double y;

        /**
         * @param x X value.
         * @param y Y value.
         */
        Location(double x, double y) {
            this.x = x;
            this.y = y;
        }

        /**
         * @return Check-in location on X axis (longitude).
         */
        public double x() {
            return x;
        }

        /**
         * @return Check-in location on Y axis (latitude).
         */
        public double y() {
            return y;
        }

        /** {@inheritDoc} */
        @Override public String toString() {
            return "Location [x=" + x + ", y=" + y + ']';
        }
    }

    /**
     * Entity class representing a place, where
     * users can check-in.
     */
    private static class Place {
        /** Place name. */
        private final String name;

        /** Location. */
        private final Location location;

        /**
         * @param name Name.
         * @param location Location.
         */
        Place(String name, Location location) {
            this.name = name;
            this.location = location;
        }

        /**
         * @return Place name.
         */
        public String name() {
            return name;
        }

        /**
         * @return Location.
         */
        public Location location() {
            return location;
        }

        /** {@inheritDoc} */
        @Override public String toString() {
            return "Place [name=" + name + ", location=" + location + ']';
        }
    }

    /**
     * Check-in event.
     */
    private static class CheckInEvent {
        /** User name. */
        private final String userName;

        /** User location. */
        private final Location location;

        /**
         * @param userName User name.
         * @param location Location.
         */
        CheckInEvent(String userName, Location location) {
            this.userName = userName;
            this.location = location;
        }

        /**
         * @return User name.
         */
        public String userName() {
            return userName;
        }

        /**
         * @return User location.
         */
        public Location location() {
            return location;
        }

        /** {@inheritDoc} */
        @Override public String toString() {
            return "CheckInEvent [userName=" + userName + ", location=" + location + ']';
        }
    }

    /**
     * Helper data structure for keeping information about
     * check-in location and a corresponding place if found.
     */
    private static class LocationInfo {
        /** User name. */
        private final String userName;

        /** A detected check-in place. */
        private final Place place;

        /**
         * @param userName User name.
         * @param place Place.
         */
        LocationInfo(String userName, Place place) {
            this.userName = userName;
            this.place = place;
        }

        /**
         * @return User name.
         */
        public String userName() {
            return userName;
        }

        /**
         * @return A detected check-in place.
         */
        public Place place() {
            return place;
        }

        /** {@inheritDoc} */
        @Override public String toString() {
            return "LocationInfo [userName=" + userName + ", place=" + place + ']';
        }
    }

    /**
     * Check-in event processing stage that adds events window
     * with unique index to block repetitive check-ins.
     */
    @SuppressWarnings("PublicInnerClass")
    public static class AddToWindowStage implements StreamerStage<CheckInEvent> {
        /** {@inheritDoc} */
        @Override public String name() {
            return getClass().getSimpleName();
        }

        /** {@inheritDoc} */
        @Nullable @Override public Map<String, Collection<?>> run(
            StreamerContext ctx, Collection<CheckInEvent> evts) {
            StreamerWindow<CheckInEvent> win = ctx.window(name());

            assert win != null;

            Collection<CheckInEvent> evts0 = new LinkedList<>();

            // Add events to window. Our unique index should reject
            // repetitive check-ins within a period of time, defined
            // by the window.
            for (CheckInEvent evt : evts) {
                try {
                    win.enqueue(evt);

                    evts0.add(evt);
                }
                catch (IgniteException e) {
                    if (e.getMessage().contains("Index unique key violation"))
                        System.err.println("Cannot check-in twice within the specified period of time [evt=" + evt + ']');
                    else
                        throw e;
                }
            }

            // Clear evicted events.
            win.pollEvictedAll();

            // Move to the next stage in pipeline, if there are valid events.
            if (!evts0.isEmpty())
                return Collections.<String, Collection<?>>singletonMap(ctx.nextStageName(), evts0);

            // Break the pipeline execution.
            return null;
        }
    }

    /**
     * Check-in event processing stage that detects the
     * check-in places.
     */
    private static class DetectPlacesStage implements StreamerStage<CheckInEvent> {
        /** {@inheritDoc} */
        @Override public String name() {
            return getClass().getSimpleName();
        }

        /** {@inheritDoc} */
        @Nullable @Override public Map<String, Collection<?>> run(StreamerContext ctx,
            Collection<CheckInEvent> evts) {
            StreamerWindow<LocationInfo> win = ctx.window(name());

            assert win != null;

            for (CheckInEvent evt : evts) {
                for (Place place : TRACKED_PLACES) {
                    if (distance(evt.location(), place.location()) <= NEARBY_DISTANCE) {
                        win.enqueue(new LocationInfo(evt.userName(), place));

                        break;
                    }
                }
            }

            // Clear evicted location infos.
            win.pollEvictedAll();

            // Null means there are no more stages and
            // we should finish the pipeline.
            return null;
        }

        /**
         * Calculates the distance between 2 locations.
         *
         * @param loc1 First location.
         * @param loc2 Second location.
         * @return Distance between locations.
         */
        private double distance(Location loc1, Location loc2) {
            double xDiff = Math.abs(loc1.x() - loc2.x());
            double yDiff = Math.abs(loc1.y() - loc2.y());

            // Return a vector distance between the points.
            return Math.sqrt(xDiff * xDiff + yDiff * yDiff);
        }
    }

    /**
     * Index updater for check-in events. Updaters are specified for {@link StreamerIndexProviderAdapter} in
     * streamer configuration.
     */
    private static class CheckInEventIndexUpdater implements StreamerIndexUpdater<CheckInEvent, String, Location> {
        /** {@inheritDoc} */
        @Nullable @Override public String indexKey(CheckInEvent evt) {
            return evt.userName(); // Index key is an event user name.
        }

        /** {@inheritDoc} */
        @Nullable @Override public Location initialValue(CheckInEvent evt, String key) {
            return evt.location(); // Index value is an event location.
        }

        /** {@inheritDoc} */
        @Nullable @Override public Location onAdded(
            StreamerIndexEntry<CheckInEvent, String, Location> entry,
            CheckInEvent evt) {
            throw new AssertionError("onAdded() shouldn't be called on unique index.");
        }

        /** {@inheritDoc} */
        @Nullable @Override public Location onRemoved(
            StreamerIndexEntry<CheckInEvent, String, Location> entry,
            CheckInEvent evt) {
            return null;
        }
    }

    /**
     * Index updater for location info. Updaters are specified for {@link StreamerIndexProviderAdapter} in
     * streamer configuration.
     */
    private static class PlacesIndexUpdater implements StreamerIndexUpdater<LocationInfo, String, Place> {
        /** {@inheritDoc} */
        @Nullable @Override public String indexKey(LocationInfo info) {
            return info.userName();
        }

        /** {@inheritDoc} */
        @Nullable @Override public Place initialValue(LocationInfo info, String key) {
            return info.place();
        }

        /** {@inheritDoc} */
        @Nullable @Override public Place onAdded(
            StreamerIndexEntry<LocationInfo, String, Place> entry,
            LocationInfo evt) {
            throw new AssertionError("onAdded() shouldn't be called on unique index.");
        }

        /** {@inheritDoc} */
        @Nullable @Override public Place onRemoved(
            StreamerIndexEntry<LocationInfo, String, Place> entry,
            LocationInfo evt) {
            return null;
        }
    }
}
