/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.blob.datastore;

import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.jackrabbit.core.data.DataRecord;
import org.apache.jackrabbit.guava.common.base.Stopwatch;
import org.apache.jackrabbit.oak.commons.FileIOUtils;
import org.apache.jackrabbit.oak.commons.collections.IterableUtils;
import org.apache.jackrabbit.oak.commons.collections.ListUtils;
import org.apache.jackrabbit.oak.commons.concurrent.ExecutorCloser;
import org.apache.jackrabbit.oak.commons.io.BurnOnCloseFileIterator;
import org.apache.jackrabbit.oak.commons.io.FileLineDifferenceIterator;
import org.apache.jackrabbit.oak.commons.io.FileTreeTraverser;
import org.apache.jackrabbit.oak.plugins.blob.SharedDataStore;
import org.apache.jackrabbit.oak.plugins.blob.datastore.BlobTracker;
import org.apache.jackrabbit.oak.stats.Clock;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlobIdTracker
implements Closeable,
BlobTracker {
    private static final Logger LOG = LoggerFactory.getLogger(BlobIdTracker.class);
    private static final String datastoreMeta = "blobids";
    private static final String fileNamePrefix = "blob";
    private static final String mergedFileSuffix = ".refs";
    private static final String snapshotMarkerSuffix = ".snapshot";
    private String instanceId = UUID.randomUUID().toString();
    private SharedDataStore datastore;
    private long snapshotInterval;
    private ActiveDeletionTracker deleteTracker;
    protected BlobIdStore store;
    private ScheduledExecutorService scheduler;
    private String prefix;
    private File rootDir;

    private BlobIdTracker() {
    }

    private BlobIdTracker(String path, String repositoryId, long snapshotIntervalSecs, SharedDataStore datastore) throws IOException {
        this(path, repositoryId, Executors.newSingleThreadScheduledExecutor(), snapshotIntervalSecs, snapshotIntervalSecs, datastore);
    }

    private BlobIdTracker(String path, String repositoryId, ScheduledExecutorService scheduler, long snapshotDelaySecs, long snapshotIntervalSecs, SharedDataStore datastore) throws IOException {
        String root = FilenameUtils.concat((String)path, (String)datastoreMeta);
        this.rootDir = new File(root);
        this.datastore = datastore;
        this.scheduler = scheduler;
        this.snapshotInterval = TimeUnit.SECONDS.toMillis(snapshotIntervalSecs);
        try {
            FileUtils.forceMkdir((File)this.rootDir);
            this.prefix = "blob-" + repositoryId;
            this.store = new BlobIdStore(this.rootDir, this.prefix);
            scheduler.scheduleAtFixedRate(new SnapshotJob(), TimeUnit.SECONDS.toMillis(snapshotDelaySecs), TimeUnit.SECONDS.toMillis(snapshotIntervalSecs), TimeUnit.MILLISECONDS);
            this.deleteTracker = new ActiveDeletionTracker(this.rootDir, this.prefix);
        }
        catch (IOException e) {
            LOG.error("Error initializing blob tracker", (Throwable)e);
            this.close();
            throw e;
        }
    }

    public static BlobIdTracker build(String path, String repositoryId, long snapshotIntervalSecs, SharedDataStore datastore) throws IOException {
        if (snapshotIntervalSecs > 0L) {
            return new BlobIdTracker(path, repositoryId, snapshotIntervalSecs, datastore);
        }
        return new BlobIdTracker(){

            @Override
            public ActiveDeletionTracker getDeleteTracker() {
                return new ActiveDeletionTracker(){

                    @Override
                    public void track(File recs) {
                    }

                    @Override
                    public File retrieve(String path) {
                        File copiedRecsFile = new File(path);
                        return copiedRecsFile;
                    }

                    @Override
                    public void reconcile(File recs) {
                    }

                    @Override
                    public Iterator<String> filter(File recs) {
                        return Collections.emptyIterator();
                    }
                };
            }

            @Override
            public void remove(File recs, BlobTracker.Options options) {
            }

            @Override
            public void remove(File recs) {
            }

            @Override
            public void remove(Iterator<String> recs) {
            }

            @Override
            public void add(String id) {
            }

            @Override
            public void add(Iterator<String> recs) {
            }

            @Override
            public void add(File recs) {
            }

            @Override
            public Iterator<String> get() {
                return Collections.emptyIterator();
            }

            @Override
            public File get(String path) {
                return new File(path);
            }

            @Override
            public void close() {
            }
        };
    }

    public ActiveDeletionTracker getDeleteTracker() {
        return this.deleteTracker;
    }

    @Override
    public void remove(File recs, BlobTracker.Options options) throws IOException {
        this.globalMerge();
        if (options == BlobTracker.Options.ACTIVE_DELETION) {
            this.deleteTracker.track(recs);
        }
        this.store.removeRecords(recs);
        this.snapshot(true);
    }

    @Override
    public void remove(File recs) throws IOException {
        this.globalMerge();
        this.store.removeRecords(recs);
        this.snapshot(true);
    }

    @Override
    public void remove(Iterator<String> recs) throws IOException {
        this.globalMerge();
        this.store.removeRecords(recs);
        this.snapshot(true);
    }

    @Override
    public void add(String id) throws IOException {
        this.store.addRecord(id);
    }

    @Override
    public void add(Iterator<String> recs) throws IOException {
        this.store.addRecords(recs);
    }

    @Override
    public void add(File recs) throws IOException {
        this.store.addRecords(recs);
    }

    @Override
    public Iterator<String> get() throws IOException {
        try {
            this.globalMerge();
            return this.store.getRecords();
        }
        catch (IOException e) {
            LOG.error("Error in retrieving blob records iterator", (Throwable)e);
            throw e;
        }
    }

    @Override
    public File get(String path) throws IOException {
        this.globalMerge();
        return this.store.getRecords(path);
    }

    private void globalMerge() throws IOException {
        try {
            Stopwatch watch = Stopwatch.createStarted();
            LOG.trace("Retrieving all blob id files available form the DataStore");
            List<DataRecord> refRecords = this.datastore.getAllMetadataRecords(fileNamePrefix);
            List refFiles = ListUtils.toList((Iterable)IterableUtils.transform(refRecords, input -> {
                InputStream inputStream = null;
                try {
                    inputStream = input.getStream();
                    File file = FileIOUtils.copy((InputStream)inputStream);
                    return file;
                }
                catch (Exception e) {
                    LOG.warn("Error copying data store file locally {}", (Object)input.getIdentifier(), (Object)e);
                }
                finally {
                    IOUtils.closeQuietly((InputStream)inputStream);
                }
                return null;
            }));
            LOG.info("Retrieved all blob id files in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
            watch = Stopwatch.createStarted();
            this.store.merge(refFiles, true);
            LOG.info("Merged all retrieved blob id files in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
            watch = Stopwatch.createStarted();
            for (DataRecord rec : refRecords) {
                this.datastore.deleteMetadataRecord(rec.getIdentifier().toString());
                LOG.debug("Deleted metadata record {}", (Object)rec.getIdentifier().toString());
            }
            LOG.info("Deleted all blob id metadata files in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
        }
        catch (IOException e) {
            LOG.error("Error in merging blob records iterator from the data store", (Throwable)e);
            throw e;
        }
    }

    private void snapshot() throws IOException {
        this.snapshot(false);
    }

    private void snapshot(boolean skipStoreSnapshot) throws IOException {
        try {
            Stopwatch watch = Stopwatch.createStarted();
            if (!skipStoreSnapshot) {
                this.store.snapshot();
                LOG.debug("Completed snapshot in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
            }
            watch = Stopwatch.createStarted();
            File recs = this.store.getBlobRecordsFile();
            this.datastore.addMetadataRecord(recs, this.prefix + this.instanceId + System.currentTimeMillis() + mergedFileSuffix);
            LOG.info("Added blob id metadata record in DataStore in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
            try {
                FileUtils.forceDelete((File)recs);
                LOG.info("Deleted blob record file after snapshot and upload {}", (Object)recs);
                FileUtils.touch((File)this.getSnapshotMarkerFile());
                LOG.info("Updated snapshot marker");
            }
            catch (IOException e) {
                LOG.debug("Failed to in cleaning up {}", (Object)recs, (Object)e);
            }
        }
        catch (Exception e) {
            LOG.error("Error taking snapshot", (Throwable)e);
            throw new IOException("Snapshot error", e);
        }
    }

    private File getSnapshotMarkerFile() {
        File snapshotMarker = new File(this.rootDir, this.prefix + snapshotMarkerSuffix);
        return snapshotMarker;
    }

    @Override
    public void close() throws IOException {
        this.store.close();
        new ExecutorCloser((ExecutorService)this.scheduler).close();
    }

    class SnapshotJob
    implements Runnable {
        private long interval;
        private Clock clock;

        public SnapshotJob() {
            this.interval = BlobIdTracker.this.snapshotInterval;
            this.clock = Clock.SIMPLE;
        }

        public SnapshotJob(long millis, Clock clock) {
            this.interval = millis;
            this.clock = clock;
        }

        @Override
        public void run() {
            if (!this.skip()) {
                try {
                    BlobIdTracker.this.snapshot();
                    LOG.info("Finished taking snapshot");
                }
                catch (Exception e) {
                    LOG.warn("Failure in taking snapshot", (Throwable)e);
                }
            } else {
                LOG.info("Skipping scheduled snapshot as it last executed within {} seconds", (Object)TimeUnit.MILLISECONDS.toSeconds(this.interval));
            }
        }

        private boolean skip() {
            File snapshotMarker = BlobIdTracker.this.getSnapshotMarkerFile();
            return snapshotMarker.exists() && snapshotMarker.lastModified() > this.clock.getTime() - this.interval;
        }
    }

    static class BlobIdStore
    implements Closeable {
        private static final String genFileNameSuffix = ".gen";
        private static final String workingCopySuffix = ".process";
        private BufferedWriter writer;
        private File processFile;
        private final List<File> generations;
        private final File rootDir;
        private final String prefix;
        private final ReentrantLock refLock;
        private final ReentrantLock snapshotLock;

        BlobIdStore(File rootDir, String prefix) throws IOException {
            this.rootDir = rootDir;
            this.prefix = prefix;
            this.refLock = new ReentrantLock();
            this.snapshotLock = new ReentrantLock();
            this.processFile = FileTreeTraverser.breadthFirst((File)rootDir).filter(file -> Type.IN_PROCESS.filter().test((File)file)).findFirst().orElse(null);
            this.generations = Collections.synchronizedList(FileTreeTraverser.breadthFirst((File)rootDir).filter(file -> Type.GENERATION.filter().test((File)file)).collect(Collectors.toList()));
            this.nextGeneration();
        }

        protected synchronized void addRecord(String id) throws IOException {
            this.writer.append(id);
            this.writer.newLine();
            this.writer.flush();
            LOG.debug("Added record {}", (Object)id);
        }

        protected Iterator<String> getRecords() throws IOException {
            try {
                String path = File.createTempFile("temp", null).getAbsolutePath();
                return BurnOnCloseFileIterator.wrap((Iterator)FileUtils.lineIterator((File)this.getRecords(path)), (File)new File(path));
            }
            catch (IOException e) {
                LOG.error("Error in retrieving blob records iterator", (Throwable)e);
                throw e;
            }
        }

        protected File getRecords(String path) throws IOException {
            this.refLock.lock();
            File copiedRecsFile = new File(path);
            try {
                FileUtils.copyFile((File)this.getBlobRecordsFile(), (File)copiedRecsFile);
                File file = copiedRecsFile;
                return file;
            }
            catch (IOException e) {
                LOG.error("Error in retrieving blob records file", (Throwable)e);
                throw e;
            }
            finally {
                this.refLock.unlock();
            }
        }

        protected File getBlobRecordsFile() throws IOException {
            File refs = new File(this.rootDir, this.prefix + Type.REFS.getFileNameSuffix());
            if (!refs.exists()) {
                LOG.debug("File created {}", (Object)refs.createNewFile());
            }
            return refs;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void merge(List<File> refFiles, boolean doSort) throws IOException {
            this.refLock.lock();
            try {
                if (refFiles != null && !refFiles.isEmpty()) {
                    File merged = new File(this.rootDir, this.prefix + Type.REFS.getFileNameSuffix());
                    FileIOUtils.append(refFiles, (File)merged, (boolean)true);
                    LOG.debug("Merged files into references {}", refFiles);
                    refFiles.clear();
                }
                if (doSort) {
                    FileIOUtils.sort((File)this.getBlobRecordsFile());
                }
            }
            finally {
                this.refLock.unlock();
            }
        }

        protected void removeRecords(Iterator<String> recs) throws IOException {
            File deleted = File.createTempFile("deleted", null);
            FileIOUtils.writeStrings(recs, (File)deleted, (boolean)false);
            this.removeRecords(deleted);
            LOG.trace("Removed records");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void removeRecords(File recs) throws IOException {
            this.snapshot();
            this.refLock.lock();
            try {
                FileIOUtils.sort((File)this.getBlobRecordsFile());
                FileIOUtils.sort((File)recs);
                LOG.trace("Sorted files");
                File temp = File.createTempFile("sorted", null);
                try (FileLineDifferenceIterator iterator = null;){
                    iterator = new FileLineDifferenceIterator(recs, this.getBlobRecordsFile(), null);
                    FileIOUtils.writeStrings((Iterator)iterator, (File)temp, (boolean)false);
                }
                File blobRecs = this.getBlobRecordsFile();
                Files.move(temp.toPath(), blobRecs.toPath(), StandardCopyOption.REPLACE_EXISTING);
                LOG.trace("removed records");
            }
            finally {
                this.refLock.unlock();
                try {
                    FileUtils.forceDelete((File)recs);
                }
                catch (IOException e) {
                    LOG.debug("Failed to delete file {}", (Object)recs, (Object)e);
                }
            }
        }

        private synchronized void nextGeneration() throws IOException {
            this.close();
            this.processFile = new File(this.rootDir, this.prefix + Type.IN_PROCESS.getFileNameSuffix());
            this.writer = new BufferedWriter(new FileWriter(this.processFile, StandardCharsets.UTF_8));
            LOG.info("Created new process file and writer over {} ", (Object)this.processFile.getAbsolutePath());
        }

        protected void addRecords(Iterator<String> recs) throws IOException {
            File added = File.createTempFile("added", null);
            FileIOUtils.writeStrings(recs, (File)added, (boolean)false);
            this.merge(new ArrayList<File>(Arrays.asList(added)), false);
        }

        protected void addRecords(File recs) throws IOException {
            this.merge(new ArrayList<File>(Arrays.asList(recs)), false);
        }

        protected void snapshot() throws IOException {
            this.snapshotLock.lock();
            try {
                this.nextGeneration();
                this.merge(this.generations, false);
            }
            finally {
                this.snapshotLock.unlock();
            }
        }

        @Override
        public synchronized void close() {
            IOUtils.closeQuietly((Writer)this.writer);
            LOG.info("Closed writer");
            if (this.processFile != null) {
                File renamed = new File(FilenameUtils.removeExtension((String)this.processFile.getAbsolutePath()));
                boolean success = this.processFile.renameTo(renamed);
                LOG.debug("File renamed {}", (Object)success);
                if (success) {
                    this.generations.add(renamed);
                    LOG.info("Process file renamed to {}", (Object)renamed.getAbsolutePath());
                } else {
                    LOG.trace("Trying a copy file operation");
                    try {
                        if (renamed.createNewFile()) {
                            Files.copy(this.processFile.toPath(), renamed.toPath(), StandardCopyOption.REPLACE_EXISTING);
                            this.generations.add(renamed);
                            LOG.info("{} File copied to {}", (Object)this.processFile.getAbsolutePath(), (Object)renamed.getAbsolutePath());
                        }
                    }
                    catch (Exception e) {
                        LOG.warn("Unable to copy process file to corresponding gen file. Some elements may be missed", (Throwable)e);
                    }
                }
            }
        }

        static enum Type {
            IN_PROCESS{

                @Override
                String getFileNameSuffix() {
                    return "." + System.currentTimeMillis() + ".gen.process";
                }

                @Override
                Predicate<File> filter() {
                    return new Predicate<File>(){

                        @Override
                        public boolean test(File input) {
                            return input.getName().endsWith(BlobIdStore.workingCopySuffix) && input.getName().startsWith(BlobIdTracker.fileNamePrefix);
                        }
                    };
                }
            }
            ,
            GENERATION{

                @Override
                String getFileNameSuffix() {
                    return "." + System.currentTimeMillis() + BlobIdStore.genFileNameSuffix;
                }

                @Override
                Predicate<File> filter() {
                    return new Predicate<File>(){

                        @Override
                        public boolean test(File input) {
                            return input.getName().startsWith(BlobIdTracker.fileNamePrefix) && input.getName().contains(BlobIdStore.genFileNameSuffix) && !input.getName().endsWith(BlobIdStore.workingCopySuffix);
                        }
                    };
                }
            }
            ,
            REFS{

                @Override
                String getFileNameSuffix() {
                    return BlobIdTracker.mergedFileSuffix;
                }

                @Override
                Predicate<File> filter() {
                    return new Predicate<File>(){

                        @Override
                        public boolean test(File input) {
                            return input.getName().endsWith(BlobIdTracker.mergedFileSuffix) && input.getName().startsWith(BlobIdTracker.fileNamePrefix);
                        }
                    };
                }
            };


            String getFileNameSuffix() {
                return "";
            }

            Predicate<File> filter() {
                return x -> true;
            }
        }
    }

    public static class ActiveDeletionTracker {
        private static final String DEL_SUFFIX = ".del";
        private File delFile;
        public static final String DELIM = ",";
        private ReentrantLock lock;
        private static final Function<String, String> transformer = new Function<String, String>(){

            @Override
            @Nullable
            public String apply(@Nullable String input) {
                if (input != null) {
                    return input.split(ActiveDeletionTracker.DELIM)[0];
                }
                return "";
            }
        };

        ActiveDeletionTracker(File rootDir, String prefix) throws IOException {
            this.delFile = new File(rootDir, prefix + DEL_SUFFIX);
            FileUtils.touch((File)this.delFile);
            this.lock = new ReentrantLock();
        }

        private ActiveDeletionTracker() {
        }

        public void track(File recs) throws IOException {
            this.lock.lock();
            try {
                FileIOUtils.append(new ArrayList<File>(Arrays.asList(recs)), (File)this.delFile, (boolean)false);
                FileIOUtils.sort((File)this.delFile);
            }
            finally {
                this.lock.unlock();
            }
        }

        public File retrieve(String path) throws IOException {
            File copiedRecsFile = new File(path);
            try {
                FileUtils.copyFile((File)this.delFile, (File)copiedRecsFile);
                return copiedRecsFile;
            }
            catch (IOException e) {
                LOG.error("Error in retrieving active deletions file", (Throwable)e);
                throw e;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void reconcile(File recs) throws IOException {
            this.lock.lock();
            try {
                File toBeRemoved = File.createTempFile("toBeRemoved", null);
                File removed = File.createTempFile("removed", null);
                FileLineDifferenceIterator toBeRemovedIterator = null;
                FileLineDifferenceIterator removeIterator = null;
                try {
                    toBeRemovedIterator = new FileLineDifferenceIterator(recs, this.delFile, null);
                    FileIOUtils.writeStrings((Iterator)toBeRemovedIterator, (File)toBeRemoved, (boolean)false);
                    removeIterator = new FileLineDifferenceIterator(toBeRemoved, this.delFile, null);
                    FileIOUtils.writeStrings((Iterator)removeIterator, (File)removed, (boolean)false);
                }
                finally {
                    if (toBeRemovedIterator != null) {
                        toBeRemovedIterator.close();
                    }
                    if (removeIterator != null) {
                        removeIterator.close();
                    }
                    if (toBeRemoved != null) {
                        toBeRemoved.delete();
                    }
                }
                Files.move(removed.toPath(), this.delFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                LOG.trace("removed active delete records");
            }
            finally {
                this.lock.unlock();
            }
        }

        public Iterator<String> filter(File recs) throws IOException {
            return new FileLineDifferenceIterator(this.delFile, recs, transformer);
        }
    }
}

