/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.lucene.hybrid;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.io.FileUtils;
import org.apache.jackrabbit.oak.plugins.index.lucene.IndexCopier;
import org.apache.jackrabbit.oak.plugins.index.lucene.LuceneIndexDefinition;
import org.apache.jackrabbit.oak.plugins.index.lucene.directory.DirectoryUtils;
import org.apache.jackrabbit.oak.plugins.index.lucene.hybrid.NRTDirectoryFactory;
import org.apache.jackrabbit.oak.plugins.index.lucene.reader.LuceneIndexReader;
import org.apache.jackrabbit.oak.plugins.index.lucene.writer.IndexWriterUtils;
import org.apache.jackrabbit.oak.plugins.index.lucene.writer.LuceneIndexWriter;
import org.apache.jackrabbit.oak.plugins.index.search.update.IndexUpdateListener;
import org.apache.jackrabbit.oak.plugins.index.search.update.ReaderRefreshPolicy;
import org.apache.jackrabbit.oak.stats.HistogramStats;
import org.apache.jackrabbit.oak.stats.MeterStats;
import org.apache.jackrabbit.oak.stats.StatisticsProvider;
import org.apache.jackrabbit.oak.stats.StatsOptions;
import org.apache.jackrabbit.oak.stats.TimerStats;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.IndexableField;
import org.apache.lucene.search.suggest.analyzing.AnalyzingInfixSuggester;
import org.apache.lucene.store.Directory;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NRTIndex
implements Closeable {
    private static final boolean REGULAR_CLOSE = Boolean.getBoolean("oak.lucene.nrt.regularClose");
    private static final boolean RETAIN_DURING_CLOSE = Boolean.getBoolean("oak.lucene.nrt.retainDuringClose");
    private static final AtomicInteger COUNTER = new AtomicInteger();
    private static final Logger log = LoggerFactory.getLogger(NRTIndex.class);
    public static final String NRT_DIR_PREFIX = "nrt-";
    private final LuceneIndexDefinition definition;
    private final IndexCopier indexCopier;
    private final IndexUpdateListener refreshPolicy;
    private final StatisticsProvider statisticsProvider;
    private final TimerStats refreshTimer;
    private final HistogramStats sizeHisto;
    private final TimerStats.Context openTime;
    private final NRTDirectoryFactory directoryFactory;
    private NRTIndex previous;
    private IndexWriter indexWriter;
    private NRTIndexWriter nrtIndexWriter;
    private File indexDir;
    private Directory directory;
    private DirectoryReader dirReader;
    private DirectoryReader dirReaderUsedForPrevious;
    private boolean closed;
    private boolean previousModeEnabled;
    private List<LuceneIndexReader> readers;
    private final List<IndexReader> openedReaders;
    private final boolean assertAllReadersClosed;

    public NRTIndex(LuceneIndexDefinition definition, IndexCopier indexCopier, IndexUpdateListener refreshPolicy, @Nullable NRTIndex previous, StatisticsProvider statisticsProvider, NRTDirectoryFactory directoryFactory, boolean assertAllReadersClosed) {
        this.definition = definition;
        this.indexCopier = indexCopier;
        this.refreshPolicy = refreshPolicy;
        this.previous = previous;
        this.statisticsProvider = statisticsProvider;
        this.directoryFactory = directoryFactory;
        this.assertAllReadersClosed = assertAllReadersClosed;
        this.openedReaders = assertAllReadersClosed ? new LinkedList() : Collections.emptyList();
        this.refreshTimer = statisticsProvider.getTimer(this.metricName("REFRESH_TIME"), StatsOptions.METRICS_ONLY);
        this.sizeHisto = statisticsProvider.getHistogram(this.metricName("SIZE"), StatsOptions.METRICS_ONLY);
        this.openTime = statisticsProvider.getTimer(this.metricName("OPEN_TIME"), StatsOptions.METRICS_ONLY).time();
    }

    @Nullable
    private LuceneIndexReader getPrimaryReader() {
        DirectoryReader latestReader = this.createReader(this.dirReaderUsedForPrevious);
        while (latestReader != null && !latestReader.tryIncRef()) {
            latestReader = this.createReader(this.dirReaderUsedForPrevious);
        }
        if (latestReader != this.dirReaderUsedForPrevious) {
            this.decrementReaderUseCount(this.dirReaderUsedForPrevious);
            this.dirReaderUsedForPrevious = latestReader;
        }
        return latestReader != null ? new NRTReader(latestReader, this.directory) : null;
    }

    public LuceneIndexWriter getWriter() throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        if (this.nrtIndexWriter == null) {
            this.nrtIndexWriter = this.createWriter();
        }
        return this.nrtIndexWriter;
    }

    public synchronized List<LuceneIndexReader> getReaders() {
        LuceneIndexReader previousReader;
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        Preconditions.checkState((!this.previousModeEnabled ? 1 : 0) != 0);
        DirectoryReader latestReader = this.createReader(this.dirReader);
        if (latestReader == this.dirReader && this.readers != null) {
            return this.readers;
        }
        ArrayList newReaders = Lists.newArrayListWithCapacity((int)2);
        if (latestReader != null) {
            newReaders.add(new NRTReader(latestReader, this.directory));
        }
        LuceneIndexReader luceneIndexReader = previousReader = this.previous != null ? this.previous.getPrimaryReader() : null;
        if (previousReader != null) {
            newReaders.add(previousReader);
        }
        this.decrementReaderUseCount(this.readers);
        this.dirReader = latestReader;
        this.readers = ImmutableList.copyOf((Collection)newReaders);
        return this.readers;
    }

    public ReaderRefreshPolicy getRefreshPolicy() {
        return this.refreshPolicy;
    }

    public void disconnectPrevious() {
        this.decrementReaderUseCount(this.readers);
        this.readers = Collections.emptyList();
        this.previousModeEnabled = true;
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        log.debug("[{}] Closing NRTIndex [{}]", (Object)this.definition.getIndexPath(), (Object)this.getName());
        this.decrementReaderUseCount(this.dirReaderUsedForPrevious);
        this.decrementReaderUseCount(this.readers);
        this.assertAllReadersAreClosed();
        if (this.indexWriter != null) {
            long time = System.nanoTime();
            if (REGULAR_CLOSE) {
                this.indexWriter.close();
            } else {
                if (!RETAIN_DURING_CLOSE) {
                    this.indexWriter.deleteAll();
                }
                this.indexWriter.close(false);
            }
            time = System.nanoTime() - time;
            if (time > 100000000L) {
                log.info("Closing time: {} ns", (Object)time);
            } else if (log.isTraceEnabled()) {
                log.trace("Closing time: {} ns", (Object)time);
            }
            this.sizeHisto.update(DirectoryUtils.dirSize(this.directory));
            this.directory.close();
            FileUtils.deleteQuietly((File)this.indexDir);
            log.debug("[{}] Removed directory [{}]", (Object)this, (Object)this.indexDir);
        }
        this.previous = null;
        this.closed = true;
        this.openTime.stop();
    }

    public boolean isClosed() {
        return this.closed;
    }

    NRTIndex getPrevious() {
        return this.previous;
    }

    public String toString() {
        return String.format("%s (%s)", this.definition.getIndexPath(), this.getName());
    }

    File getIndexDir() {
        return this.indexDir;
    }

    private String getName() {
        return this.indexDir != null ? this.indexDir.getName() : "UNKNOWN";
    }

    private void assertAllReadersAreClosed() {
        for (IndexReader r : this.openedReaders) {
            if (r.getRefCount() == 0) continue;
            String msg = String.format("Unclosed reader found with refCount %d for index %s", r.getRefCount(), this.toString());
            throw new IllegalStateException(msg);
        }
    }

    private void decrementReaderUseCount(@Nullable List<LuceneIndexReader> readers) {
        if (readers != null) {
            for (LuceneIndexReader r : readers) {
                this.decrementReaderUseCount(r.getReader());
            }
        }
    }

    private void decrementReaderUseCount(IndexReader reader) {
        try {
            if (reader != null) {
                reader.decRef();
            }
        }
        catch (IOException e) {
            log.warn("[{}] Error occurred while releasing reader instance {}", new Object[]{this.definition.getIndexPath(), this.toString(), e});
        }
    }

    @Nullable
    private synchronized DirectoryReader createReader(DirectoryReader dirReader) {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        if (this.indexWriter == null) {
            return null;
        }
        DirectoryReader result = dirReader;
        try {
            TimerStats.Context ctx = this.refreshTimer.time();
            if (dirReader == null || dirReader.getRefCount() == 0) {
                result = DirectoryReader.open(this.indexWriter, false);
            } else {
                DirectoryReader newReader = DirectoryReader.openIfChanged(dirReader, this.indexWriter, false);
                if (newReader != null) {
                    result = newReader;
                }
            }
            ctx.stop();
            if (this.assertAllReadersClosed && result != null && result != dirReader) {
                this.openedReaders.add(result);
            }
            return result;
        }
        catch (IOException e) {
            log.warn("Error opening index [{}]", (Throwable)e);
            return null;
        }
    }

    private synchronized NRTIndexWriter createWriter() throws IOException {
        String dirName = NRTIndex.generateDirName();
        this.indexDir = this.indexCopier.getIndexDir(this.definition, this.definition.getIndexPath(), dirName);
        this.directory = this.directoryFactory.createNRTDir(this.definition, this.indexDir);
        IndexWriterConfig config = IndexWriterUtils.getIndexWriterConfig(this.definition, false);
        this.indexWriter = new IndexWriter(this.directory, config);
        log.debug("[{}] Created NRTIndex [{}]", (Object)this.definition.getIndexPath(), (Object)this.getName());
        return new NRTIndexWriter(this.indexWriter);
    }

    IndexReader getPrimaryReaderForTest() {
        return this.getReaders().get(0).getReader();
    }

    public static String generateDirName() {
        long uniqueCount = System.currentTimeMillis() + (long)COUNTER.incrementAndGet();
        return NRT_DIR_PREFIX + uniqueCount;
    }

    private String metricName(String suffix) {
        return String.format("%s_NRT_%s", this.definition.getIndexPath(), suffix);
    }

    private class NRTIndexWriter
    implements LuceneIndexWriter {
        private final IndexWriter indexWriter;
        private final MeterStats updateMeter;

        public NRTIndexWriter(IndexWriter indexWriter) {
            this.indexWriter = indexWriter;
            this.updateMeter = NRTIndex.this.statisticsProvider.getMeter(NRTIndex.this.metricName("UPDATES"), StatsOptions.METRICS_ONLY);
        }

        @Override
        public void updateDocument(String path, Iterable<? extends IndexableField> doc) throws IOException {
            this.indexWriter.addDocument(doc);
            NRTIndex.this.refreshPolicy.updated();
            this.updateMeter.mark();
        }

        @Override
        public void deleteDocuments(String path) throws IOException {
        }

        @Override
        public boolean close(long timestamp) throws IOException {
            throw new IllegalStateException("Close should not be called");
        }
    }

    private static class NRTReader
    implements LuceneIndexReader {
        private final IndexReader indexReader;
        private final Directory directory;

        public NRTReader(IndexReader indexReader, Directory directory) {
            this.indexReader = (IndexReader)Preconditions.checkNotNull((Object)indexReader);
            this.directory = directory;
        }

        @Override
        public IndexReader getReader() {
            return this.indexReader;
        }

        @Override
        public AnalyzingInfixSuggester getLookup() {
            return null;
        }

        @Override
        public Directory getSuggestDirectory() {
            return null;
        }

        @Override
        public long getIndexSize() throws IOException {
            return DirectoryUtils.dirSize(this.directory);
        }

        @Override
        public void close() throws IOException {
        }
    }
}

