/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.search;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.jackrabbit.guava.common.cache.Cache;
import org.apache.jackrabbit.guava.common.cache.CacheBuilder;
import org.apache.jackrabbit.guava.common.cache.Weigher;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.cache.CacheStats;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.commons.PathUtils;
import org.apache.jackrabbit.oak.plugins.index.fulltext.ExtractedText;
import org.apache.jackrabbit.oak.plugins.index.fulltext.PreExtractedTextProvider;
import org.apache.jackrabbit.oak.plugins.index.search.TextExtractionStatsMBean;
import org.apache.jackrabbit.oak.stats.StatisticsProvider;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExtractedTextCache {
    private static final boolean CACHE_ONLY_SUCCESS = Boolean.getBoolean("oak.extracted.cacheOnlySuccess");
    private static final int EXTRACTION_TIMEOUT_SECONDS = Integer.getInteger("oak.extraction.timeoutSeconds", 60);
    private static final int EXTRACTION_MAX_THREADS = Integer.getInteger("oak.extraction.maxThreads", 10);
    private static final boolean EXTRACT_IN_CALLER_THREAD = Boolean.getBoolean("oak.extraction.inCallerThread");
    private static final boolean EXTRACT_FORGET_TIMEOUT = Boolean.getBoolean("oak.extraction.forgetTimeout");
    private static final String TIMEOUT_MAP = "textExtractionTimeout.properties";
    private static final String EMPTY_STRING = "";
    private static final Logger log = LoggerFactory.getLogger(ExtractedTextCache.class);
    private volatile PreExtractedTextProvider extractedTextProvider;
    private int textExtractionCount;
    private long totalBytesRead;
    private long totalTextSize;
    private long totalTime;
    private int preFetchedCount;
    private final StatisticsProvider statisticsProvider;
    private final Cache<String, String> cache;
    private final ConcurrentHashMap<String, String> timeoutMap;
    private final File indexDir;
    private final CacheStats cacheStats;
    private final boolean alwaysUsePreExtractedCache;
    private volatile ExecutorService executorService;
    private volatile int timeoutCount;
    private long extractionTimeoutMillis = (long)EXTRACTION_TIMEOUT_SECONDS * 1000L;

    public ExtractedTextCache(long maxWeight, long expiryTimeInSecs) {
        this(maxWeight, expiryTimeInSecs, false, null);
    }

    public ExtractedTextCache(long maxWeight, long expiryTimeInSecs, boolean alwaysUsePreExtractedCache, File indexDir) {
        this(maxWeight, expiryTimeInSecs, alwaysUsePreExtractedCache, indexDir, null);
    }

    public ExtractedTextCache(long maxWeight, long expiryTimeInSecs, boolean alwaysUsePreExtractedCache, File indexDir, StatisticsProvider statisticsProvider) {
        if (maxWeight > 0L) {
            this.cache = CacheBuilder.newBuilder().weigher((Weigher)EmpiricalWeigher.INSTANCE).maximumWeight(maxWeight).expireAfterAccess(expiryTimeInSecs, TimeUnit.SECONDS).recordStats().build();
            this.cacheStats = new CacheStats(this.cache, "ExtractedTextCache", EmpiricalWeigher.INSTANCE, maxWeight);
        } else {
            this.cache = null;
            this.cacheStats = null;
        }
        this.alwaysUsePreExtractedCache = alwaysUsePreExtractedCache;
        this.timeoutMap = new ConcurrentHashMap();
        this.indexDir = indexDir;
        this.loadTimeoutMap();
        this.statisticsProvider = statisticsProvider;
    }

    @Nullable
    public String get(String nodePath, String propertyName, Blob blob, boolean reindexMode) {
        String result = null;
        String propertyPath = PathUtils.concat((String)nodePath, (String)propertyName);
        if (log.isTraceEnabled()) {
            log.trace("Looking for extracted text for [{}] with blobId [{}]", (Object)propertyPath, (Object)blob.getContentIdentity());
        }
        if ((reindexMode || this.alwaysUsePreExtractedCache) && this.extractedTextProvider != null) {
            try {
                ExtractedText text = this.extractedTextProvider.getText(propertyPath, blob);
                if (text != null) {
                    ++this.preFetchedCount;
                    result = ExtractedTextCache.getText(text);
                }
            }
            catch (IOException e) {
                log.warn("Error occurred while fetching pre extracted text for {}", (Object)propertyPath, (Object)e);
            }
        }
        String id = blob.getContentIdentity();
        if (this.cache != null && id != null && result == null) {
            result = (String)this.cache.getIfPresent((Object)id);
        }
        if (result == null && id != null) {
            result = this.timeoutMap.get(id);
        }
        return result;
    }

    public void put(@NotNull Blob blob, @NotNull ExtractedText extractedText) {
        String id = blob.getContentIdentity();
        if (!(this.cache == null || id == null || extractedText.getExtractionResult() != ExtractedText.ExtractionResult.SUCCESS && CACHE_ONLY_SUCCESS)) {
            this.cache.put((Object)id, (Object)ExtractedTextCache.getText(extractedText));
        }
    }

    public void putTimeout(@NotNull Blob blob, @NotNull ExtractedText extractedText) {
        if (EXTRACT_FORGET_TIMEOUT) {
            return;
        }
        String id = blob.getContentIdentity();
        if (id != null) {
            this.timeoutMap.put(id, ExtractedTextCache.getText(extractedText));
            this.storeTimeoutMap();
        }
    }

    private static String getText(ExtractedText text) {
        switch (text.getExtractionResult()) {
            case SUCCESS: {
                return text.getExtractedText().toString();
            }
            case ERROR: {
                return "TextExtractionError";
            }
            case EMPTY: {
                return EMPTY_STRING;
            }
        }
        throw new IllegalArgumentException();
    }

    public void addStats(int count, long timeInMillis, long bytesRead, long textLength) {
        this.textExtractionCount += count;
        this.totalTime += timeInMillis;
        this.totalBytesRead += bytesRead;
        this.totalTextSize += textLength;
    }

    public StatisticsProvider getStatisticsProvider() {
        return this.statisticsProvider;
    }

    public TextExtractionStatsMBean getStatsMBean() {
        return new TextExtractionStatsMBean(){

            @Override
            public boolean isPreExtractedTextProviderConfigured() {
                return ExtractedTextCache.this.extractedTextProvider != null;
            }

            @Override
            public int getTextExtractionCount() {
                return ExtractedTextCache.this.textExtractionCount;
            }

            @Override
            public long getTotalTime() {
                return ExtractedTextCache.this.totalTime;
            }

            @Override
            public int getPreFetchedCount() {
                return ExtractedTextCache.this.preFetchedCount;
            }

            @Override
            public String getExtractedTextSize() {
                return IOUtils.humanReadableByteCount((long)ExtractedTextCache.this.totalTextSize);
            }

            @Override
            public String getBytesRead() {
                return IOUtils.humanReadableByteCount((long)ExtractedTextCache.this.totalBytesRead);
            }

            @Override
            public boolean isAlwaysUsePreExtractedCache() {
                return ExtractedTextCache.this.alwaysUsePreExtractedCache;
            }

            @Override
            public int getTimeoutCount() {
                return ExtractedTextCache.this.timeoutCount;
            }
        };
    }

    @Nullable
    public CacheStats getCacheStats() {
        return this.cacheStats;
    }

    public void setExtractedTextProvider(PreExtractedTextProvider extractedTextProvider) {
        this.extractedTextProvider = extractedTextProvider;
    }

    public PreExtractedTextProvider getExtractedTextProvider() {
        return this.extractedTextProvider;
    }

    public void resetCache() {
        if (this.cache != null) {
            this.cache.invalidateAll();
        }
    }

    public boolean isAlwaysUsePreExtractedCache() {
        return this.alwaysUsePreExtractedCache;
    }

    public void close() {
        this.resetCache();
        this.closeExecutorService();
    }

    public void process(String name, Callable<Void> callable) throws Throwable {
        Callable<Void> callable2 = () -> {
            Thread t = Thread.currentThread();
            String oldThreadName = t.getName();
            t.setName(oldThreadName + ": " + name);
            try {
                Void void_ = (Void)callable.call();
                return void_;
            }
            finally {
                Thread.currentThread().setName(oldThreadName);
            }
        };
        try {
            if (EXTRACT_IN_CALLER_THREAD) {
                callable2.call();
            } else {
                Future<Void> future = this.getExecutor().submit(callable2);
                future.get(this.extractionTimeoutMillis, TimeUnit.MILLISECONDS);
            }
        }
        catch (TimeoutException e) {
            ++this.timeoutCount;
            throw e;
        }
        catch (ExecutionException e) {
            throw e.getCause();
        }
    }

    public void setExtractionTimeoutMillis(int extractionTimeoutMillis) {
        this.extractionTimeoutMillis = extractionTimeoutMillis;
    }

    private ExecutorService getExecutor() {
        if (this.executorService == null) {
            this.createExecutor();
        }
        return this.executorService;
    }

    private synchronized void createExecutor() {
        if (this.executorService != null) {
            return;
        }
        log.debug("ExtractedTextCache createExecutor {}", (Object)this);
        ThreadPoolExecutor executor = new ThreadPoolExecutor(1, EXTRACTION_MAX_THREADS, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), new ThreadFactory(){
            private final AtomicInteger counter = new AtomicInteger();
            private final Thread.UncaughtExceptionHandler handler = (t, e) -> log.warn("Error occurred in asynchronous processing ", e);

            @Override
            public Thread newThread(@NotNull Runnable r) {
                Thread thread = new Thread(r, this.createName());
                thread.setDaemon(true);
                thread.setPriority(1);
                thread.setUncaughtExceptionHandler(this.handler);
                return thread;
            }

            private String createName() {
                int index = this.counter.getAndIncrement();
                return "oak binary text extractor" + (String)(index == 0 ? ExtractedTextCache.EMPTY_STRING : " " + index);
            }
        });
        executor.setKeepAliveTime(1L, TimeUnit.MINUTES);
        executor.allowCoreThreadTimeOut(true);
        this.executorService = executor;
    }

    private synchronized void closeExecutorService() {
        if (this.executorService != null) {
            log.debug("ExtractedTextCache closeExecutorService {}", (Object)this);
            this.executorService.shutdown();
            try {
                this.executorService.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                log.warn("Interrupted", (Throwable)e);
            }
            this.executorService = null;
        }
    }

    private synchronized void loadTimeoutMap() {
        if (this.indexDir == null || !this.indexDir.exists()) {
            return;
        }
        File file = new File(this.indexDir, TIMEOUT_MAP);
        if (!file.exists()) {
            return;
        }
        try (FileInputStream in = new FileInputStream(file);){
            Properties prop = new Properties();
            prop.load(in);
            for (Map.Entry<Object, Object> e : prop.entrySet()) {
                this.timeoutMap.put(e.getKey().toString(), e.getValue().toString());
            }
        }
        catch (Exception e) {
            log.warn("Could not load timeout map {} from {}", new Object[]{TIMEOUT_MAP, this.indexDir, e});
        }
    }

    private synchronized void storeTimeoutMap() {
        if (this.indexDir == null || !this.indexDir.exists()) {
            return;
        }
        File file = new File(this.indexDir, TIMEOUT_MAP);
        try (FileOutputStream out = new FileOutputStream(file);){
            Properties prop = new Properties();
            prop.putAll((Map<?, ?>)this.timeoutMap);
            prop.store(out, "Text extraction timed out for the following binaries, and will not be retried");
        }
        catch (Exception e) {
            log.warn("Could not store timeout map {} from {}", new Object[]{TIMEOUT_MAP, this.indexDir, e});
        }
    }

    private static class EmpiricalWeigher
    implements Weigher<String, String> {
        public static final EmpiricalWeigher INSTANCE = new EmpiricalWeigher();

        private EmpiricalWeigher() {
        }

        private static long getMemory(@NotNull String s) {
            return 56L + (long)s.length() * 2L;
        }

        public int weigh(String key, String value) {
            long size = 168L;
            size += EmpiricalWeigher.getMemory(key);
            if ((size += EmpiricalWeigher.getMemory(value)) > Integer.MAX_VALUE) {
                log.debug("Calculated weight larger than Integer.MAX_VALUE: {}.", (Object)size);
                size = Integer.MAX_VALUE;
            }
            return (int)size;
        }
    }
}

