/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.index.indexer.document.flatfile;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.Stack;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.commons.Compression;
import org.apache.jackrabbit.oak.index.indexer.document.IndexerConfiguration;
import org.apache.jackrabbit.oak.index.indexer.document.NodeStateEntry;
import org.apache.jackrabbit.oak.index.indexer.document.flatfile.FlatFileStoreUtils;
import org.apache.jackrabbit.oak.index.indexer.document.flatfile.NodeStateEntryReader;
import org.apache.jackrabbit.oak.index.indexer.document.flatfile.SimpleNodeStateHolder;
import org.apache.jackrabbit.oak.index.indexer.document.indexstore.IndexStoreUtils;
import org.apache.jackrabbit.oak.plugins.index.search.IndexDefinition;
import org.apache.jackrabbit.oak.query.ast.NodeTypeInfo;
import org.apache.jackrabbit.oak.query.ast.NodeTypeInfoProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FlatFileSplitter {
    private static final Logger LOG = LoggerFactory.getLogger(FlatFileSplitter.class);
    private static final String SPLIT_DIR_NAME = "split";
    private final File workDir;
    private final NodeTypeInfoProvider infoProvider;
    private final File flatFile;
    private final NodeStateEntryReader entryReader;
    private final Compression algorithm;
    private final Set<IndexDefinition> indexDefinitions;
    private Set<String> splitNodeTypeNames;
    static Predicate<File> IS_SPLIT = path -> path.getParent().endsWith(SPLIT_DIR_NAME);

    public FlatFileSplitter(File flatFile, File workdir, NodeTypeInfoProvider infoProvider, NodeStateEntryReader entryReader, Set<IndexDefinition> indexDefinitions) {
        this.flatFile = flatFile;
        this.indexDefinitions = indexDefinitions;
        this.workDir = new File(workdir, SPLIT_DIR_NAME);
        this.infoProvider = infoProvider;
        this.entryReader = entryReader;
        this.algorithm = IndexStoreUtils.compressionAlgorithm();
    }

    private List<File> returnOriginalFlatFile() {
        return Collections.singletonList(this.flatFile);
    }

    public List<File> split() throws IOException {
        return this.split(true);
    }

    public List<File> split(boolean deleteOriginal) throws IOException {
        ArrayList<File> splitFlatFiles = new ArrayList<File>();
        try {
            FileUtils.forceMkdir((File)this.workDir);
        }
        catch (IOException e) {
            LOG.error("failed to create split directory {}", (Object)this.workDir.getAbsolutePath());
            return this.returnOriginalFlatFile();
        }
        long fileSizeInBytes = this.flatFile.length();
        long splitThreshold = Math.round((double)(fileSizeInBytes / IndexerConfiguration.splitSize()));
        LOG.info("original flat file size: ~{}", (Object)FileUtils.byteCountToDisplaySize((long)fileSizeInBytes));
        LOG.info("split threshold is ~{} bytes, estimate split size >={} files", (Object)FileUtils.byteCountToDisplaySize((long)splitThreshold), (Object)IndexerConfiguration.splitSize());
        if (splitThreshold < IndexerConfiguration.minSplitThreshold() || IndexerConfiguration.splitSize() <= 1L) {
            LOG.info("split is not necessary, skip splitting");
            return this.returnOriginalFlatFile();
        }
        Set<String> splitNodeTypesName = this.getSplitNodeTypeNames();
        LOG.info("unsafe split types: {}", splitNodeTypesName);
        if (splitNodeTypesName.contains("nt:base")) {
            LOG.info("Skipping split because split node types set contains {}", (Object)"nt:base");
            return this.returnOriginalFlatFile();
        }
        try (BufferedReader reader = FlatFileStoreUtils.createReader(this.flatFile, this.algorithm);){
            String line;
            long readPos = 0L;
            int outFileIndex = 1;
            File currentFile = new File(this.workDir, "split-" + outFileIndex + "-" + FlatFileStoreUtils.getSortedStoreFileName(this.algorithm));
            BufferedWriter writer = FlatFileStoreUtils.createWriter(currentFile, this.algorithm);
            splitFlatFiles.add(currentFile);
            int lineCount = 0;
            Stack<String> nodeTypeNameStack = new Stack<String>();
            while ((line = reader.readLine()) != null) {
                boolean shouldSplit;
                FlatFileSplitter.updateNodeTypeStack(nodeTypeNameStack, line, this.entryReader);
                boolean bl = shouldSplit = readPos > splitThreshold;
                if (shouldSplit && FlatFileSplitter.canSplit(splitNodeTypesName, nodeTypeNameStack)) {
                    writer.close();
                    LOG.info("created split flat file {} with size {}", (Object)currentFile.getAbsolutePath(), (Object)FileUtils.byteCountToDisplaySize((long)currentFile.length()));
                    readPos = 0L;
                    currentFile = new File(this.workDir, "split-" + ++outFileIndex + "-" + FlatFileStoreUtils.getSortedStoreFileName(this.algorithm));
                    writer = FlatFileStoreUtils.createWriter(currentFile, this.algorithm);
                    splitFlatFiles.add(currentFile);
                    LOG.info("split position found at line {}, creating new split file {}", (Object)lineCount, (Object)currentFile.getAbsolutePath());
                }
                writer.append(line);
                writer.newLine();
                readPos += (long)(line.length() + 1);
                ++lineCount;
            }
            writer.close();
            LOG.info("created split flat file {} with size {}", (Object)currentFile.getAbsolutePath(), (Object)FileUtils.byteCountToDisplaySize((long)currentFile.length()));
            LOG.info("split total line count: {}", (Object)lineCount);
        }
        if (deleteOriginal) {
            LOG.info("removing original flat file {} after splitting into {} files", (Object)this.flatFile.getAbsolutePath(), splitFlatFiles);
            this.flatFile.delete();
        }
        return splitFlatFiles;
    }

    private static void updateNodeTypeStack(Stack<String> parentNodeTypeNames, String line, NodeStateEntryReader entryReader) {
        int parentTypesDepth;
        SimpleNodeStateHolder ns = new SimpleNodeStateHolder(line);
        List<String> pathElements = ns.getPathElements();
        int currentLineDepth = pathElements.size();
        if (currentLineDepth > (parentTypesDepth = parentNodeTypeNames.size())) {
            parentNodeTypeNames.add(FlatFileSplitter.getJCRPrimaryType(line, entryReader));
        } else {
            int popSize = parentTypesDepth - currentLineDepth + 1;
            if (parentTypesDepth > 0) {
                for (int i = 0; i < popSize; ++i) {
                    parentNodeTypeNames.pop();
                }
            }
            parentNodeTypeNames.add(FlatFileSplitter.getJCRPrimaryType(line, entryReader));
        }
    }

    private static String getJCRPrimaryType(String line, NodeStateEntryReader entryReader) {
        NodeStateEntry nse = entryReader.read(line);
        PropertyState property = nse.getNodeState().getProperty("jcr:primaryType");
        if (property == null) {
            return "";
        }
        Type type = property.getType();
        if (type == Type.NAME) {
            return (String)property.getValue(Type.NAME);
        }
        return "";
    }

    private static boolean canSplit(Set<String> nodeTypes, Stack<String> nodeTypeNameStack) {
        if (nodeTypeNameStack.contains("")) {
            return false;
        }
        for (String parentNodeTypeName : nodeTypeNameStack.subList(0, nodeTypeNameStack.size() - 1)) {
            if (!nodeTypes.contains(parentNodeTypeName)) continue;
            return false;
        }
        return true;
    }

    private Set<NodeTypeInfo> getSubTypes(String nodeTypeName) {
        HashSet<NodeTypeInfo> initialSet = new HashSet<NodeTypeInfo>();
        NodeTypeInfo nodeType = this.infoProvider.getNodeTypeInfo(nodeTypeName);
        Set subTypes = nodeType.getMixinSubTypes();
        subTypes.addAll(nodeType.getPrimarySubTypes());
        for (String subTypeName : subTypes) {
            initialSet.add(this.infoProvider.getNodeTypeInfo(subTypeName));
            initialSet.addAll(this.getSubTypes(subTypeName));
        }
        return initialSet;
    }

    public Set<String> getSplitNodeTypeNames() {
        if (this.splitNodeTypeNames == null) {
            Set<NodeTypeInfo> splitNodeTypes = this.getSplitNodeType();
            this.splitNodeTypeNames = splitNodeTypes.stream().map(NodeTypeInfo::getNodeTypeName).collect(Collectors.toSet());
        }
        return this.splitNodeTypeNames;
    }

    private Set<NodeTypeInfo> getSplitNodeType() {
        HashSet nodeTypeNameSet = new HashSet();
        HashSet<NodeTypeInfo> setOfNodeType = new HashSet<NodeTypeInfo>();
        for (IndexDefinition indexDf : this.indexDefinitions) {
            Map aggregateMap = indexDf.getAggregates();
            nodeTypeNameSet.addAll(Objects.requireNonNull(aggregateMap).keySet());
            nodeTypeNameSet.addAll(indexDf.getDefinedRules().stream().map(IndexDefinition.IndexingRule::getBaseNodeType).collect(Collectors.toList()));
        }
        for (String nodeTypeName : nodeTypeNameSet) {
            setOfNodeType.add(this.infoProvider.getNodeTypeInfo(nodeTypeName));
            setOfNodeType.addAll(this.getSubTypes(nodeTypeName));
        }
        return setOfNodeType;
    }
}

