/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.azure;

import com.google.common.base.Preconditions;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.BlobListingDetails;
import com.microsoft.azure.storage.blob.CloudBlob;
import com.microsoft.azure.storage.blob.CloudBlobDirectory;
import com.microsoft.azure.storage.blob.CloudBlockBlob;
import com.microsoft.azure.storage.blob.CopyStatus;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.jackrabbit.oak.segment.azure.AzureSegmentArchiveReader;
import org.apache.jackrabbit.oak.segment.azure.AzureSegmentArchiveWriter;
import org.apache.jackrabbit.oak.segment.azure.AzureUtilities;
import org.apache.jackrabbit.oak.segment.spi.monitor.FileStoreMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.IOMonitor;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveManager;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveReader;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AzureArchiveManager
implements SegmentArchiveManager {
    private static final Logger log = LoggerFactory.getLogger(AzureSegmentArchiveReader.class);
    protected final CloudBlobDirectory cloudBlobDirectory;
    protected final IOMonitor ioMonitor;
    protected final FileStoreMonitor monitor;

    public AzureArchiveManager(CloudBlobDirectory cloudBlobDirectory, IOMonitor ioMonitor, FileStoreMonitor fileStoreMonitor) {
        this.cloudBlobDirectory = cloudBlobDirectory;
        this.ioMonitor = ioMonitor;
        this.monitor = fileStoreMonitor;
    }

    public List<String> listArchives() throws IOException {
        try {
            List<String> archiveNames = StreamSupport.stream(this.cloudBlobDirectory.listBlobs(null, false, EnumSet.noneOf(BlobListingDetails.class), null, null).spliterator(), false).filter(i -> i instanceof CloudBlobDirectory).map(i -> (CloudBlobDirectory)i).filter(i -> AzureUtilities.getName(i).endsWith(".tar")).map(CloudBlobDirectory::getPrefix).map(x$0 -> Paths.get(x$0, new String[0])).map(Path::getFileName).map(Path::toString).collect(Collectors.toList());
            Iterator it = archiveNames.iterator();
            while (it.hasNext()) {
                String archiveName = (String)it.next();
                if (!this.isArchiveEmpty(archiveName)) continue;
                this.delete(archiveName);
                it.remove();
            }
            return archiveNames;
        }
        catch (StorageException | URISyntaxException e) {
            throw new IOException(e);
        }
    }

    private boolean isArchiveEmpty(String archiveName) throws IOException, URISyntaxException, StorageException {
        return !this.getDirectory(archiveName).listBlobs("0000.").iterator().hasNext();
    }

    public SegmentArchiveReader open(String archiveName) throws IOException {
        try {
            CloudBlobDirectory archiveDirectory = this.getDirectory(archiveName);
            if (!archiveDirectory.getBlockBlobReference("closed").exists()) {
                throw new IOException("The archive " + archiveName + " hasn't been closed correctly.");
            }
            return new AzureSegmentArchiveReader(archiveDirectory, this.ioMonitor);
        }
        catch (StorageException | URISyntaxException e) {
            throw new IOException(e);
        }
    }

    public SegmentArchiveReader forceOpen(String archiveName) throws IOException {
        CloudBlobDirectory archiveDirectory = this.getDirectory(archiveName);
        return new AzureSegmentArchiveReader(archiveDirectory, this.ioMonitor);
    }

    public SegmentArchiveWriter create(String archiveName) throws IOException {
        return new AzureSegmentArchiveWriter(this.getDirectory(archiveName), this.ioMonitor, this.monitor);
    }

    public boolean delete(String archiveName) {
        try {
            this.getBlobs(archiveName).forEach(cloudBlob -> {
                try {
                    cloudBlob.delete();
                }
                catch (StorageException e) {
                    log.error("Can't delete segment {}", (Object)cloudBlob.getUri().getPath(), (Object)e);
                }
            });
            return true;
        }
        catch (IOException e) {
            log.error("Can't delete archive {}", (Object)archiveName, (Object)e);
            return false;
        }
    }

    public boolean renameTo(String from, String to) {
        try {
            CloudBlobDirectory targetDirectory = this.getDirectory(to);
            this.getBlobs(from).forEach(cloudBlob -> {
                try {
                    this.renameBlob((CloudBlob)cloudBlob, targetDirectory);
                }
                catch (IOException e) {
                    log.error("Can't rename segment {}", (Object)cloudBlob.getUri().getPath(), (Object)e);
                }
            });
            return true;
        }
        catch (IOException e) {
            log.error("Can't rename archive {} to {}", new Object[]{from, to, e});
            return false;
        }
    }

    public void copyFile(String from, String to) throws IOException {
        CloudBlobDirectory targetDirectory = this.getDirectory(to);
        this.getBlobs(from).forEach(cloudBlob -> {
            try {
                this.copyBlob((CloudBlob)cloudBlob, targetDirectory);
            }
            catch (IOException e) {
                log.error("Can't copy segment {}", (Object)cloudBlob.getUri().getPath(), (Object)e);
            }
        });
    }

    public boolean exists(String archiveName) {
        try {
            return this.getDirectory(archiveName).listBlobsSegmented(null, false, null, 1, null, null, null).getLength() > 0;
        }
        catch (StorageException | IOException | URISyntaxException e) {
            log.error("Can't check the existence of {}", (Object)archiveName, (Object)e);
            return false;
        }
    }

    public void recoverEntries(String archiveName, LinkedHashMap<UUID, byte[]> entries) throws IOException {
        Pattern pattern = Pattern.compile(AzureUtilities.SEGMENT_FILE_NAME_PATTERN);
        ArrayList<RecoveredEntry> entryList = new ArrayList<RecoveredEntry>();
        for (CloudBlob b : this.getBlobs(archiveName)) {
            String name = AzureUtilities.getName(b);
            Matcher m = pattern.matcher(name);
            if (!m.matches()) continue;
            int position = Integer.parseInt(m.group(1), 16);
            UUID uuid = UUID.fromString(m.group(2));
            long length = b.getProperties().getLength();
            if (length <= 0L) continue;
            byte[] data = new byte[(int)length];
            try {
                b.downloadToByteArray(data, 0);
            }
            catch (StorageException e) {
                throw new IOException(e);
            }
            entryList.add(new RecoveredEntry(position, uuid, data, name));
        }
        Collections.sort(entryList);
        int i = 0;
        for (RecoveredEntry e : entryList) {
            if (e.position != i) {
                log.warn("Missing entry {}.??? when recovering {}. No more segments will be read.", (Object)String.format("%04X", i), (Object)archiveName);
                break;
            }
            log.info("Recovering segment {}/{}", (Object)archiveName, (Object)e.fileName);
            entries.put(e.uuid, e.data);
            ++i;
        }
    }

    protected CloudBlobDirectory getDirectory(String archiveName) throws IOException {
        try {
            return this.cloudBlobDirectory.getDirectoryReference(archiveName);
        }
        catch (URISyntaxException e) {
            throw new IOException(e);
        }
    }

    private List<CloudBlob> getBlobs(String archiveName) throws IOException {
        return AzureUtilities.getBlobs(this.getDirectory(archiveName));
    }

    private void renameBlob(CloudBlob blob, CloudBlobDirectory newParent) throws IOException {
        this.copyBlob(blob, newParent);
        try {
            blob.delete();
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
    }

    private void copyBlob(CloudBlob blob, CloudBlobDirectory newParent) throws IOException {
        Preconditions.checkArgument((boolean)(blob instanceof CloudBlockBlob), (Object)"Only page blobs are supported for the rename");
        try {
            String blobName = AzureUtilities.getName(blob);
            CloudBlockBlob newBlob = newParent.getBlockBlobReference(blobName);
            newBlob.startCopy(blob.getUri());
            boolean isStatusPending = true;
            while (isStatusPending) {
                newBlob.downloadAttributes();
                if (newBlob.getCopyState().getStatus() == CopyStatus.PENDING) {
                    Thread.sleep(100L);
                    continue;
                }
                isStatusPending = false;
            }
            CopyStatus finalStatus = newBlob.getCopyState().getStatus();
            if (newBlob.getCopyState().getStatus() != CopyStatus.SUCCESS) {
                throw new IOException("Invalid copy status for " + blob.getUri().getPath() + ": " + (Object)((Object)finalStatus));
            }
        }
        catch (StorageException | InterruptedException | URISyntaxException e) {
            throw new IOException(e);
        }
    }

    private static class RecoveredEntry
    implements Comparable<RecoveredEntry> {
        private final byte[] data;
        private final UUID uuid;
        private final int position;
        private final String fileName;

        public RecoveredEntry(int position, UUID uuid, byte[] data, String fileName) {
            this.data = data;
            this.uuid = uuid;
            this.position = position;
            this.fileName = fileName;
        }

        @Override
        public int compareTo(RecoveredEntry o) {
            return Integer.compare(this.position, o.position);
        }
    }
}

