/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.azure;

import com.microsoft.azure.storage.AccessCondition;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.CloudBlockBlob;
import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.jackrabbit.oak.segment.spi.persistence.RepositoryLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AzureRepositoryLock
implements RepositoryLock {
    private static final Logger log = LoggerFactory.getLogger(AzureRepositoryLock.class);
    private static final int TIMEOUT_SEC = Integer.getInteger("oak.segment.azure.lock.timeout", 0);
    private static int INTERVAL = 60;
    private final Runnable shutdownHook;
    private final CloudBlockBlob blob;
    private final ExecutorService executor;
    private final int timeoutSec;
    private String leaseId;
    private volatile boolean doUpdate;

    public AzureRepositoryLock(CloudBlockBlob blob, Runnable shutdownHook) {
        this(blob, shutdownHook, TIMEOUT_SEC);
    }

    public AzureRepositoryLock(CloudBlockBlob blob, Runnable shutdownHook, int timeoutSec) {
        this.shutdownHook = shutdownHook;
        this.blob = blob;
        this.executor = Executors.newSingleThreadExecutor();
        this.timeoutSec = timeoutSec;
    }

    public AzureRepositoryLock lock() throws IOException {
        long start = System.currentTimeMillis();
        Exception ex = null;
        do {
            try {
                this.blob.openOutputStream().close();
                this.leaseId = this.blob.acquireLease(INTERVAL, null);
                log.info("Acquired lease {}", (Object)this.leaseId);
            }
            catch (StorageException | IOException e) {
                if (ex == null) {
                    log.info("Can't acquire the lease. Retrying every 1s. Timeout is set to {}s.", (Object)this.timeoutSec);
                }
                ex = e;
                if ((System.currentTimeMillis() - start) / 1000L >= (long)this.timeoutSec) break;
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException e1) {
                    throw new IOException(e1);
                }
            }
        } while (this.leaseId == null);
        if (this.leaseId == null) {
            log.error("Can't acquire the lease in {}s.", (Object)this.timeoutSec);
            throw new IOException(ex);
        }
        this.executor.submit(this::refreshLease);
        return this;
    }

    private void refreshLease() {
        this.doUpdate = true;
        long lastUpdate = 0L;
        while (this.doUpdate) {
            try {
                long timeSinceLastUpdate = (System.currentTimeMillis() - lastUpdate) / 1000L;
                if (timeSinceLastUpdate > (long)(INTERVAL / 2)) {
                    this.blob.renewLease(AccessCondition.generateLeaseCondition(this.leaseId));
                    lastUpdate = System.currentTimeMillis();
                }
            }
            catch (StorageException e) {
                log.error("Can't renew the lease", (Throwable)e);
                this.shutdownHook.run();
                this.doUpdate = false;
                return;
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                log.error("Interrupted the lease renewal loop", (Throwable)e);
            }
        }
    }

    public void unlock() throws IOException {
        this.doUpdate = false;
        this.executor.shutdown();
        try {
            this.executor.awaitTermination(1L, TimeUnit.MINUTES);
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
        finally {
            this.releaseLease();
        }
    }

    private void releaseLease() throws IOException {
        try {
            this.blob.releaseLease(AccessCondition.generateLeaseCondition(this.leaseId));
            this.blob.delete();
            log.info("Released lease {}", (Object)this.leaseId);
            this.leaseId = null;
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
    }
}

