/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.azure;

import com.google.common.base.Stopwatch;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.CloudBlobDirectory;
import com.microsoft.azure.storage.blob.CloudBlockBlob;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.apache.jackrabbit.oak.commons.Buffer;
import org.apache.jackrabbit.oak.segment.azure.AzureBlobMetadata;
import org.apache.jackrabbit.oak.segment.azure.AzureSegmentArchiveEntry;
import org.apache.jackrabbit.oak.segment.azure.AzureSegmentArchiveReader;
import org.apache.jackrabbit.oak.segment.azure.AzureUtilities;
import org.apache.jackrabbit.oak.segment.azure.queue.SegmentWriteAction;
import org.apache.jackrabbit.oak.segment.azure.queue.SegmentWriteQueue;
import org.apache.jackrabbit.oak.segment.spi.monitor.FileStoreMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.IOMonitor;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveWriter;

public class AzureSegmentArchiveWriter
implements SegmentArchiveWriter {
    private final CloudBlobDirectory archiveDirectory;
    private final IOMonitor ioMonitor;
    private final FileStoreMonitor monitor;
    private final Optional<SegmentWriteQueue> queue;
    private Map<UUID, AzureSegmentArchiveEntry> index = Collections.synchronizedMap(new LinkedHashMap());
    private int entries;
    private long totalLength;
    private volatile boolean created = false;

    public AzureSegmentArchiveWriter(CloudBlobDirectory archiveDirectory, IOMonitor ioMonitor, FileStoreMonitor monitor) {
        this.archiveDirectory = archiveDirectory;
        this.ioMonitor = ioMonitor;
        this.monitor = monitor;
        this.queue = SegmentWriteQueue.THREADS > 0 ? Optional.of(new SegmentWriteQueue(this::doWriteEntry)) : Optional.empty();
    }

    public void writeSegment(long msb, long lsb, byte[] data, int offset, int size, int generation, int fullGeneration, boolean compacted) throws IOException {
        this.created = true;
        AzureSegmentArchiveEntry entry = new AzureSegmentArchiveEntry(msb, lsb, this.entries++, size, generation, fullGeneration, compacted);
        if (this.queue.isPresent()) {
            this.queue.get().addToQueue(entry, data, offset, size);
        } else {
            this.doWriteEntry(entry, data, offset, size);
        }
        this.index.put(new UUID(msb, lsb), entry);
        this.totalLength += (long)size;
        this.monitor.written((long)size);
    }

    private void doWriteEntry(AzureSegmentArchiveEntry indexEntry, byte[] data, int offset, int size) throws IOException {
        long msb = indexEntry.getMsb();
        long lsb = indexEntry.getLsb();
        String segmentName = AzureUtilities.getSegmentFileName(indexEntry);
        CloudBlockBlob blob = this.getBlob(segmentName);
        this.ioMonitor.beforeSegmentWrite(new File(blob.getName()), msb, lsb, size);
        Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            blob.setMetadata(AzureBlobMetadata.toSegmentMetadata(indexEntry));
            blob.uploadFromByteArray(data, offset, size);
            blob.uploadMetadata();
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
        this.ioMonitor.afterSegmentWrite(new File(blob.getName()), msb, lsb, size, stopwatch.elapsed(TimeUnit.NANOSECONDS));
    }

    public Buffer readSegment(long msb, long lsb) throws IOException {
        UUID uuid = new UUID(msb, lsb);
        Optional<SegmentWriteAction> segment = this.queue.map(q -> q.read(uuid));
        if (segment.isPresent()) {
            return segment.get().toBuffer();
        }
        AzureSegmentArchiveEntry indexEntry = this.index.get(new UUID(msb, lsb));
        if (indexEntry == null) {
            return null;
        }
        Buffer buffer = AzureSegmentArchiveReader.OFF_HEAP ? Buffer.allocateDirect((int)indexEntry.getLength()) : Buffer.allocate((int)indexEntry.getLength());
        AzureUtilities.readBufferFully(this.getBlob(AzureUtilities.getSegmentFileName(indexEntry)), buffer);
        return buffer;
    }

    public boolean containsSegment(long msb, long lsb) {
        UUID uuid = new UUID(msb, lsb);
        Optional<SegmentWriteAction> segment = this.queue.map(q -> q.read(uuid));
        if (segment.isPresent()) {
            return true;
        }
        return this.index.containsKey(new UUID(msb, lsb));
    }

    public void writeGraph(byte[] data) throws IOException {
        this.writeDataFile(data, ".gph");
    }

    public void writeBinaryReferences(byte[] data) throws IOException {
        this.writeDataFile(data, ".brf");
    }

    private void writeDataFile(byte[] data, String extension) throws IOException {
        try {
            this.getBlob(this.getName() + extension).uploadFromByteArray(data, 0, data.length);
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
        this.totalLength += (long)data.length;
        this.monitor.written((long)data.length);
    }

    public long getLength() {
        return this.totalLength;
    }

    public int getEntryCount() {
        return this.index.size();
    }

    public void close() throws IOException {
        if (this.queue.isPresent()) {
            SegmentWriteQueue q = this.queue.get();
            q.flush();
            q.close();
        }
        try {
            this.getBlob("closed").uploadFromByteArray(new byte[0], 0, 0);
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
    }

    public boolean isCreated() {
        return this.created || !this.queueIsEmpty();
    }

    public void flush() throws IOException {
        if (this.queue.isPresent()) {
            this.queue.get().flush();
        }
    }

    private boolean queueIsEmpty() {
        return this.queue.map(SegmentWriteQueue::isEmpty).orElse(true);
    }

    public String getName() {
        return AzureUtilities.getName(this.archiveDirectory);
    }

    private CloudBlockBlob getBlob(String name) throws IOException {
        try {
            return this.archiveDirectory.getBlockBlobReference(name);
        }
        catch (StorageException | URISyntaxException e) {
            throw new IOException(e);
        }
    }
}

