/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.azure.tool;

import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.CloudBlobDirectory;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.apache.jackrabbit.oak.commons.Buffer;
import org.apache.jackrabbit.oak.segment.azure.AzurePersistence;
import org.apache.jackrabbit.oak.segment.azure.tool.ToolUtils;
import org.apache.jackrabbit.oak.segment.file.tar.TarPersistence;
import org.apache.jackrabbit.oak.segment.spi.RepositoryNotReachableException;
import org.apache.jackrabbit.oak.segment.spi.monitor.FileStoreMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.FileStoreMonitorAdapter;
import org.apache.jackrabbit.oak.segment.spi.monitor.IOMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.IOMonitorAdapter;
import org.apache.jackrabbit.oak.segment.spi.monitor.RemoteStoreMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.RemoteStoreMonitorAdapter;
import org.apache.jackrabbit.oak.segment.spi.persistence.GCJournalFile;
import org.apache.jackrabbit.oak.segment.spi.persistence.JournalFileReader;
import org.apache.jackrabbit.oak.segment.spi.persistence.JournalFileWriter;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveEntry;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveManager;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveReader;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveWriter;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentNodeStorePersistence;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentStoreMigrator
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(SegmentStoreMigrator.class);
    private static final int READ_THREADS = 20;
    private final SegmentNodeStorePersistence source;
    private final SegmentNodeStorePersistence target;
    private final String sourceName;
    private final String targetName;
    private final boolean appendMode;
    private final Integer revisionCount;
    private ExecutorService executor = Executors.newFixedThreadPool(21);

    private SegmentStoreMigrator(Builder builder) {
        this.source = builder.source;
        this.target = builder.target;
        this.sourceName = builder.sourceName;
        this.targetName = builder.targetName;
        this.appendMode = builder.appendMode;
        this.revisionCount = builder.revisionCount;
    }

    public void migrate() throws IOException, ExecutionException, InterruptedException {
        SegmentStoreMigrator.runWithRetry(() -> this.migrateJournal(), 16, 5);
        SegmentStoreMigrator.runWithRetry(() -> this.migrateGCJournal(), 16, 5);
        SegmentStoreMigrator.runWithRetry(() -> this.migrateManifest(), 16, 5);
        this.migrateArchives();
    }

    private Void migrateJournal() throws IOException {
        log.info("{}/journal.log -> {}", (Object)this.sourceName, (Object)this.targetName);
        if (!this.source.getJournalFile().exists()) {
            log.info("No journal at {}; skipping.", (Object)this.sourceName);
            return null;
        }
        ArrayList<String> journal = new ArrayList<String>();
        try (JournalFileReader reader = this.source.getJournalFile().openJournalReader();){
            String line;
            while ((line = reader.readLine()) != null) {
                if (line.length() > 0 && !line.trim().equals("")) {
                    journal.add(line);
                }
                if (journal.size() != this.revisionCount.intValue()) continue;
                break;
            }
        }
        Collections.reverse(journal);
        var3_3 = null;
        try (JournalFileWriter writer = this.target.getJournalFile().openJournalWriter();){
            writer.truncate();
            for (String line : journal) {
                writer.writeLine(line);
            }
        }
        catch (Throwable throwable) {
            var3_3 = throwable;
            throw throwable;
        }
        return null;
    }

    private Void migrateGCJournal() throws IOException {
        log.info("{}/gc.log -> {}", (Object)this.sourceName, (Object)this.targetName);
        GCJournalFile targetGCJournal = this.target.getGCJournalFile();
        if (this.appendMode) {
            targetGCJournal.truncate();
        }
        for (String line : this.source.getGCJournalFile().readLines()) {
            targetGCJournal.writeLine(line);
        }
        return null;
    }

    private Void migrateManifest() throws IOException {
        log.info("{}/manifest -> {}", (Object)this.sourceName, (Object)this.targetName);
        if (!this.source.getManifestFile().exists()) {
            log.info("No manifest at {}; skipping.", (Object)this.sourceName);
            return null;
        }
        Properties manifest = this.source.getManifestFile().load();
        this.target.getManifestFile().save(manifest);
        return null;
    }

    private void migrateArchives() throws IOException, ExecutionException, InterruptedException {
        if (!this.source.segmentFilesExist()) {
            log.info("No segment archives at {}; skipping.", (Object)this.sourceName);
            return;
        }
        SegmentArchiveManager sourceManager = this.source.createArchiveManager(false, false, (IOMonitor)new IOMonitorAdapter(), (FileStoreMonitor)new FileStoreMonitorAdapter(), (RemoteStoreMonitor)new RemoteStoreMonitorAdapter());
        SegmentArchiveManager targetManager = this.target.createArchiveManager(false, false, (IOMonitor)new IOMonitorAdapter(), (FileStoreMonitor)new FileStoreMonitorAdapter(), (RemoteStoreMonitor)new RemoteStoreMonitorAdapter());
        List targetArchives = targetManager.listArchives();
        for (String archiveName : sourceManager.listArchives()) {
            log.info("{}/{} -> {}", new Object[]{this.sourceName, archiveName, this.targetName});
            if (this.appendMode && targetArchives.contains(archiveName)) {
                log.info("Already exists, skipping.");
                continue;
            }
            SegmentArchiveReader reader = sourceManager.forceOpen(archiveName);
            Throwable throwable = null;
            try (SegmentArchiveWriter writer = targetManager.create(archiveName);){
                this.migrateSegments(reader, writer);
                this.migrateBinaryRef(reader, writer);
                this.migrateGraph(reader, writer);
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (reader == null) continue;
                if (throwable != null) {
                    try {
                        reader.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                reader.close();
            }
        }
    }

    private void migrateSegments(SegmentArchiveReader reader, SegmentArchiveWriter writer) throws ExecutionException, InterruptedException, IOException {
        ArrayList<Future<Segment>> futures = new ArrayList<Future<Segment>>();
        for (SegmentArchiveEntry segmentArchiveEntry : reader.listSegments()) {
            futures.add(this.executor.submit(() -> SegmentStoreMigrator.runWithRetry(() -> {
                Segment segment = new Segment(entry);
                segment.read(reader);
                return segment;
            }, 16, 5)));
        }
        for (Future future : futures) {
            Segment segment = (Segment)future.get();
            segment.write(writer);
        }
    }

    private void migrateBinaryRef(SegmentArchiveReader reader, SegmentArchiveWriter writer) throws IOException, ExecutionException, InterruptedException {
        Future<Buffer> future = this.executor.submit(() -> SegmentStoreMigrator.runWithRetry(() -> reader.getBinaryReferences(), 16, 5));
        Buffer binaryReferences = future.get();
        if (binaryReferences != null) {
            byte[] array = ToolUtils.fetchByteArray(binaryReferences);
            writer.writeBinaryReferences(array);
        }
    }

    private void migrateGraph(SegmentArchiveReader reader, SegmentArchiveWriter writer) throws IOException, ExecutionException, InterruptedException {
        Future<Buffer> future = this.executor.submit(() -> SegmentStoreMigrator.runWithRetry(() -> {
            if (reader.hasGraph()) {
                return reader.getGraph();
            }
            return null;
        }, 16, 5));
        Buffer graph = future.get();
        if (graph != null) {
            byte[] array = ToolUtils.fetchByteArray(graph);
            writer.writeGraph(array);
        }
    }

    private static <T> T runWithRetry(Producer<T> producer, int maxAttempts, int intervalSec) throws IOException {
        IOException ioException = null;
        RepositoryNotReachableException repoNotReachableException = null;
        for (int i = 0; i < maxAttempts; ++i) {
            try {
                return producer.produce();
            }
            catch (IOException e) {
                log.error("Can't execute the operation. Retrying (attempt {})", (Object)i, (Object)e);
                ioException = e;
            }
            catch (RepositoryNotReachableException e) {
                log.error("Can't execute the operation. Retrying (attempt {})", (Object)i, (Object)e);
                repoNotReachableException = e;
            }
            try {
                Thread.sleep(intervalSec * 1000);
                continue;
            }
            catch (InterruptedException e) {
                log.error("Interrupted", (Throwable)e);
            }
        }
        if (ioException != null) {
            throw ioException;
        }
        if (repoNotReachableException != null) {
            throw repoNotReachableException;
        }
        throw new IllegalStateException();
    }

    @Override
    public void close() throws IOException {
        this.executor.shutdown();
        try {
            while (!this.executor.awaitTermination(100L, TimeUnit.MILLISECONDS)) {
            }
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
    }

    public static class Builder {
        private SegmentNodeStorePersistence source;
        private SegmentNodeStorePersistence target;
        private String sourceName;
        private String targetName;
        private boolean appendMode;
        private Integer revisionCount = Integer.MAX_VALUE;

        public Builder withSource(File dir) {
            this.source = new TarPersistence(dir);
            this.sourceName = ToolUtils.storeDescription(ToolUtils.SegmentStoreType.TAR, dir.getPath());
            return this;
        }

        public Builder withSource(CloudBlobDirectory dir) throws URISyntaxException, StorageException {
            this.source = new AzurePersistence(dir);
            this.sourceName = ToolUtils.storeDescription(ToolUtils.SegmentStoreType.AZURE, dir.getContainer().getName() + "/" + dir.getPrefix());
            return this;
        }

        public Builder withSourcePersistence(SegmentNodeStorePersistence source, String sourceName) {
            this.source = source;
            this.sourceName = sourceName;
            return this;
        }

        public Builder withTargetPersistence(SegmentNodeStorePersistence target, String targetName) {
            this.target = target;
            this.targetName = targetName;
            return this;
        }

        public Builder withTarget(File dir) {
            this.target = new TarPersistence(dir);
            this.targetName = ToolUtils.storeDescription(ToolUtils.SegmentStoreType.TAR, dir.getPath());
            return this;
        }

        public Builder withTarget(CloudBlobDirectory dir) throws URISyntaxException, StorageException {
            this.target = new AzurePersistence(dir);
            this.targetName = ToolUtils.storeDescription(ToolUtils.SegmentStoreType.AZURE, dir.getContainer().getName() + "/" + dir.getPrefix());
            return this;
        }

        public Builder setAppendMode() {
            this.appendMode = true;
            return this;
        }

        public Builder withRevisionCount(Integer revisionCount) {
            this.revisionCount = revisionCount;
            return this;
        }

        public SegmentStoreMigrator build() {
            return new SegmentStoreMigrator(this);
        }
    }

    private static class Segment {
        private final SegmentArchiveEntry entry;
        private volatile Buffer data;

        private Segment(SegmentArchiveEntry entry) {
            this.entry = entry;
        }

        private void read(SegmentArchiveReader reader) throws IOException {
            this.data = reader.readSegment(this.entry.getMsb(), this.entry.getLsb());
        }

        private void write(SegmentArchiveWriter writer) throws IOException {
            byte[] array = this.data.array();
            boolean offset = false;
            writer.writeSegment(this.entry.getMsb(), this.entry.getLsb(), array, 0, this.entry.getLength(), this.entry.getGeneration(), this.entry.getFullGeneration(), this.entry.isCompacted());
        }

        public String toString() {
            return new UUID(this.entry.getMsb(), this.entry.getLsb()).toString();
        }
    }

    @FunctionalInterface
    private static interface Producer<T> {
        public T produce() throws IOException;
    }
}

