/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.azure;

import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.CloudAppendBlob;
import com.microsoft.azure.storage.blob.CloudBlob;
import com.microsoft.azure.storage.blob.CloudBlobDirectory;
import com.microsoft.azure.storage.blob.ListBlobItem;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.jackrabbit.guava.common.collect.ImmutableList;
import org.apache.jackrabbit.guava.common.collect.Lists;
import org.apache.jackrabbit.oak.segment.azure.AzureUtilities;
import org.apache.jackrabbit.oak.segment.azure.ReverseFileReader;
import org.apache.jackrabbit.oak.segment.azure.util.CaseInsensitiveKeysMapAccess;
import org.apache.jackrabbit.oak.segment.remote.WriteAccessController;
import org.apache.jackrabbit.oak.segment.spi.persistence.JournalFile;
import org.apache.jackrabbit.oak.segment.spi.persistence.JournalFileReader;
import org.apache.jackrabbit.oak.segment.spi.persistence.JournalFileWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AzureJournalFile
implements JournalFile {
    private static final Logger log = LoggerFactory.getLogger(AzureJournalFile.class);
    private static final int JOURNAL_LINE_LIMIT = Integer.getInteger("org.apache.jackrabbit.oak.segment.azure.journal.lines", 40000);
    private final CloudBlobDirectory directory;
    private final String journalNamePrefix;
    private final int lineLimit;
    private final WriteAccessController writeAccessController;

    AzureJournalFile(CloudBlobDirectory directory, String journalNamePrefix, WriteAccessController writeAccessController, int lineLimit) {
        this.directory = directory;
        this.journalNamePrefix = journalNamePrefix;
        this.lineLimit = lineLimit;
        this.writeAccessController = writeAccessController;
    }

    public AzureJournalFile(CloudBlobDirectory directory, String journalNamePrefix, WriteAccessController writeAccessController) {
        this(directory, journalNamePrefix, writeAccessController, JOURNAL_LINE_LIMIT);
    }

    public JournalFileReader openJournalReader() throws IOException {
        return new CombinedReader(this.getJournalBlobs());
    }

    public JournalFileWriter openJournalWriter() throws IOException {
        return new AzureJournalWriter();
    }

    public String getName() {
        return this.journalNamePrefix;
    }

    public boolean exists() {
        try {
            return !this.getJournalBlobs().isEmpty();
        }
        catch (IOException e) {
            log.error("Can't check if the file exists", (Throwable)e);
            return false;
        }
    }

    private String getJournalFileName(int index) {
        return String.format("%s.%03d", this.journalNamePrefix, index);
    }

    private List<CloudAppendBlob> getJournalBlobs() throws IOException {
        try {
            ArrayList<CloudAppendBlob> result = new ArrayList<CloudAppendBlob>();
            for (ListBlobItem b : this.directory.listBlobs(this.journalNamePrefix)) {
                if (b instanceof CloudAppendBlob) {
                    result.add((CloudAppendBlob)b);
                    continue;
                }
                log.warn("Invalid blob type: {} {}", (Object)b.getUri(), b.getClass());
            }
            result.sort(Comparator.comparing(AzureUtilities::getName).reversed());
            return result;
        }
        catch (StorageException | URISyntaxException e) {
            throw new IOException(e);
        }
    }

    private static class CombinedReader
    implements JournalFileReader {
        private final Iterator<AzureJournalReader> readers;
        private JournalFileReader currentReader;

        private CombinedReader(List<CloudAppendBlob> blobs) {
            this.readers = blobs.stream().map(x$0 -> new AzureJournalReader((CloudBlob)x$0)).iterator();
        }

        public String readLine() throws IOException {
            String line;
            do {
                if (this.currentReader == null) {
                    if (!this.readers.hasNext()) {
                        return null;
                    }
                    this.currentReader = this.readers.next();
                }
                while ("".equals(line = this.currentReader.readLine())) {
                }
                if (line != null) continue;
                this.currentReader.close();
                this.currentReader = null;
            } while (line == null);
            return line;
        }

        public void close() throws IOException {
            while (this.readers.hasNext()) {
                this.readers.next().close();
            }
            if (this.currentReader != null) {
                this.currentReader.close();
                this.currentReader = null;
            }
        }
    }

    private class AzureJournalWriter
    implements JournalFileWriter {
        private CloudAppendBlob currentBlob;
        private int lineCount;

        public AzureJournalWriter() throws IOException {
            List<CloudAppendBlob> blobs = AzureJournalFile.this.getJournalBlobs();
            if (blobs.isEmpty()) {
                try {
                    this.currentBlob = AzureJournalFile.this.directory.getAppendBlobReference(AzureJournalFile.this.getJournalFileName(1));
                    this.currentBlob.createOrReplace();
                    this.currentBlob.downloadAttributes();
                }
                catch (StorageException | URISyntaxException e) {
                    throw new IOException(e);
                }
            } else {
                this.currentBlob = blobs.get(0);
            }
            try {
                this.currentBlob.downloadAttributes();
            }
            catch (StorageException e) {
                throw new IOException(e);
            }
            String lc = this.currentBlob.getMetadata().get("lineCount");
            this.lineCount = lc == null ? 0 : Integer.parseInt(lc);
        }

        public void truncate() throws IOException {
            try {
                AzureJournalFile.this.writeAccessController.checkWritingAllowed();
                for (CloudAppendBlob cloudAppendBlob : AzureJournalFile.this.getJournalBlobs()) {
                    cloudAppendBlob.delete();
                }
                this.createNextFile(0);
            }
            catch (StorageException e) {
                throw new IOException(e);
            }
        }

        public void writeLine(String line) throws IOException {
            this.batchWriteLines((List<String>)ImmutableList.of((Object)line));
        }

        public void batchWriteLines(List<String> lines) throws IOException {
            AzureJournalFile.this.writeAccessController.checkWritingAllowed();
            if (lines.isEmpty()) {
                return;
            }
            int firstBlockSize = Math.min(AzureJournalFile.this.lineLimit - this.lineCount, lines.size());
            List<String> firstBlock = lines.subList(0, firstBlockSize);
            List remainingBlocks = Lists.partition(lines.subList(firstBlockSize, lines.size()), (int)AzureJournalFile.this.lineLimit);
            ImmutableList allBlocks = ImmutableList.builder().addAll((Iterable)(firstBlock.isEmpty() ? ImmutableList.of() : ImmutableList.of(firstBlock))).addAll((Iterable)remainingBlocks).build();
            for (List entries : allBlocks) {
                if (this.lineCount >= AzureJournalFile.this.lineLimit) {
                    int parsedSuffix = this.parseCurrentSuffix();
                    this.createNextFile(parsedSuffix);
                }
                StringBuilder text = new StringBuilder();
                for (String line : entries) {
                    text.append(line).append("\n");
                }
                try {
                    this.currentBlob.appendText(text.toString());
                    this.currentBlob.getMetadata().put("lastEntry", (String)entries.get(entries.size() - 1));
                    this.lineCount += entries.size();
                    this.currentBlob.getMetadata().put("lineCount", Integer.toString(this.lineCount));
                    this.currentBlob.uploadMetadata();
                }
                catch (StorageException e) {
                    throw new IOException(e);
                }
            }
        }

        private void createNextFile(int suffix) throws IOException {
            try {
                this.currentBlob = AzureJournalFile.this.directory.getAppendBlobReference(AzureJournalFile.this.getJournalFileName(suffix + 1));
                this.currentBlob.createOrReplace();
                this.lineCount = 0;
            }
            catch (StorageException | URISyntaxException e) {
                throw new IOException(e);
            }
        }

        private int parseCurrentSuffix() {
            int parsedSuffix;
            String name = AzureUtilities.getName(this.currentBlob);
            Pattern pattern = Pattern.compile(Pattern.quote(AzureJournalFile.this.journalNamePrefix) + "\\.(\\d+)");
            Matcher matcher = pattern.matcher(name);
            if (matcher.find()) {
                String suffix = matcher.group(1);
                try {
                    parsedSuffix = Integer.parseInt(suffix);
                }
                catch (NumberFormatException e) {
                    log.warn("Can't parse suffix for journal file {}", (Object)name);
                    parsedSuffix = 0;
                }
            } else {
                log.warn("Can't parse journal file name {}", (Object)name);
                parsedSuffix = 0;
            }
            return parsedSuffix;
        }

        public void close() throws IOException {
        }
    }

    private static class AzureJournalReader
    implements JournalFileReader {
        private final CloudBlob blob;
        private ReverseFileReader reader;
        private boolean metadataFetched;
        private boolean firstLineReturned;

        private AzureJournalReader(CloudBlob blob) {
            this.blob = blob;
        }

        public String readLine() throws IOException {
            if (this.reader == null) {
                try {
                    if (!this.metadataFetched) {
                        this.blob.downloadAttributes();
                        this.metadataFetched = true;
                        Map<String, String> metadata = CaseInsensitiveKeysMapAccess.convert(this.blob.getMetadata());
                        if (metadata.containsKey("lastEntry")) {
                            this.firstLineReturned = true;
                            return metadata.get("lastEntry");
                        }
                    }
                    this.reader = new ReverseFileReader(this.blob);
                    if (this.firstLineReturned) {
                        while ("".equals(this.reader.readLine())) {
                        }
                    }
                }
                catch (StorageException e) {
                    throw new IOException(e);
                }
            }
            return this.reader.readLine();
        }

        public void close() throws IOException {
        }
    }
}

